/******************************************************************************/
/*  RGB-LED Demo with DMX Interface using MSP430F2232 from Texas Instruments  */
/*                                                                            */
/*  Description: This program provides a receiver for a three channel dimmer  */
/*  signal send on a DMX512 bus. It receives DMX data via USCIA0 and          */
/*  evaluates the channels with a state machine. A DIP switch on Port 1       */
/*  selects the address of the RGB channels in the DMX frame.                 */
/*  Timer_A3 and Timer_B3 generate three PWM signals out of the received      */
/*  bytes for controlling each LED.                                           */
/*----------------------------------------------------------------------------*/
/*  Texas Instruments Deutschland GmbH                                        */
/*  Walter Nitzold, August 2008                                               */
/******************************************************************************/
#include "msp430x22x2.h"


// States of the Statemachine
#define    IDLE        0
#define    STARTB      1
#define    STARTADDR   2



// ******* Array Definitions for every LED  ***************
// used for implementing a special
// characteristic for every LED because LEDs have mostly an logarithmic 
// dependency of PWM versus light intensity

const unsigned int LED_Red_Tab[]={ 65535, 65424, 65368, 64512, 64256, 64000,
63744, 63488, 63232, 62976, 62720, 62464, 62208, 61952, 61696, 61440, 61184,
60928, 60672, 60416, 60160, 59904, 59648, 59392, 59136, 58880, 58624, 58368,
58112, 57856, 57600, 57344, 57088, 56832, 56576, 56320, 56064, 55808, 55552,
55296, 55040, 54784, 54528, 54272, 54016, 53760, 53504, 53248, 52992, 52736,
52480, 52224, 51968, 51712, 51456, 51200, 50944, 50688, 50432, 50176, 49920,
49664, 49408, 49152, 48896, 48640, 48384, 48128, 47872, 47616, 47360, 47104,
46848, 46592, 46336, 46080, 45824, 45568, 45312, 45056, 44800, 44544, 44288,
44032, 43776, 43520, 43264, 43008, 42752, 42496, 42240, 41984, 41728, 41472,
41216, 40960, 40704, 40448, 40192, 39936, 39680, 39424, 39168, 38912, 38656,
38400, 38144, 37888, 37632, 37376, 37120, 36864, 36608, 36352, 36096, 35840,
35584, 35328, 35072, 34816, 34560, 34304, 34048, 33792, 33536, 33280, 33024,
32768, 32512, 32256, 32000, 31744, 31488, 31232, 30976, 30720, 30464, 30208,
29952, 29696, 29440, 29184, 28928, 28672, 28416, 28160, 27904, 27648, 27392,
27136, 26880, 26624, 26368, 26112, 25856, 25600, 25344, 25088, 24832, 24576,
24320, 24064, 23808, 23552, 23296, 23040, 22784, 22528, 22272, 22016, 21760,
21504, 21248, 20992, 20736, 20480, 20224, 19968, 19712, 19456, 19200, 18944,
18688, 18432, 18176, 17920, 17664, 17408, 17152, 16896, 16640, 16384, 16128,
15872, 15616, 15360, 15104, 14848, 14592, 14336, 14080, 13824, 13568, 13312,
13056, 12800, 12544, 12288, 12032, 11776, 11520, 11264, 11008, 10752, 10496,
10240,  9984,  9728,  9472,  9216,  8960,  8704,  8448,  8192,  7936,  7680,
 7424,  7168,  6912,  6656,  6400,  6144,  5888,  5632,  5376,  5120,  4864,
 4608,  4352,  4096,  3840,  3584,  3328,  3072,  2816,  2560,  2304,  2048,
 1792,  1536,  1280,  1024,   768,   512,   256, 80}; //red LED
const unsigned int LED_Blue_Tab[]={ 65535, 65424, 65368, 64512, 64256, 64000,
63744, 63488, 63232, 62976, 62720, 62464, 62208, 61952, 61696, 61440, 61184,
60928, 60672, 60416, 60160, 59904, 59648, 59392, 59136, 58880, 58624, 58368,
58112, 57856, 57600, 57344, 57088, 56832, 56576, 56320, 56064, 55808, 55552,
55296, 55040, 54784, 54528, 54272, 54016, 53760, 53504, 53248, 52992, 52736,
52480, 52224, 51968, 51712, 51456, 51200, 50944, 50688, 50432, 50176, 49920,
49664, 49408, 49152, 48896, 48640, 48384, 48128, 47872, 47616, 47360, 47104,
46848, 46592, 46336, 46080, 45824, 45568, 45312, 45056, 44800, 44544, 44288,
44032, 43776, 43520, 43264, 43008, 42752, 42496, 42240, 41984, 41728, 41472,
41216, 40960, 40704, 40448, 40192, 39936, 39680, 39424, 39168, 38912, 38656,
38400, 38144, 37888, 37632, 37376, 37120, 36864, 36608, 36352, 36096, 35840,
35584, 35328, 35072, 34816, 34560, 34304, 34048, 33792, 33536, 33280, 33024,
32768, 32512, 32256, 32000, 31744, 31488, 31232, 30976, 30720, 30464, 30208,
29952, 29696, 29440, 29184, 28928, 28672, 28416, 28160, 27904, 27648, 27392,
27136, 26880, 26624, 26368, 26112, 25856, 25600, 25344, 25088, 24832, 24576,
24320, 24064, 23808, 23552, 23296, 23040, 22784, 22528, 22272, 22016, 21760,
21504, 21248, 20992, 20736, 20480, 20224, 19968, 19712, 19456, 19200, 18944,
18688, 18432, 18176, 17920, 17664, 17408, 17152, 16896, 16640, 16384, 16128,
15872, 15616, 15360, 15104, 14848, 14592, 14336, 14080, 13824, 13568, 13312,
13056, 12800, 12544, 12288, 12032, 11776, 11520, 11264, 11008, 10752, 10496,
10240,  9984,  9728,  9472,  9216,  8960,  8704,  8448,  8192,  7936,  7680,
 7424,  7168,  6912,  6656,  6400,  6144,  5888,  5632,  5376,  5120,  4864,
 4608,  4352,  4096,  3840,  3584,  3328,  3072,  2816,  2560,  2304,  2048,
 1792,  1536,  1280,  1024,   768,   512,   256, 80}; //blue LED
const unsigned int LED_Green_Tab[]={ 65535, 65424, 65368, 64512, 64256, 64000,
63744, 63488, 63232, 62976, 62720, 62464, 62208, 61952, 61696, 61440, 61184,
60928, 60672, 60416, 60160, 59904, 59648, 59392, 59136, 58880, 58624, 58368,
58112, 57856, 57600, 57344, 57088, 56832, 56576, 56320, 56064, 55808, 55552,
55296, 55040, 54784, 54528, 54272, 54016, 53760, 53504, 53248, 52992, 52736,
52480, 52224, 51968, 51712, 51456, 51200, 50944, 50688, 50432, 50176, 49920,
49664, 49408, 49152, 48896, 48640, 48384, 48128, 47872, 47616, 47360, 47104,
46848, 46592, 46336, 46080, 45824, 45568, 45312, 45056, 44800, 44544, 44288,
44032, 43776, 43520, 43264, 43008, 42752, 42496, 42240, 41984, 41728, 41472,
41216, 40960, 40704, 40448, 40192, 39936, 39680, 39424, 39168, 38912, 38656,
38400, 38144, 37888, 37632, 37376, 37120, 36864, 36608, 36352, 36096, 35840,
35584, 35328, 35072, 34816, 34560, 34304, 34048, 33792, 33536, 33280, 33024,
32768, 32512, 32256, 32000, 31744, 31488, 31232, 30976, 30720, 30464, 30208,
29952, 29696, 29440, 29184, 28928, 28672, 28416, 28160, 27904, 27648, 27392,
27136, 26880, 26624, 26368, 26112, 25856, 25600, 25344, 25088, 24832, 24576,
24320, 24064, 23808, 23552, 23296, 23040, 22784, 22528, 22272, 22016, 21760,
21504, 21248, 20992, 20736, 20480, 20224, 19968, 19712, 19456, 19200, 18944,
18688, 18432, 18176, 17920, 17664, 17408, 17152, 16896, 16640, 16384, 16128,
15872, 15616, 15360, 15104, 14848, 14592, 14336, 14080, 13824, 13568, 13312,
13056, 12800, 12544, 12288, 12032, 11776, 11520, 11264, 11008, 10752, 10496,
10240,  9984,  9728,  9472,  9216,  8960,  8704,  8448,  8192,  7936,  7680,
 7424,  7168,  6912,  6656,  6400,  6144,  5888,  5632,  5376,  5120,  4864,
 4608,  4352,  4096,  3840,  3584,  3328,  3072,  2816,  2560,  2304,  2048,
 1792,  1536,  1280,  1024,   768,   512,   256, 80}; //green LED


unsigned int  DmxState  = IDLE;         // Statemachine Variable
unsigned char DmxStat   = 0;            // USCI Statusregister
unsigned char DmxByte   = 0;            // Received Byte
unsigned int  Dmx_Channel_Count = 0;    // Counter Variable for Address
unsigned char dmx_rgb[] = {0,0,0};      // received RGB Channels
unsigned int  Dmx_field_count = 0;      // Counter Variable inside the DmxField
unsigned char DMX_val_TACCR1;           // Temporary storage variable for TACCR1

// ******************************
// Function Prototypes
void Timer_Init(void);
void Port_Init(void);
void DMX_Init(void);
unsigned int get_address( void );
// ******************************



//______________________________________________________________________________
//
// Main Program
//______________________________________________________________________________
//
void main(void)
{  
    WDTCTL=WDTPW+WDTHOLD;       // disable Watchdog

    BCSCTL1= CALBC1_8MHZ;       //--- System Clock Settings --------------------
    DCOCTL = CALDCO_8MHZ;       // use calibrated 8MHz settings
    
    Timer_Init();               //---- PWM Timer Initialization ----------------
    Port_Init();                //--- Port Initialization ----------------------
    
    TACTL |= MC_2;              // start Timer_A (continuous mode)
    TBCTL |= MC_2;              // start Timer_B (continuous mode) -  
                                //       there is a delay compared to Timer_A!!!
    
    DMX_Init();                 //---- DMX interface Initialization ------------
    
  
//--- Main Loop with Statemachine Handling -------------------------------------
    while(1)
    { 
        __low_power_mode_0();                // Sleep until charcter is received
      
        if(DmxStat & UCBRK)                  // Check if break error occured
        {
            if(DmxByte == 0x00)              // Check if first byte is StartCode
            {
                Dmx_Channel_Count = get_address();
                                             // Get Address of RGB Board
                DmxState = STARTB;           // SET State to "STARTB"
            }else
            {
                DmxState = IDLE;             // Error -> Go to Idle mode
            }
        }else
        {
            if(DmxState == STARTB)
            {
                Dmx_Channel_Count--;         // Decrement DMXCount for 
                                             //                  Startaddress
                if(Dmx_Channel_Count == 0)   // Check if RGB address reached
                {
                    Dmx_field_count = 0;
                    DmxState = STARTADDR;    // Set state to "STARTADDR"
                    dmx_rgb[Dmx_field_count] = DmxByte;
                                             // Store first channel of RGB
                }
            }else
            {
                if(DmxState == STARTADDR)    // Check if in State "STARTADDR" 
                {
                    Dmx_field_count++;       // increment array variable
                    dmx_rgb[Dmx_field_count] = DmxByte;
                                             // Store second and third 
                                             //                 channel of RGB
                    if(Dmx_field_count == 2) // If everything is received
                    {                        // set new PWM
                        DMX_val_TACCR1 = dmx_rgb[2];
                                             // store channel for updating 
                                             // TACCR1 (blue) synchronous in ISR
                        TBCCR1=LED_Green_Tab[dmx_rgb[1]];    
                                             // update TBCCR1 directly (green)
                        TBCCR2=LED_Red_Tab[dmx_rgb[0]];           
                                             // update TBCCR2 directly (red)
                        TACTL &= ~TAIFG;     // Clear IFG to avoid 
                                             //             unintentional ISR
                        TACTL |= TAIE;       // Enable Timer Overflow Interrupt
                                             // for updating TACCR1 synchronous
                        DmxState = IDLE;     // When all is set -> go idle
                    }
                }
            }
        }
    }
}


//******************************************************************************
// void Timer_Init(void);
// Initialize the two timers in MSP430F2274 (Timer_A3 and Timer_B3) for 
// generating the PWM signal for the RGB LED board
// blue  -> Timer_A3 (TACCR1)
// green -> Timer_B3 (TBCCR1)
// red   -> Timer_B3 (TBCCR2)
//******************************************************************************
void Timer_Init(void)
{
  
    //---- Timer_B3:  used for 1st and 2nd PWM channel  
    TBCTL = TBCLGRP_0+CNTL_0+TBSSEL_2+ID_0+MC_0+TBCLR;
                                     // Timer clock = SMCLK = 8MHz/1
    TBCCTL0 = CM_0+CCIS_2+CLLD_1+OUTMOD_0; 
                                     // TBCCR1 and TBCCR2 are loaded into TBCL1 
                                     // and TBCL2 registers when TBR counts to 0
    TBCCTL1 = CM_0+CCIS_2+CLLD_1+OUTMOD_7; // TBCCR1 defines duty cycle of 1st  
                                           //    PWM signal
    TBCCTL2 = CM_0+CCIS_2+CLLD_1+OUTMOD_7; // TBCCR2 defines duty cycle of 1st  
                                         //    PWM signal
    TBCCR0=0;               // TBCCR0 defines when TB1/TB2 output is cleared 
                            //   (after each overflow)
    TBCCR1=32768;           // TBCCR1 defines duty cycle of TB1 output
    TBCCR2=32768;           // TBCCR2 defines duty cycle of TB2 output

  
    //---- Timer_A3:  used for 3rd PWM channel  
    TACTL = TASSEL_2+ID_0+MC_0+TACLR;  // Timer clock = SMCLK = 8MHz/1
  
    TACCTL0 = CM_0+CCIS_2+OUTMOD_0; // TACCR0 block used for Output Unit "Reset" 
    TACCTL1 = CM_0+CCIS_2+OUTMOD_7; // TACCR1 defines duty cycle of 3rd PWM signal
    TACCTL2 = CM_0+CCIS_2+OUTMOD_0; // TACCR2 is not used
  
    TACCR0=0;               // TACCR0 defines when TA1 output is cleared (after 
                            //   each overflow)
    TACCR1=32768;           // TACCR1 defines duty cycle of TA1 output
  
}



//******************************************************************************
// void Port_Init(void);
// Initialize all of the port pins for usage as outputs for PWM and inputs for
// DMX bus signal and DIP address decoder
//******************************************************************************
void Port_Init(void)
{
    // Termination of unused digital I/Os
    P2SEL &= ~(BIT7 + BIT6);    // set back (XOUT/XIN) to port function
    P2DIR = 0xFF;               // set as output
    P3DIR = 0xFF;               // set as output
    P4DIR = 0xFF;               // set as output
  
    // Timer_A and Timer_B pin configuration
    P4DIR |= BIT4+BIT5; // P4.4/TB1 and P4.5/TB2 are used in module function
    P4SEL |= BIT4+BIT5;
    P2DIR |= BIT3;      // P2.3/TA1 is used in module function
    P2SEL |= BIT3;

    // DIP Switch Input / Address
    P1SEL = 0x00;       //             Port 1 for DIP switch
    P1DIR = 0x00;       //             as decoder of address
    P1OUT = 0x00;       //             of th RGB board in DMX frame
    P1REN = 0xFF;       // use internal pull down resistors
  
 
    

  
}

//******************************************************************************
// void DMX_Init(void);
// Initialize all of the port pins for usage as outputs for PWM and inputs for
// DMX bus signal and DIP address decoder
//******************************************************************************
void DMX_Init(void)
{
    UCA0CTL1 |= UCSWRST;                      // **Reset USCI state machine**
    
    // USCI pin configuration
    P3DIR &= ~BIT5;
    P3SEL |= BIT5;      // P3.5/UCA0RXD is used in module function (DMX input)
    
    UCA0CTL0 |= UCSPB;                        // Two Stop Bits  
    UCA0CTL1 |= UCSSEL_2 + UCRXEIE;           // SMCLK; receive erroneous bytes
    UCA0BR0 = 32;                             // 8MHz/250000 = 32
    UCA0BR1 = 0;                              // 8MHz/250000 = 32
    UCA0MCTL = 0;                             // no Modulation
    UCA0CTL1 &= ~UCSWRST;                     // *Initialize USCI state machine*
    IE2 |= UCA0RXIE;                          // Enable USCI_A0 RX interrupt
  
}


//******************************************************************************
// unsigned int get_address( void );
// function to give back the address that was set with the DIP switch connected
// to Port 1 (only Channel 1 - 255) -> for expansion of addresses just add code
// for another bit and return the evaluated value
//******************************************************************************
unsigned int get_address( void )
{
    return P1IN;
}


//______________________________________________________________________________
//
// Interrupt Service routines
//______________________________________________________________________________
//


//
//******************************************************************************
// Timer_A Interrupt Service Routine
// Executed when Timer_A3 has an overflow; Used for updating the TACCR1 register
// synchronous to avoid flickering of the blue LED
//******************************************************************************
#pragma vector=TIMERA1_VECTOR
__interrupt void ISR_TimerA(void)
{ 
    TACCR1=LED_Blue_Tab[DMX_val_TACCR1];    // update PWM duty cycle settings
    TACTL &= ~(TAIFG+TAIE);                 // PWM signal generation
}



//
//******************************************************************************
// USCIA0 Interrupt Service Routine
// Executed when a byte is received and copied to UCA0RXBUF; 
// Used to store status of USCI module and received byte and to wake up from LPM
//******************************************************************************
#pragma vector=USCIAB0RX_VECTOR
__interrupt void USCI0RX_ISR(void)
{
    DmxStat = UCA0STAT;                // Get status of USCI
    DmxByte = UCA0RXBUF;               // Get received Byte
    IFG2 &= ~UCA0RXIFG;                // Clear Interrupt Flag
    __low_power_mode_off_on_exit();    // Exit LPM0 when received a byte
}
