/*===========================================================================*
 | Implementation file scI2C_PIC18.c
 |   implements : 
 |
 |      summary :  Methods to use a I2C communication of a MCU PIC18
 |
 | Creator : Sylvie Rohrer
 | Creation date : 30.11.2006
 | Copyright: HE-ARC, SWISSCUBE all rights reserved
 |
 | Modification tag : //MODIF1
 |
 | Version of the file : 1
 *===========================================================================*/

#include <pic18.h> 
#include "scI2C_PIC18.h"

#define R_W_MASK     0x04 	/* 00000100b == R/W = 1  */
#define SSPIF_MASK   0x08	/* 00001000b == SSPIF = 1 */
#define RC3RC4_MASK	 0x18	/* 00011000b == RC3 and RC4 = 1 */
#define ADRSLAVE  	 0x48	/* 1001000b */


/**
* i2c_init
* Initialize the register for the communication I2C
*
* @param char _cMaster : 1 = I2C Master Mode
*						 0 = I2C Slave Mode
*/
void i2c_init(char _cMaster)
/*************************************************************************/
{ 
	/* SSPCON1 REGISTER */ 
	if (_cMaster == 1)		// I2C Master Mode
	{
		SSPCON1 = 0x28;		// SSPEN       =    1 (enables serial port)
							// SSPM3:SSPM0 = 1000
	}

	else
	{
		SSPCON1 = 0x3E;		// SSPEN       =    1
							// CKP         =    1
							// SSPM3:SSPM0 = 1110 (7 bits interrupt)
	}

	/* SSPCON2 REGISTER */ 
	SSPCON2     = 0x00; 

	/* SSPSTAT REGISTER */ 
	SMP    		= 0;   		// Slew Rate Controle Enabled (400kHz)
	CKE			= 0;   		// SMBus Specific Inputs Disabled
	
	/* SSPADD REGISTER */
	if (_cMaster)
		SSPADD  = 0x0A;		// Set i2c speed
	else
		SSPADD	= ADRSLAVE;	// Affect an valid address of the slave mcu

	/* TRISC REGISTER */
	if (!_cMaster)
	{
		TRISC = TRISC | RC3RC4_MASK;
		SSPIF = 0;
		SSPCON2 = 1;
	}
}


/**
* i2c_start
* Start the communication I2C
*/
void i2c_start(void)
/*************************************************************************/
{ 
   SEN = 1;         // Generate Start Condition 
   i2c_wait(); 		// The MSSP module will wait the
					// required start time before any other
					// operation takes place.
} 


/**
* i2c_restart
* Restart the communication I2C
*/
void i2c_restart(void)
/*************************************************************************/
{ 
   RSEN = 1;        // Generate Start Condition 
   i2c_wait(); 		// The MSSP module will wait the
					// required start time before any other
					// operation takes place.
}


/**
* i2c_write
* Write a char in the i2c communication
*
* @param char _cData : Address (of the command or of the slave, etc.)
*/
void i2c_write(char _cData)
/*************************************************************************/
{ 
   SSPBUF = _cData;	// Load DATA to send 
   i2c_wait();		// Wait for completion 
}


/**
* i2c_read
* Read a char in the i2c communication
*
* @return : Data received
*/
char i2c_read(void)
/*************************************************************************/
{
	char l_cData = 0;

	while(SSPSTAT & R_W_MASK);  // Wait the end of transmission
  	RCEN = 1;					// Enable I2C receiver 

	i2c_wait();
	l_cData = SSPBUF;			// Calcule the temperature

	return l_cData;
}


/**
* i2c_stop
* Stop the communication I2C
*/
void i2c_stop(void)
/*************************************************************************/
{
   PEN = 1;            // Generate Stop Condition 
   i2c_wait(); 
} 


/**
* i2c_wait
* Wait that the interrupt is generated
*/
void i2c_wait(void)
/*************************************************************************/
{
	while (!(PIR1 & SSPIF_MASK)); 	// While (SSPIF == 0)
									// Wait for interrupt
	SSPIF = 0;						// Clear interrupt flag
}


/**
* i2c_ack
* Send an acknowledgement
*/
void i2c_ack(void)
/*************************************************************************/
{
	ACKDT = 0;		// Setup for ACK
	ACKEN = 1;		// Send ACK
}


/**
* i2c_nack
* Send an non acknowledgement
*/
void i2c_nack(void)
/*************************************************************************/
{
	ACKDT = 1;		// Setup for NACK
	ACKEN = 0;		// Send NACK
}


/**
* i2c_initSerialPortInterrupt
* Set the Master Synchronous Serial Port Interrupt Flag bit
*   to the status "Waiting to transmit/receive"
*   See : PIR1 (Peripheral Interrupt Request)
*/
void i2c_initSerialPortInterrupt(void)
/*************************************************************************/
{
	SSPIF = 0;
}


/**
* delayMs
* Wait a moment
*
* @param char _pause : Pause duration
*/
void delayMs(char _cPause)
/*************************************************************************/
{ 
   	char l_cI;
	char l_cJ;
	char l_cK;    	 
  	
	char l_cClock = 8; 
  	char l_cCycle = 14;
    
   	for(l_cI=0; l_cI<=_cPause; l_cI++)
   		for(l_cJ=0; l_cJ<=l_cClock; l_cJ++)
   			for(l_cK=0; l_cK<=l_cCycle; l_cK++) 
      			asm("nop"); 	// asm instruction, wait 1 cycle of clock
}



/**
* i2c_write_slave
* Write a char in the i2c communication (in slave mode)
*
* @param char _cData : Data to send to the master
*/
void i2c_write_slave(char _cData)
/*************************************************************************/
{ 
	i2c_wait();			// While (SSPIF == 0) SSPIF = 0
	SSPBUF = _cData;	// Load the byte to transfert
	CKP = 1;			// Release clock (unlock SCL)
	while(SSPIF==0);	// Wait a ready status
}


/**
* i2c_read_slave
* Read a char from the i2c communication (in slave mode)
*
* @return : Received data
*/
char i2c_read_slave(void)
/*************************************************************************/
{
	char l_cData = 0;

	i2c_wait();			// While (SSPIF == 0) SSPIF = 0
	l_cData = SSPBUF;

	return l_cData;		// Read the received character
}


/**
* i2c_wait_address_slave
* Read a char from the i2c communication (in slave mode)
*
* @return : SSPSTAT bit R/W
*/
char i2c_wait_address_slave(void)
/*************************************************************************/
{
	char l_cTmp;

	i2c_wait();			// While (SSPIF == 0) SSPIF = 0
	l_cTmp = SSPBUF;

	return l_cTmp;		// Read the received character
}
