/*
    ATtiny85 Solar Tracker 1.0 by Adam Humphreys
    using Arduino 1.0 IDE for the ATtiny25/45/85 @ 1MHz internal clock.
    This work is licensed under a Creative Commons
    Attribution-NonCommercial-ShareAlike 3.0 Unported License (CC BY-NC-SA 3.0).
    http://adamlhumphreys.com/
    http://creativecommons.org/licenses/by-nc-sa/3.0/
*/

#include <avr/sleep.h>
#include "elapsedMillis.h"

//Uncomment "#define DEBUG" to debug using Teensy 2.0 or Arduino with Serial communications:
//#define DEBUG

void setup()
{
  #if defined(DEBUG)
    Serial.begin(38400);
  #endif
  DDRB |= 1<<0;  //Set pin 0 on PORT B as an output pin (Pin 5 on ATtiny85, Digital Pin 8 on Arduino)
  DDRB |= 1<<1;  //Set pin 1 on PORT B as an output pin (Pin 6 on ATtiny85, Digital Pin 9 on Arduino)
}

#define MOVE_LEFT_ON (PORTB |= 1<<0)       //Turn on pin 0 on PORT B  (Pin 5 on ATtiny85, Digital Pin 8 on Arduino)
#define MOVE_RIGHT_ON (PORTB |= 1<<1)      //Turn on pin 1 on PORT B  (Pin 6 on ATtiny85, Digital Pin 9 on Arduino)
#define MOVE_LEFT_OFF (PORTB &= ~(1<<0))   //Turn off pin 0 on PORT B (Pin 5 on ATtiny85, Digital Pin 8 on Arduino)
#define MOVE_RIGHT_OFF (PORTB &= ~(1<<1))  //Turn off pin 1 on PORT B (Pin 6 on ATtiny85, Digital Pin 9 on Arduino)

const int lightPoint = 800;      // Lower this number if you want the tracker to continue tracking
                                 // under darker light before reverting back to its default position.
const int toleranceMargin = 10;  // Increase this number to allow for more play and less precision
                                 // if the tracker seems to bounce back and forth.

bool moveLeftHalted = false;
bool moveRightHalted = false;
elapsedMillis darknessDuration;

void updatePosition();
int checkHaltSwitches();
void idle();

void loop()
{
  while(analogRead(3) >= lightPoint || darknessDuration < (6*60*60*1000)) //While it's light outside or has been dark for less than 6 hours.
  {
    if (analogRead(3) >= lightPoint) //If it's light outside
    {
      updatePosition();
      darknessDuration = 0;
    }
    delay(15*60*1000); //Wait 15 minutes before checking again.
    #if defined(DEBUG)
      Serial.print("darknessDuration: ");
      Serial.println(darknessDuration/1000);
    #endif
  }
  
  //Go to default left position.
  if (!moveLeftHalted)
    MOVE_LEFT_ON;
  #if defined(DEBUG)
    Serial.println("Moving back to default left position.");
  #endif
  while (checkHaltSwitches() != 1); //While left is not halted, keep checking.
  
  //Go to sleep.
  #if defined(DEBUG)
    Serial.println("Sleping...");
  #endif
  while (analogRead(3) < lightPoint)
  {
    idle();
    delay(100);
  }
  
  #if defined(DEBUG)
    Serial.println("Wakeup!");
  #endif
}

void updatePosition()
{
  int rightSide, leftSide, noChangeCount = 0;
  while (noChangeCount < 5)
  {
    checkHaltSwitches();
    rightSide = analogRead(3);
    leftSide = analogRead(2);
    
    if (leftSide + toleranceMargin < rightSide && !moveRightHalted)
    {
      MOVE_LEFT_OFF;
      delay(50);
      MOVE_RIGHT_ON;
      #if defined(DEBUG)
        Serial.print("Move Right: ");
      #endif
    }
    else if (leftSide > rightSide + toleranceMargin && !moveLeftHalted)
    {
      MOVE_RIGHT_OFF;
      delay(50);
      MOVE_LEFT_ON;
      #if defined(DEBUG)
        Serial.print("Move Left:  ");
      #endif
    }
    else
    {
      MOVE_LEFT_OFF;
      MOVE_RIGHT_OFF;
      #if defined(DEBUG)
        Serial.print("-Centered-: ");
      #endif
      noChangeCount++;
      delay(1000);
    }
    #if defined(DEBUG)
      Serial.print("leftSide: ");
      Serial.print(leftSide);
      Serial.print(", rightSide: ");
      Serial.println(rightSide);
    #endif
    delay(100);
  }
  MOVE_LEFT_OFF;
  MOVE_RIGHT_OFF;
  #if defined(DEBUG)
    Serial.println("Position updated.");
  #endif
  
  return;
}

int checkHaltSwitches()
{
  int haltSwitches;
  int results[] = {-1, -2};
  
  while (results[0] != results[1])
  {
    haltSwitches = analogRead(1);
    if (haltSwitches >= 256 && haltSwitches < 666) //If signal is between 1.25 and 3.33 volts
      results[0] = 1;
    else if (haltSwitches < 256) //If signal is less than 1.25 volts (should be near 0, or ground)
      results[0] = 2;
    else
      results[0] = 0;

    delay(50);
    
    haltSwitches = analogRead(1);
    if (haltSwitches >= 256 && haltSwitches < 666)
      results[1] = 1;
    else if (haltSwitches < 256)
      results[1] = 2;
    else
      results[1] = 0;
  }
  
  if (results[0] == 0)
  {
    moveLeftHalted = false;
    moveRightHalted = false;
  }
  else if (results[0] == 1)
  {
    MOVE_LEFT_OFF;
    moveLeftHalted = true;
    #if defined(DEBUG)
      Serial.println("Move Left Halted");
    #endif
  }
  else //if (results[0] == 2)
  {
    MOVE_RIGHT_OFF;
    moveRightHalted = true;
    #if defined(DEBUG)
      Serial.println("Move Right Halted");
    #endif
  }
  
  return results[0];
}

void idle()
{
  set_sleep_mode(SLEEP_MODE_IDLE);
  noInterrupts();
  sleep_enable();
  interrupts();
  sleep_cpu();
  sleep_disable();
}
