#include "I2C_18fmaster.h"

//******** PIC18 Configuration Bits required *********//
#pragma config LVP = OFF
#pragma config FOSC = HS
#pragma config WDT = OFF
#pragma config BOR = ON

//#define	I2C_FREQ	0x10	// Clock divider ( Fcpu / (2*(X+2))	avce X = 16 ) ~ 80KHz  
//#define	I2C_FREQ    0x20	// Clock divider ( Fcpu / (2*(X+2))	avce X = 32 ) ~ 40KHz
//#define	I2C_FREQ	0x30	// Clock divider ( Fcpu / (2*(X+2))	avce X = 48 ) ~ 30KHz
#define	I2C_FREQ	0x40	// Clock divider ( Fcpu / (2*(X+2))	avce X = 64 ) ~ 23KHz
#define I2C_PERIOD 100    		/* 1000 avant */

/*************************************************************************************
 * ROUTINE NAME : 	I2Cm_Init
 * INPUT/OUTPUT : 	NONE
 * DESCRIPTION : 		Initialize I2C registers 
 * COMMENTS : 		Select sending data frequency and enable I2C peripheral 	
 *************************************************************************************/
void I2Cm_Init(void)
{
  SSPSTAT &= 0x3F;                // power on state 
  SSPCON1 = 0x00;                 // power on state
  SSPCON2 = 0x00;                 // power on state
  SSPCON1 |= MASTER;           	  // Master Mode selected
  SSPSTAT |= SLEW_OFF;            // slew rate off
  ADCON1=6; 
  DDRBbits.RB1 = 1;               // Set SCL (PORTB,1) pin to input
  DDRBbits.RB0 = 1;               // Set SDA (PORTB,0) pin to input
  SSPCON1 |= SSPENB;              // enable synchronous serial port
  SSPADD = I2C_FREQ; 
}

/*************************************************************************************
 * ROUTINE NAME : 	I2Cm_Start
 * DESCRIPTION : 	Transmit start condition.
 * COMMENTS : 		Wait for the SB flag generation  	
 *************************************************************************************/
void I2Cm_Start(void)
{
  SSPCON2bits.SEN = 1;            // initiate bus start condition
}

/*************************************************************************************
 * ROUTINE NAME : 	I2Cm_Stop
 * DESCRIPTION : 	Transmit stop condition.
 *************************************************************************************/
void I2Cm_Stop(void)
{
  SSPCON2bits.PEN = 1;            // initiate bus stop condition
}

/*************************************************************************************
 * ROUTINE NAME : 	I2Cm_Idle
 * DESCRIPTION : 	Wait for the I2C bus to be idle.	
 ************************************************************************************/
void I2Cm_Idle(void)
{
  while( (SSPCON2 & 0x1F) | (SSPSTATbits.R_W) )
     continue;
}

/*************************************************************************************
 * ROUTINE NAME : 	I2Cm_DataRdy	
 * DESCRIPTION : 	Check if a byte is present on I2C bus.
 *************************************************************************************/
uint8_t I2Cm_DataRdy(void)
{
	if(SSPSTATbits.BF == 1)           // test if buffer full bit is set     
    {
		return (+1);                // data in SSPBUF register
	}
	else
	{
		return (0);                 // no data in SSPBUF register
	}
}

/********************************************************************
*     ROUTINE NAME :  I2Cm_Ack		                                                   			                 
*     DESCRIPTION :      Initiate ACK bus condition.        			
********************************************************************/
void I2Cm_Ack( void )
{
  SSPCON2bits.ACKDT = 0;           // set acknowledge bit state for ACK
  SSPCON2bits.ACKEN = 1;           // initiate bus acknowledge sequence
}

/********************************************************************
*     Function Name:   I2Cm_NotAck                                                                     
*     Description:      Initiate NOT ACK bus condition.             
********************************************************************/
void I2Cm_NotAck ( void )
{
  SSPCON2bits.ACKDT = 1;          // set acknowledge bit for not ACK
  SSPCON2bits.ACKEN = 1;          // initiate bus acknowledge sequence
}

/*************************************************************************************
 * ROUTINE NAME : 		I2Cm_WriteByte
 * RETURN :			0 if problem, 1 if ok
 * INPUT/OUTPUT : 		 Byte to be transmitted.
 * DESCRIPTION : 		Transmit byte to extern device.	
 *************************************************************************************/
uint8_t I2Cm_WriteByte(uint8_t byte)
{
	SSPBUF = byte;           // write single byte to SSPBUF
	if(SSPCON1bits.WCOL == 1)      // test if write collision occurred
	{
		return ( -1 );	// if WCOL bit is set return negative #
	}              
	else
	{
		while(SSPSTATbits.BF == 1);   // wait until write cycle is complete         
		return (0);              	  // if WCOL bit is not set return non-negative #
	}
}

/********************************************************************
*     Function Name:    I2Cm_WriteString                            
*     Return Value:     error condition status                      
*     Parameters:       address of write string storage location   
*     Description:      This routine writes a string to the I2C bus,
*                              until a null character is reached. 
********************************************************************/
int8_t I2Cm_WriteString (uint8_t *buffer)
{
	while (*buffer)                // transmit data until null character 
	{
		if ( SSPCON1bits.SSPM3 )      // if Master transmitter then execute the following
		{
			if ( I2Cm_WriteByte( *buffer ) )   // write 1 byte
			{
				return ( -3 );            // return with write collision error
			}
			I2Cm_Idle();                  // test for idle condition
		
			if ( SSPCON2bits.ACKSTAT )  // test received ack bit state
			{
				return ( -2 );            // bus device responded with  NOT ACK
			}                           // terminate putsI2C() function
		}
		else                          // else Slave transmitter
		{
			PIR1bits.SSPIF = 0;         // reset SSPIF bit
			SSPBUF = *buffer;            // load SSPBUF with new data
			SSPCON1bits.CKP = 1;        // release clock line 
			while(!PIR1bits.SSPIF == 1);  // wait until ninth clock pulse received
    
			if ( ( !SSPSTATbits.R_W ) && ( !SSPSTATbits.BF ) )// if R/W=0 and BF=0, NOT ACK was received
			{
				return ( -2 );            // terminate PutsI2C() function
			}
		}
			buffer ++;                       // increment pointer
	}                               // continue data writes until null character
	return ( 0 );
}

/*************************************************************************************
 * ROUTINE NAME : 	I2Cm_ReadByte
 * DESCRIPTION : 		Read a single byte from extern EEPROM.
 * RETURN : 		the 	read  byte 
 *************************************************************************************/
uint8_t I2Cm_ReadByte(void)
{
  SSPCON2bits.RCEN = 1;           // enable master for 1 byte reception
  while(!SSPSTATbits.BF == 1);      // wait until byte received  
  return (SSPBUF);              // return with read byte 
}

/************************************************************************
*     Function Name:    I2Cm_ReadString                             			
*     Return Value:     error condition status                    			
*     Parameters:       address of read string storage location llength of string bytes to read          				
*     Description:      Reads a string from I2C bus	
********************************************************************/
uint8_t I2Cm_ReadString(uint8_t *buffer, uint8_t length)
{
    while (length --)           // perform I2Cm_ReadByte() for 'length' number of bytes
    {
		*buffer++ = I2Cm_ReadByte();       // save byte received
		while ( SSPCON2bits.RCEN ); // check that receive sequence is over    
		
		if (PIR2bits.BCLIF)       // test for bus collision
		{
			return (-1);            // return with Bus Collision error 
		}
		
		if (length)               // test if 'length' bytes have been read
		{
			SSPCON2bits.ACKDT = 0;    // set acknowledge bit state for ACK        
			SSPCON2bits.ACKEN = 1;    // initiate bus acknowledge sequence
			while ( SSPCON2bits.ACKEN ); // wait until ACK sequence is over 
		} 
    }
    return ( 0 );                 // last byte received so don't send ACK      
}

/*************************************************************************************
 * ROUTINE NAME : 		I2Cm_TransByte8
 * RETURN :			0 If Non-Ack from slave, 1 : Ok
 * INPUT/OUTPUT : 		(dev_addr) : Slave adress - (data) : byte to be transmitted
 * DESCRIPTION : 		Transmit byte to extern device.		
 *************************************************************************************/
error_mess I2Cm_TransByte8( uint8_t dev_addr, uint8_t data )
{
	I2Cm_Idle();  
	I2Cm_Start();                       
	I2Cm_Idle();	
	I2Cm_WriteByte(dev_addr & 0xFE);	 // Sending the address
	I2Cm_Idle();	
	if(SSPCON2bits.ACKSTAT == 1) // If the slave didn't receive the adress byte
   	{ 
        I2Cm_Stop();
	//	return (I2C_ACK_ERROR);		
	}
	else
	{
		I2Cm_WriteByte(data); 		 // Sending data
		if(SSPCON2bits.ACKSTAT == 1) // If the slave didn't receive the adress byte
		{ 
			I2Cm_Stop();
		//	return (I2C_ACK_ERROR);
		}
	}
	I2Cm_Idle();                 
	I2Cm_Stop();
//	return (I2C_RESULT_OK);
}

/*************************************************************************************
 * ROUTINE NAME : 		I2Cm_TransByte16
 * RETURN :			0 If Non-Ack from slave, 1 : Ok
 * INPUT/OUTPUT : 		dev_addr : Slave adress 
					mem_addr : Memory Address where to write
 					data : byte to be transmitted
 * DESCRIPTION : 		Transmit byte to extern device.		
 *************************************************************************************/
error_mess I2Cm_TransByte16( uint8_t dev_addr, uint8_t mem_addr, uint8_t data)
{
	I2Cm_Idle();  
	I2Cm_Start();                       
	I2Cm_Idle();	
	I2Cm_WriteByte(dev_addr & 0xFE);	 // Sending the address
	I2Cm_Idle();
	
	if(SSPCON2bits.ACKSTAT == 1) // If the slave didn't receive the adress byte
   	{ 
        I2Cm_Stop();
		return (I2C_ACK_ERROR);		
	}
	else
	{
		I2Cm_WriteByte(mem_addr); 		 // Sending Memory Address
		if(SSPCON2bits.ACKSTAT == 1) // If the slave didn't receive the adress byte
		{ 
			I2Cm_Stop();
			return (I2C_ACK_ERROR);
		}
		else
		{
			I2Cm_WriteByte(data);
			if(SSPCON2bits.ACKSTAT == 1) // If the slave didn't receive the adress byte
			{ 
				I2Cm_Stop();
				return (I2C_ACK_ERROR);
			}
		}	
	}
	I2Cm_Idle();                 
	I2Cm_Stop();
	return (I2C_RESULT_OK);
}


/*************************************************************************************
 * ROUTINE NAME : 	I2Cm_TransString8
 * RETURN :		Ack Error or Ok
 * INPUT/OUTPUT : 	dev_addr : Slave adress  
				data: byte to be transmitted
 * DESCRIPTION : 	Transmit byte to extern device.		
 *************************************************************************************/
error_mess I2Cm_TransString8(uint8_t dev_addr, uint8_t length, uint8_t* string)
{
	I2Cm_Idle();   
	I2Cm_Start();                       
	I2Cm_Idle();   
	I2Cm_WriteByte(dev_addr & 0xFE);
	I2Cm_Idle();	
	
	if (SSPCON2bits.ACKSTAT == 1) 
   	{ 
  		I2Cm_Stop();
		return (I2C_ACK_ERROR);      				
	}
	else
	{
		I2Cm_WriteByte(length);
		I2Cm_Idle();
		if (SSPCON2bits.ACKSTAT == 1) 
		{ 
			I2Cm_Stop();
			return (I2C_ACK_ERROR);      				
		}
		else
		{
			I2Cm_WriteString(string);
			I2Cm_Idle();
			if (SSPCON2bits.ACKSTAT == 1) 
			{ 
				I2Cm_Stop();
				return (I2C_ACK_ERROR);      				
			}		
		}
	}
	I2Cm_Idle();                 
	I2Cm_Stop();
	return (I2C_RESULT_OK);	
}

/*************************************************************************************
 * ROUTINE NAME : 	I2Cm_TransString16
 * RETURN :		Ack Error or Ok
  * INPUT/OUTPUT : 	dev_addr : Slave adress
				mem_addr : Memory Address where to write
				data: byte to be transmitted
 * DESCRIPTION : 	Transmit byte to extern device.		
 *************************************************************************************/
error_mess I2Cm_TransString16(uint8_t dev_addr, uint8_t mem_addr, uint8_t length, uint8_t* string)
{
	I2Cm_Idle();   
	I2Cm_Start();                       
	I2Cm_Idle();   
	I2Cm_WriteByte(dev_addr & 0xFE);
	I2Cm_Idle();	
	if (SSPCON2bits.ACKSTAT == 1) 
   	{ 
  		I2Cm_Stop();
		return (I2C_ACK_ERROR);      				
	}
	else
	{
		I2Cm_WriteByte(mem_addr);
		I2Cm_Idle();
		if (SSPCON2bits.ACKSTAT == 1)
		{
			I2Cm_Stop();
			return (I2C_ACK_ERROR);
		}
		else
		{		
			I2Cm_WriteByte(length);
			I2Cm_Idle();
			if (SSPCON2bits.ACKSTAT == 1)
			{
				I2Cm_Stop();
				return (I2C_ACK_ERROR);
			}
			else
			{					
				I2Cm_WriteString(string);
				I2Cm_Idle();			
				if (SSPCON2bits.ACKSTAT == 1) 
				{ 
					I2Cm_Stop();
					return (I2C_ACK_ERROR);      				
				}
			}
		}
	}
	I2Cm_Idle();                 
	I2Cm_Stop();
	return (I2C_RESULT_OK);	
}

/*************************************************************************************
 * ROUTINE NAME : 	    	I2Cm_ReceptByte8
 * INPUT/OUTPUT : 	    	Slave Adress,
 * DESCRIPTION : 		read a specified length of string in the I2C bus.
 * RETURN : 			the readed string 	
 *************************************************************************************/
uint8_t I2Cm_ReceptByte8(uint8_t dev_addr)
{
  I2Cm_Idle();       
  I2Cm_Start();                     
  I2Cm_Idle();       
  I2Cm_WriteByte(dev_addr | 0x01);  
  I2Cm_Idle();                 
	
    if (SSPCON2bits.ACKSTAT == 0) 			// If a Ack has been sent from slave
    {
	    SSPCON2bits.RCEN = 1;       		// enable master for 1 byte reception
        I2Cm_Idle();    
		I2Cm_NotAck();    // Send Not Ack : The master don't want more data        
        I2Cm_Idle();      // Waiting for Notack condition to be over
		I2Cm_Stop();      // Stop I2C Operation        
        I2Cm_Idle();      // Waiting for stop condition to be over  
    }
    else
    {
        return (I2C_ACK_ERROR);          // return with Not Ack error
    }
    return ((uint8_t) SSPBUF );     // return with data
}

/*************************************************************************************
 * ROUTINE NAME : 	    	I2Cm_ReceptByte16
 * INPUT/OUTPUT : 	    	Slave Adress and memory location where to read
 * DESCRIPTION : 		read a specified length of string in the I2C bus.
 * RETURN : 			the readed string 	
 *************************************************************************************/
uint8_t I2Cm_ReceptByte16(uint8_t dev_addr, uint8_t mem_addr)
{
	I2Cm_Idle();       
	I2Cm_Start();                     
	I2Cm_Idle();       
	I2Cm_WriteByte(dev_addr & 0xFE);  
	I2Cm_Idle();                 
	
	if (SSPCON2bits.ACKSTAT == 1)
	{
		I2Cm_Stop();
		return (I2C_ACK_ERROR); 
	}
	else
	{
		I2Cm_WriteByte(mem_addr);
		I2Cm_Idle();
		
		if (SSPCON2bits.ACKSTAT == 1)
		{
			I2Cm_Stop();
			return (I2C_ACK_ERROR); 
		}
		else
		{
			I2Cm_Restart();
			I2Cm_Idle();
			I2Cm_WriteByte(dev_addr | 0x01);
			I2Cm_Idle();
		    if (SSPCON2bits.ACKSTAT == 0) 			// If a Ack has been sent from slave
		    {
			    SSPCON2bits.RCEN = 1;       		// enable master for 1 byte reception
		        I2Cm_Idle();    
				I2Cm_NotAck();    // Send Not Ack : The master don't want more data        
		        I2Cm_Idle();      // Waiting for Notack condition to be over
				I2Cm_Stop();      // Stop I2C Operation        
		        I2Cm_Idle();      // Waiting for stop condition to be over  
		    }
		    else
		    {
		        I2Cm_Stop();
				return (I2C_ACK_ERROR);          // return with Not Ack error
		    }
		}
	}
    return ((uint8_t) SSPBUF );     // return with data
}

/******************************************************************
*     Function Name:    I2Cm_ReceptString8 			                    
*     Return Value:     error condition status                    		      
*     Parameters:       Slave Adress,  Buffer where to save data and length of data.                              			  
*     Description:     Reads data string from I2C device. 
********************************************************************/
error_mess  I2Cm_ReceptString8(uint8_t dev_addr, uint8_t *buffer, uint8_t length )
{
  I2Cm_Idle();   
  I2Cm_Start();                     
  I2Cm_Idle();   
  I2Cm_WriteByte(dev_addr | 0x01); // Write Slave Adress with last bit = 1 for receiving operation  
  I2Cm_Idle();   
 
   if (SSPCON2bits.ACKSTAT == 0) 		// Test if the Slave has received the adress. If yes :
   {
		I2Cm_Idle();	
		if (I2Cm_ReadString(buffer, length) == 1) // If an error has been detected
		{
			I2Cm_Stop();
			return (I2C_ACK_ERROR);        
		}
		I2Cm_Idle();	
		I2Cm_NotAck();
		I2Cm_Idle();	
		I2Cm_Stop();
		I2Cm_Idle();
	}
    else
    {
       return (I2C_ACK_ERROR);            // return with Not Ack error
    } 	  
    return (I2C_RESULT_OK);                // return with no error
}


/******************************************************************
*     Function Name:    I2Cm_ReceptString16 			                    
*     Return Value:     error condition status                    		      
*     Parameters:       Slave Adress,  Memory Adress where to read, Buffer where to save data and length of data.                              			  
*     Description:     Reads data string from I2C device. 
********************************************************************/
error_mess I2Cm_ReceptString16(uint8_t dev_addr, uint8_t mem_addr, uint8_t *buffer, uint8_t length )
{
	I2Cm_Idle();       
	I2Cm_Start();                     
	I2Cm_Idle();       
	I2Cm_WriteByte(dev_addr & 0xFE);  
	I2Cm_Idle();                 
	
	if (SSPCON2bits.ACKSTAT == 1)
	{
		I2Cm_Stop();
		return (I2C_ACK_ERROR); 
	}
	else
	{
		I2Cm_WriteByte(mem_addr);
		I2Cm_Idle();
		if (SSPCON2bits.ACKSTAT == 1)
		{
			I2Cm_Stop();
			return (I2C_ACK_ERROR); 
		}
		else
		{
			I2Cm_WriteByte(length);
			I2Cm_Idle();
			if (SSPCON2bits.ACKSTAT == 1)
			{
				I2Cm_Stop();
				return (I2C_ACK_ERROR); 
			}
			else
			{
				I2Cm_Restart(); 
				I2Cm_Idle();
				I2Cm_WriteByte(dev_addr | 0x01);
				I2Cm_Idle();
				if (SSPCON2bits.ACKSTAT == 1)
				{
					I2Cm_Stop();
					return (I2C_ACK_ERROR); 
				}
				else
				{
					if(I2Cm_ReadString(buffer, length) == 1) // If an error has been detected
					{
						I2Cm_Stop();
						return (I2C_ACK_ERROR);        
					}
					else
					{
						I2Cm_Idle();	
						I2Cm_NotAck();
						I2Cm_Idle();	
						I2Cm_Stop();
						I2Cm_Idle();
					}
				}
			}
		}
	}
    return (I2C_RESULT_OK);                // return with no error
}

/*************************************************************************************
 * ROUTINE NAME : 	I2Cm_Restart
 * DESCRIPTION : 	Reset the I2C Bus.
 *************************************************************************************/
void I2Cm_Restart(void)
{
  SSPCON2bits.RSEN = 1;           // initiate bus restart condition
}

