#include "sdmmc.h"
#include "system.h"

#define FPB GetPeripheralClock()
// I/O definitions
#define SDWP    0    // Write Protect input
#define SDCD    0    // Card Detect input
#define SDCS    _RB13    // Card Select output

// SD card commands
#define RESET           0 // a.k.a. GO_IDLE (CMD0)
#define INIT            1 // a.k.a. SEND_OP_COND (CMD1)
#define READ_SINGLE     17
#define WRITE_SINGLE    24

// additional commands (not used)
#define SEND_CSD        9
#define SEND_CID        10
#define SET_BLEN        16
#define APP_CMD         55
#define SEND_APP_OP     41

// SD card responses
#define DATA_START      0xFE
#define DATA_ACCEPT     0x05

// timeouts
#define I_TIMEOUT       100000
#define R_TIMEOUT       25000
#define W_TIMEOUT       250000

static BOOL highspeed = FALSE;

void initSD( void)
{
    SDCS = 1;           // initially keep the SD card disabled
    _TRISB13 = 0;        // make Card select an output pin

    // init the spi module for a slow (safe) clock speed first
    SPI2CON = 0x8120;   // ON, CKE=1; CKP=0, sample middle
    SPI2BRG = highspeed ? FPB/2/20000000-1 /* Fsck=20MHz */ : FPB/2/400000-1; /* Fsck=400kHz *///71;       // clock = Fpb/144 = 250kHz

}   // initSD


// send one byte of data and receive one back at the same time
unsigned char writeSPI( unsigned char b)
{
    SPI2BUF = b;                  // write to buffer for TX
    while( !SPI2STATbits.SPIRBF); // wait transfer complete
    return SPI2BUF;               // read the received value
}// writeSPI

#define readSPI()   writeSPI( 0xFF)
#define clockSPI()  writeSPI( 0xFF)
#define disableSD() SDCS = 1; clockSPI()
#define enableSD()  SDCS = 0


int sendSDCmd( unsigned char c, unsigned a)
// c    command code
// a    byte address of data block
{
    int i, r;

    // enable SD card
    enableSD();

    // send a comand packet (6 bytes)
    writeSPI( c | 0x40);    // send command
    writeSPI( a>>24);       // msb of the address
    writeSPI( a>>16);
    writeSPI( a>>8);
    writeSPI( a);           // lsb

    writeSPI( 0x95);        // send CMD0 CRC

    // now wait for a response, allow for up to 8 bytes delay
    for( i=0; i<8; i++)
    {
        r = readSPI();
        if ( r != 0xFF)
            break;
    }
    return (r);

/* return response
    FF - timeout
    00 - command accepted
    01 - command received, card in idle state after RESET

other codes:
    bit 0 = Idle state
    bit 1 = Erase Reset
    bit 2 = Illegal command
    bit 3 = Communication CRC error
    bit 4 = Erase sequence error
    bit 5 = Address error
    bit 6 = Parameter error
    bit 7 = Always 0
*/
    // NOTE CSCD is still low!
} // sendSDCmd


int initMedia( void)
// returns 0 if successful
//          E_COMMAND_ACK   failed to acknowledge reset command
//          E_INIT_TIMEOUT  failed to initialize
{
    int i, r;

    // 1. with the card NOT selected
    disableSD();

    // 2. send 80 clock cycles start up
    for ( i=0; i<10; i++)
        clockSPI();

    // 3. now select the card
    enableSD();


    // 4. send a single RESET command
    r = sendSDCmd( RESET, 0); disableSD();
    if ( r != 1)                // must return Idle
        return E_COMMAND_ACK;   // comand rejected
    // 5. send repeatedly INIT until Idle terminates
    for (i=0; i<I_TIMEOUT; i++){
        r = sendSDCmd(INIT, 0); disableSD();
        if ( !r)
            break;
    }
    if ( i == I_TIMEOUT)
        return E_INIT_TIMEOUT;  // init timed out

    return 0;
} // init media
