;*******************************************************************
;
;	CPU configuration
;

;	MESSG		"Processor = 16F628"


;*******************************************************************
;
;	Assembler messages
;
	errorlevel -302		; PA3BCA - rot op met die bank not 0 messages

#include	<macros.inc>
#include	<defines.inc>

	__CONFIG        _CP_OFF & _WDT_OFF & _PWRTE_ON & _HS_OSC & _BODEN_ON & _LVP_OFF


	extern	FPS24, FPD24, FPM24, INT2424, FLO2424
	extern  LCD_Init, LCD_Clear, LCD_Home, LCD_Line2, Write_BCD_Nybble, LCD_Write_Char
	extern	Delay_2MS, MS100, MS200

	global CHR, PB_data, COUNT1, COUNT2
	global AARGB4, AARGB3, AARGB2, AARGB1, AARGB0, AEXP, SIGN
	global TEMPB0, TEMPB1
	global FPFLAGS, BEXP, BARGB0, BARGB1, BARGB2

	udata	0x20	; PA3BCA - Force in bank 1 to avoid banking mayhem
;
;       Floating Point Stack and other locations used by FP.TXT
;
;	FP Stack: TOS	A = 	AEXP:AARGB0:AARGB1:AARGB3:AARGB4
;			B = 	BEXP:BARGB0:BARGB1:BARGB2
;			C = 	CEXP:CARGB0:CARGB1
AARGB4	res	1
AARGB3	res	1
AARGB2	res	1
AARGB1	res	1
AARGB0	res	1
AEXP	res	1			; 8 bit biased exponent for argument A
SIGN	res	1			; save location for sign in MSB

FPFLAGS	res	1			; floating point library exception flags
	
BARGB2	res	1
BARGB1	res	1
BARGB0	res	1
BEXP	res	1			; 8 bit biased exponent for argument B

TEMPB3	res	1			; 1 Unused byte
TEMPB2	res	1			; 1 Unused byte
TEMPB1	res	1			; Used
TEMPB0	res	1			; 1 Unused byte

CARGB1	res	1
CARGB0	res	1			; most significant byte of argument C
CEXP	res	1			; 8 bit biased exponent for argument C

;
;	"Main" Program Storage
;
 
COUNT	res	1			; Bin to BCD convert (bit count)
cnt	res	1			;                    (BCD BYTES)

CHR	res	1 
CHR1	res	1

F1	res	2
F2	res	2
F3	res	2
	
bcd	res	4			; BCD, MSD first 

TabStop	res	1			; Used to fix bad displays.
TabTemp	res	1

FPE	res	1			; Collect FP errors in here
	
R_sign	res	1			; Holds "+" or " " (sign)

EEflag	res	1		; Cal adjust flag

	

	udata_shr
cal_t	res	2			; Ccal temporary value
PB_data	res	1		; LCD output munger temp
	
links	res	1			; User test links copy

COUNT1	res	1			; Used by delay routines
			; and "prescaler flush"
COUNT2	res	1			; Timing (100ms)


;**********************************************************
;
;	Begin Executable Stuff(tm)
;

Main	code	0x0000

	call	InitIO
	bcf	relay		; Remove Ccal

;**********************************************************
;
;	Main Program
;

START	
	CALL	LCD_Init	 	; INITIALIZE LCD MODULE	
	call	EE_Read_Ccal		; Retrieve CCal integer value
;
;	PA3BCA:
;	test if we have a POR (Power on Reset) or a reset/cal via MCLR
;	if POR then wait 5 seconds before calibrating
;	This allows the osc to settle enough
;
	bank1
	btfsc	PCON, NOT_POR		; power down bit, so device just started - POR and no reset by MCLR
	goto	no_por			; POR, so delay 5 seconds before calibrating to allow startup

	bsf	PCON, NOT_POR		;POR was detected. now set bit to distinguish future MCLR resets
	bank0
	movlw	pa3bca-0x2100		;print POR startup message pa3bca enzo
	call	pmsg
	call	LCD_Line2	
	movlw	StartupMsg-0x2100	;print POR startup message = "please wait..."
	call	pmsg

	movlw	.50			; 50 x 100 ms = 5 seconds
	movwf	COUNT
startuploop
	call	MS100
	decfsz	COUNT,F
	goto	startuploop

no_por	
	bank0
	call	LCD_Clear
	call	LCD_Home		; LCD cursor to HOME

;
;	"Zero" the meter.
;
	call	Calibrate

;
;	Read state of user test links on LCD bus
;

M_F3	
	bank1			; PORTB:-
	movlw	b'01001111'	; LCD data bits to read
	movwf	TRISB		; 1 = input
	bank0			; 0 = output

	call	Delay_2MS	; Settling time for RMW problem
	copy	PORTB,links

	bank1
	movlw	b'01000000'	; restore data direction
	movwf	TRISB		; 1 = input
	bank0			; 0 = output

;---------------------------------------------------------------
;
;	Take a break from regular duties to do something interesting
;
	btfss	links,0		; Raise Ccal value
	goto	cal_up

	btfss	links,1		; Lower Ccal value
	goto	cal_dn

	btfss	links,2		; Test osc without Ccal
	goto	osc1

#ifndef	DEBUG
; PB3 is used by ICD2, if connected this will trigger following code
; so ignore when in debug mode - no need to disconnect ICD2
;
	btfss	links,3		; Test osc with Ccal
	goto	osc2
#endif

;
;	None of the above
;

	bcf	relay		; In case of osc test
	btfss	EEflag,0	; Time to save Ccal value?
	goto	cont		; No. Back to work
	
	bcf	EEflag,0	; To say we have done it
	call	EE_Write_Ccal		; So, we better save it
	goto	cont		; Hi Ho, its off to work I go

;
;	Add +10 to cal_t:2
;

cal_up	bsf	EEflag,0	; Say "we're adjusting"
	movlw	0x0a		; +10
	addwf	cal_t+1,f
	bcc	cont
	
	incf	cal_t+0,f
	goto cont

;
;	Add -10 to cal_t:2
;

cal_dn	bsf	EEflag,0	; Say "we're adjusting"
	movlw	0xf6		; -10
	addwf	cal_t+1,f
	bcc	hi_byte
	
	incf	cal_t+0,f

hi_byte	movlw	0xff
	addwf	cal_t+0,f
	goto	cont		

;
;	Measure & display osc freq for initial setup
;

osc2	bsf	relay		; Add Ccal
	
osc1	call	LCD_Home
	call	Measure		; Measure Local Osc Freq.
	call	LCD_Clear

	btfss	INTCON,T0IF	; Set = Counter overflow?
	goto	Do_Disp

	MOVLW	ovr-0x2100	; Over-range message
	call	pmsg
	goto	M_F3
	
Do_Disp	clrf	AARGB0		; Copy to 24 bit number
	movf	F3,W		; in AARGB0, 1, 2
	movwf	AARGB1		; for display
	movf	F3+1,W
	movwf	AARGB2

	call	Display
	goto	M_F3

;---------------------------------------------------------------

cont	
	call	LCD_Home
	call	MS200
	call	Measure		; Measure F3 & leave it there

	movf	F3,w		; test for "too low" frequency
	beq		OORange		; F < 2560Hz ?
	
	btfss	INTCON,T0IF	; test for "too high" frequency
	goto	OK2GO		; F > 655359Hz ?

OORange	
	MOVLW	ovr-0x2100	; Over/Under range message
	call	pmsg
	
	goto	M_F3

;
;	Precompute major bracketed terms cos
;	we need 'em both for all calculations
;

OK2GO	
	clrf	FPE		; Declare "error free"
	call	F1_F2
	call	F1_F3

;
;	See what mode we are in
;

	btfss	functn		; 0=Inductor
	goto	Do_Ind

;
;	OK, we've been told it's a capacitor
;

Do_Cap	
	call	C_calc
	movf	FPE,f		; Any FP errors?
	bne	complain
	
	movlw	Cintro-0x2100	; C =
	call	pmsg

	call	C_disp
	goto	M_F3

;
;	Now, they reckon it's a @#$*! inductor
;

Do_Ind	
	call	L_calc
	movf	FPE,f		; Any FP errors?
	bne	complain
	
	movlw	Lintro-0x2100	; L =
	call	pmsg

	call	L_disp
	goto	M_F3

;
;	Got a Floating Point Error of some sort
;

complain	
	movlw	ovr-0x2100	; Over Range
	call	pmsg
	
	goto	M_F3

;**********************************************************
;
;	Print String addressed by W
;	Note: Strings are in EEPROM
;	We do a lotta bank switching here.

pmsg	
	bank1
	movwf	EEADR		; pointer

pm1	BSF     EECON1,RD       ; EE Read
	MOVF    EEDATA,W        ; W = EEDATA, affects Z bit
	bank0			; Does not change Z bit
 
	btfsc	STATUS,Z	; ZERO = All done
	return			; so quit

	call	LCD_Write_Char		; Byte -> display

	bank1
	INCF    EEADR,F         ; bump address
	goto	pm1



;******************************************************************
;
;	Convert 24-bit binary number at <AARGB0,1,2> into a bcd number
;	at <bcd>. Uses Mike Keitz's procedure for handling bcd 
;	adjust; Modified Microchip AN526 for 24-bits.
;

B2_BCD

b2bcd   movlw   .24		; 24-bits
        movwf   COUNT		; make cycle counter

        clrf    bcd+0		; clear result area
        clrf    bcd+1
        clrf    bcd+2
        clrf    bcd+3
        
b2bcd2  movlw   bcd 		; make pointer
        movwf   FSR
        movlw   .4
        movwf   cnt

; Mike's routine:

b2bcd3  movlw   0x33            
        addwf   INDF,f          ; add to both nybbles
        btfsc   INDF,3          ; test if low result > 7
        andlw   0xf0            ; low result >7 so take the 3 out
        btfsc   INDF,7          ; test if high result > 7
        andlw   0x0f            ; high result > 7 so ok
        subwf   INDF,f          ; any results <= 7, subtract back
        incf    FSR,f           ; point to next
        decfsz  cnt,f
        goto    b2bcd3
        
        rlf     AARGB2,f	; get another bit
        rlf     AARGB1,f
        rlf     AARGB0,f

        rlf     bcd+3,f         ; put it into bcd
        rlf     bcd+2,f
        rlf     bcd+1,f
        rlf     bcd+0,f

        decfsz  COUNT,f         ; all done?
        goto    b2bcd2          ; no, loop
        return                  ; yes




;********************************************************************
;       Initialise Input & Output devices
;********************************************************************

InitIO	movlw	b'00000110'
	movwf	CMCON 	   	; Select Comp mode

	bank1

	movlw	b'00000000'
	movwf	VRCON 	   	; Set Volt ref mode to OFF

	movlw	0x37		; Option register
	movwf	OPTION_REG	; Port B weak pull-up enabled
				; INTDEG Don't care
				; Count RA4/T0CKI
				; Count on falling edge
				; Prescale Timer/counter
				; divide Timer/counter by 256

				; PORTA:-
	movlw	b'11110111'	; initialise data direction
				; 1 = input
				; 0 = output
								;
	movwf	TRISA		; PORTA<0>   = comp1 "-" in
				; PORTA<1>   = comp2 "-" in
				; PORTA<2>   = comp1&2 "+" in
				; PORTA<3>   = comp1 out
				; PORTA<4>   = comp2 out, T0CKI in
				; PORTA<5:7> = unused
				;
				;
				; PORTB:-
	movlw	b'01000000'	; initialise data direction
				; 1 = input
				; 0 = output
				;
	movwf	TRISB		; PORTB<0>   = LCD out "DB4"
				; PORTB<1>   =         "DB5"
				; PORTB<2>   =         "DB6"
				; PORTB<3>   =         "DB7"
				; PORTB<4>   = E  out to LCD
				; PORTB<5>   = RS out to LCD
				; PORTB<6>   = function in
				; PORTB<7>   = Ccal switch out

	bank0

	return	

;**********************************************************
;
;	Calibrate the meter, read F1 (without Cref)and F2 (With Cref)
;

Calibrate	
	MOVLW   Calibr-0x2100	; Display's " Calibrating "
	call	pmsg		

	IFDEF __16F628A		; PA3BCA:
	call	MS200		; If 628A version is used more delay is needed to stabilize the osc
	call	MS200		; or else the meter will not calibrate to 0.0 pF
	call	MS200
	ENDIF


	call	MS200
	call	Measure		; Get freq in F3

	copy	F3+0,F1+0	; Copy F3 to F1
	copy	F3+1,F1+1

	bsf	relay		; Add standard capacitor

	call	MS200

	call	Measure		; Get freq in F3

	copy	F3+0,F2+0	; Copy F3 to F2
	copy	F3+1,F2+1
	
	bcf	relay		; Remove standard capacitor


	call	Measure		; Dummy Run to stabilise oscillator.
	return

;**********************************************************
;
;	Measure Frequency. Stash in "F3 and F3+1"
;

Measure	bcf	INTCON,T0IF	; Declare "Not yet Over-range"
	CLRF	TMR0		; RESET INTERNAL COUNT (INCLUDING PRESCALER)
				; See page 27 Section 6.0

	CLRF	F3		; Ready to receive 16 bit number
	CLRF	F3+1

	bank1
				; OPEN GATE
	
	movlw	b'11100111'	; Enable RA4 output to T0CKI
	movwf	TRISA		; 1 = input
				; 0 = output
	
	CALL	MS100		; 100MS DELAY

				; CLOSE GATE (COUNT COMPLETE)

	movlw	b'11110111'	; Disable RA4 output to T0CKI
	movwf	TRISA		; 1 = input
				; 0 = output
	bank0

	MOVF	TMR0,W		; GET HIGH BYTE		
	MOVWF	F3		; Copy to Big end of 16 bit result

; The comparator is "outputting" a 1 'cos we've forced it high
; so T0CKI=1.

PSC1	bank1
	bsf	OPTION_REG,T0SE	; Clock the prescaler
	nop
	bcf	OPTION_REG,T0SE
	bank0
	
	DECF	F3+1,F		; Decrement the counter
	
	movf	TMR0,W		; Has TMR0 changed?
	xorwf	F3,W		; if unchanged, XOR -> 0

	beq	PSC1

	return			; F3 : F3+1 now holds 16 bit result

;**********************************************************
;
;	Display contents of AARGB0,1,2 on LCD
;	First convert to BCD, Then ASCII (nybble at a time)
;

Display	CALL	B2_BCD		; CONVERT COUNT TO BCD		

	call	Swap0		; GET NEXT DIGIT
	call	Move0		; GET OTHER BCD DIGIT
	call	Swap1
	call	Move1
	call	Swap2
	call	Move2
	call	Swap3
	goto	Move3		; includes return

;**********************************************************
;
;	Formatted display of BCD work area for Capacitor
;

C_disp	movf	R_sign,w	; Sign
	call	LCD_Write_Char

F_C1	MOVF	bcd+0,W
	ANDLW	0x0F
	beq	F_C2

	CALL	Write_BCD_Nybble
	call	Swap1
	call	Move1
	CALL	DoDP		; Print DP
	call	Swap2
	goto	F_C3U

;--------------------------------------------------

F_C2	swapf	bcd+1,W
	ANDLW	0x0F
	beq	F_C3

	CALL	Write_BCD_Nybble
	call	Move1
	CALL	DoDP		; Print DP
	call	Swap2
	call	Move2
	goto	F_C3U		; print nF. includes RETURN

;--------------------------------------------------

F_C3	MOVF	bcd+1,W
	ANDLW	0x0F
	beq	F_C4

	CALL	Write_BCD_Nybble
	CALL	DoDP		; Print DP
	call	Swap2
	call	Move2
	call	Swap3

F_C3U	movlw	Unit1-0x2100	; nF
	goto	pmsg		; includes RETURN

;--------------------------------------------------

F_C4	SWAPF	bcd+2,W		; Digit1 == 0 ?
	ANDLW	0x0F
	bne	NoB1_C

	MOVLW	0x20		; YES PRINT A SPACE
	call	LCD_Write_Char

	MOVF	bcd+2,W		; Digit2 == 0 ?
	ANDLW	0x0F
	bne	NoB2_C

	MOVLW	0x20		; YES PRINT A SPACE
	call	LCD_Write_Char
	bra	NoB3_C

NoB1_C	call	Swap2		; 1
NoB2_C	call	Move2		; 2
NoB3_C	call	Swap3		; 3
	CALL	DoDP		; Print DP
	call	Move3		; 4

	movlw	Unit2-0x2100	; pF
	goto	pmsg		; includes RETURN

;**********************************************************
;
;	Formatted display of BCD work area for Inductor
;

L_disp	movf	R_sign,w	; Sign
	call	LCD_Write_Char

F_L1	MOVF	bcd+0,W
	ANDLW	0x0F
	beq	F_L2

	CALL	Write_BCD_Nybble
	call	Swap1
	CALL	DoDP		; Print DP
	call	Move1
	call	Swap2
	goto	F_L2U		; Print mH. includes RETURN

;--------------------------------------------------

F_L2	swapf	bcd+1,W
	ANDLW	0x0F
	beq	F_L3

	CALL	Write_BCD_Nybble
	CALL	DoDP		; Print DP
	call	Move1
	call	Swap2
	call	Move2
	
F_L2U	movlw	Unit3-0x2100	; mH
	goto	pmsg		; includes RETURN

;--------------------------------------------------

F_L3	MOVF	bcd+1,W
	ANDLW	0x0F
	beq	F_L4

	CALL	Write_BCD_Nybble
	call	Swap2
	call	Move2
	CALL	DoDP		; Print DP
	call	Swap3
	goto	F_L4U		; Print uH. includes RETURN

;--------------------------------------------------

F_L4	SWAPF	bcd+2,W		; Digit1 == 0 ?
	ANDLW	0x0F
	bne	NoB1_L

	MOVLW	0x20		; YES PRINT A SPACE
	call	LCD_Write_Char

	goto	NoB2_L

NoB1_L	call	Swap2		; 1
NoB2_L	call	Move2		; 2
	CALL	DoDP		; Print DP
	call	Swap3		; 3
	call	Move3		; 4

F_L4U	movlw	Unit4-0x2100	; uH
	goto	pmsg		; includes RETURN

;--------------------------------------------------
;
;	Common subroutine for formatted output
;

DoDP	MOVLW	'.'		; Print DP
	goto	LCD_Write_Char		; Return from LCD_Write_Char

Swap0	SWAPF	bcd+0,W		; GET NEXT DIGIT
	goto	Write_BCD_Nybble		; DISPLAY IT

Move0	MOVF	bcd+0,W		; GET OTHER BCD DIGIT
	goto	Write_BCD_Nybble

Swap1	SWAPF	bcd+1,W
	goto	Write_BCD_Nybble

Move1	MOVF	bcd+1,W
	goto	Write_BCD_Nybble

Swap2	SWAPF	bcd+2,W
	goto	Write_BCD_Nybble

Move2	MOVF	bcd+2,W
	goto	Write_BCD_Nybble

Swap3	SWAPF	bcd+3,W
	goto	Write_BCD_Nybble

Move3	MOVF	bcd+3,W
	goto	Write_BCD_Nybble

;********************************************************************
;
;	Stack operations
;
;********************************************************************	

;add		call	FPA24
;		goto	S_fix

subtract	call	FPS24
		goto	S_fix

divide		call	FPD24
		goto	S_fix

multiply	call	FPM24
;		goto	S_fix

;
;	Fix stack after add, subtract, divide & multiply
;	AND Collect ALL Floating Point Errors in FPE

S_fix	iorwf	FPE,f			; W may hold Error (0xff)

	copy	CARGB1,BARGB1		; C -> B
	copy	CARGB0,BARGB0
	copy	CEXP,BEXP
	return

;
;	Push stack (duplicates TOS)
;

S_push	copy	BARGB1,CARGB1		; B -> C
	copy	BARGB0,CARGB0
	copy	BEXP,CEXP

	copy	AARGB1,BARGB1		; A -> B
	copy	AARGB0,BARGB0
	copy	AEXP,BEXP
	return

;
;	Swap A and B

S_swap	swap	AARGB1,BARGB1		; A <-> B
	swap	AARGB0,BARGB0
	swap	AEXP,BEXP
	return

;********************************************************************
;
;	Calculate Unknown Capacitance OR inductance
;	
;       Output: 24 bit positive integer (scaled)
;	right justified in AARGB0, AARGB1, AARGB2
;	also as BCD in bcd:bcd+1:bcd+2:bcd+3
;
;********************************************************************	

C_calc	call	divide
	call	Get_Ccal	; Times 10,000 ( = 1000.0pF)
	call	multiply
	goto	PorM		; includes return

;--------------------------------------------------------------------

L_calc	call	multiply
	call	Get_Lscale	; Precomputed Scale_factor/(4*PI*PI)
	call	multiply
	call	Get_Ccal
	call	S_swap
	call	divide

L_divF1	call	Get_F1		; Divide by F1^2
	call	S_push
	call	multiply
	call	S_swap
	call	divide

;
;	Handle Space or Minus in front of FP number
;	

PorM	btfss	AARGB0,7	; test sign
	goto	Pplus
	
Pminus	movlw	0x2d		; minus
	goto	PMdisp

Pplus	movlw	0x20		; plus

PMdisp	movwf	R_sign		; save for later display
	bcf	AARGB0,7	; make plus anyway

;
;	Format as raw BCD string in bcd:bcd+1:bcd+2:bcd+3
;

	call	INT2424		; To INT in AARGB0 etc.
	iorwf	FPE,f		; W may hold Error (0xff)
	goto	B2_BCD		; includes return

;********************************************************************
;
;	Calculate (F1/F3)^2-1, leave result on stack
;
;********************************************************************	

F1_F3	call	Get_F3
	goto	F1_F1

;********************************************************************
;
;	Calculate (F1/F2)^2-1, leave result on stack
;
;********************************************************************	

F1_F2	call	Get_F2
F1_F1	call	Get_F1
	call	divide		; F1/Fx
	call	S_push
	call	multiply	; (F1/Fx)^2
	call	Get_One
	call	S_swap
	goto	subtract	; (F1/Fx)^2-1
				; includes return

;********************************************************************
;	Fetch assorted things used for the calculation
;	of Unknown L and C
;
;********************************************************************	

Get_Lscale	call	S_push		; make room first
		movlw	0xB8		; 2.53303e+17
		movwf	AEXP		; Create FP version of
		movlw	0x60		; Precomputed 1/(4*PI*PI)
		movwf	AARGB0		; times any needed
		movlw	0xFA		; fiddle factor (1/100)
		movwf	AARGB1
		return

Get_One		call	S_push		; make room first
		clrf	AEXP		; Create a binary 1
		clrf	AARGB0
		clrf	AARGB1
		movlw	0x01
		goto	LSB2stak
	

Get_Ccal	movlw	cal_t		; Get integer value
		goto	W2stak		; Includes stack push

Get_F1		movlw	F1		; Includes stack push
		goto	W2stak

Get_F2		movlw	F2		; Includes stack push
		goto	W2stak

Get_F3		movlw	F3		; Includes stack push
;		goto	W2stak

;********************************************************************
;	Copy 16 bit number, pointed to by W, to stack
;	and convert to FP (positive value only)
;	via a 24 bit number in AARGB0,1,2
;********************************************************************	

W2stak		movwf	FSR
		call	S_push		; make room first

		clrf	AEXP
		clrf	AARGB0

		movf	INDF,W		; Big Byte first
		movwf	AARGB1

		incf	FSR,F		; then little byte

		movf	INDF,W
LSB2stak	movwf	AARGB2
		
		CALL	FLO2424		; 24 bit int -> 24 bit FP
		iorwf	FPE,f		; W may hold Error (0xff)
		RETURN

;********************************************************************
;	Read EEPROM into "cal_t"
;********************************************************************	

EE_Read_Ccal	
	bank1
	movlw	cal_p-0x2100	; Address to read
	MOVWF	EEADR
	CALL	EE_R
	MOVWF	cal_t+0
	INCF	EEADR,F		; bump address
	CALL	EE_R
	MOVWF	cal_t+1
	bank0
	RETURN
EE_R	
	BSF	EECON1,RD	; EE Read
	MOVF	EEDATA,W	; W = EEDATA
	RETURN


;********************************************************************
;	Write EEPROM from "cal_t"
;********************************************************************	

EE_Write_Ccal	
	bank1
	movlw	cal_p-0x2100
	MOVWF	EEADR		; Address to write

	MOVF	cal_t+0,W	; Data byte #0
	CALL	EE_W

	MOVF	cal_t+1,W	; Data byte #1
	CALL	EE_Winc

	bank0
	RETURN

	errorlevel	-302	; In Bank 2

EE_Winc	INCF	EEADR,F		; bump address
	
EE_W	MOVWF	EEDATA
	BSF	EECON1,WREN	; Enable Write
	MOVLW	0x55		;
	MOVWF	EECON2		; Write 0x55
	MOVLW	0xAA		;
	MOVWF	EECON2		; Write 0xAA
	BSF	EECON1,WR	; Set WR bit (begin write)

EE_W2	BTFSC	EECON1,WR	; Wait for write to finish
	GOTO	EE_W2

	bank0
	BCF	PIR1,EEIF	; clear interrupts
	bank1
	RETURN	

;
;	Text Strings (stored in data EEPROM)
;

EEPROM	code	0x2100

ovr	de	"   Over Range   ",0
Unit1	de	" nF",0
Unit2	de	" pF",0
Unit3	de	" mH",0
Unit4	de	" ", 0xe4,"H",0		;PA3BCA 0xe4 is greek micro char
Cintro	de	" C = ",0
Lintro	de	" L = ",0
Calibr  de	"   Calibrating  ",0
StartupMsg	de	"Please wait..",0
pa3bca	de	"Pa3bca LC-Meter", 0

;cal_p	de	0x27,0x10		; Initial value = 10000
cal_p	de	0x27,0x60		; Initial value = 10080  (1008 pf as measured by AIM4170)
					; PA0DLN supplied Silver Mica reference C

 	END

;********************************************************************
