#include <math.h>
#include "uart_lib.h"


// Actual interrupt handlers //////////////////////////////////////////////////////////////

void Serial_tx_udr_empty_irq(void)
{
	// If interrupts are enabled, there must be more data in the output
	// buffer. Send the next byte
	unsigned char c = _tx_buffer[_tx_buffer_tail];
	_tx_buffer_tail = (_tx_buffer_tail + 1) % SERIAL_BUFFER_SIZE;

	_udr = c;

	// clear the TXC bit -- "can be cleared by writing a one to its bit
	// location". This makes sure flush() won't return until the bytes
	// actually got written
	sbi(_ucsra, TXC0);

	if (_tx_buffer_head == _tx_buffer_tail) {
		// Buffer empty, so disable interrupts
		cbi(_ucsrb, UDRIE0);
	}
}


// Actual interrupt handlers //////////////////////////////////////////////////////////////

void Serial_rx_complete_irq(void)
{
	if (bit_is_clear(_ucsra, UPE0)) {
		// No Parity error, read byte and store it in the buffer if there is
		// room
		unsigned char c = _udr;
		uint8_t i = (unsigned int)(_rx_buffer_head + 1) % SERIAL_BUFFER_SIZE;

		// if we should be storing the received character into the location
		// just before the tail (meaning that the head would advance to the
		// current location of the tail), we're about to overflow the buffer
		// and so we don't write the character or advance the head.
		if (i != _rx_buffer_tail) {
			_rx_buffer[_rx_buffer_head] = c;
			_rx_buffer_head = i;
		}
		} else {
		// Parity error, read byte but discard it
		_udr;
	};
}

#if defined(USART_RX_vect)
ISR(USART_RX_vect)
#elif defined(USART0_RX_vect)
ISR(USART0_RX_vect)
#elif defined(USART_RXC_vect)
ISR(USART_RXC_vect) // ATmega8
#else
#error "Don't know what the Data Received vector is called for the first UART"
#endif
{
	Serial_rx_complete_irq();
}


#if defined(UART0_UDRE_vect)
ISR(UART0_UDRE_vect)
#elif defined(UART_UDRE_vect)
ISR(UART_UDRE_vect)
#elif defined(USART0_UDRE_vect)
ISR(USART0_UDRE_vect)
#elif defined(USART_UDRE_vect)
ISR(USART_UDRE_vect)
#else
#error "Don't know what the Data Register Empty vector is called for the first UART"
#endif
{
	Serial_tx_udr_empty_irq();
}



// Public Methods //////////////////////////////////////////////////////////////

void Serial_begin(unsigned long baud, uint8_t config)
{
	// Try u2x mode first
	uint16_t baud_setting = (F_CPU / 4 / baud - 1) / 2;
	_ucsra = 1 << U2X0;

	// hardcoded exception for 57600 for compatibility with the bootloader
	// shipped with the Duemilanove and previous boards and the firmware
	// on the 8U2 on the Uno and Mega 2560. Also, The baud_setting cannot
	// be > 4095, so switch back to non-u2x mode if the baud rate is too
	// low.
	if (((F_CPU == 16000000UL) && (baud == 57600)) || (baud_setting >4095))
	{
		_ucsra = 0;
		baud_setting = (F_CPU / 8 / baud - 1) / 2;
	}

	// assign the baud_setting, a.k.a. ubbr (USART Baud Rate Register)
	_ubrrh = baud_setting >> 8;
	_ubrrl = baud_setting;

	_written = false;
	_ucsrc = config;
	
	sbi(_ucsrb, RXEN0);
	sbi(_ucsrb, TXEN0);
	sbi(_ucsrb, RXCIE0);
	cbi(_ucsrb, UDRIE0);
	_rx_buffer_head = 0;
	_rx_buffer_tail = 0;
	_tx_buffer_head = 0;
	_tx_buffer_tail = 0;
}

void Serial_end()
{
	// wait for transmission of outgoing data
	while (_tx_buffer_head != _tx_buffer_tail)
	;

	cbi(_ucsrb, RXEN0);
	cbi(_ucsrb, TXEN0);
	cbi(_ucsrb, RXCIE0);
	cbi(_ucsrb, UDRIE0);
	
	// clear any received data
	_rx_buffer_head = _rx_buffer_tail;
}

int Serial_available(void)
{
	return (unsigned int)(SERIAL_BUFFER_SIZE + _rx_buffer_head - _rx_buffer_tail) % SERIAL_BUFFER_SIZE;
}

int Serial_peek(void)
{
	if (_rx_buffer_head == _rx_buffer_tail) {
		return -1;
		} else {
		return _rx_buffer[_rx_buffer_tail];
	}
}

uint8_t Serial_read(void)
{
	// if the head isn't ahead of the tail, we don't have any characters
	if (_rx_buffer_head == _rx_buffer_tail) {
		return -1;
		} else {
		uint8_t c = _rx_buffer[_rx_buffer_tail];
		_rx_buffer_tail = (uint8_t)(_rx_buffer_tail + 1) % SERIAL_BUFFER_SIZE;
		return c;
	}
}

void Serial_flush()
{
	// If we have never written a byte, no need to flush. This special
	// case is needed since there is no way to force the TXC (transmit
	// complete) bit to 1 during initialization
	if (!_written)
	return;

	while (bit_is_set(_ucsrb, UDRIE0) || bit_is_clear(_ucsra, TXC0)) {
		if (bit_is_clear(SREG, SREG_I) && bit_is_set(_ucsrb, UDRIE0))
		// Interrupts are globally disabled, but the DR empty
		// interrupt should be enabled, so poll the DR empty flag to
		// prevent deadlock
		if (bit_is_set(_ucsra, UDRE0))
		Serial_tx_udr_empty_irq();
	}
	// If we get here, nothing is queued anymore (DRIE is disabled) and
	// the hardware finished tranmission (TXC is set).
}

uint8_t Serial_writeChar(uint8_t c)
{
	// If the buffer and the data register is empty, just write the byte
	// to the data register and be done. This shortcut helps
	// significantly improve the effective datarate at high (>
	// 500kbit/s) bitrates, where interrupt overhead becomes a slowdown.
	if (_tx_buffer_head == _tx_buffer_tail && bit_is_set(_ucsra, UDRE0)) {
		_udr = c;
		sbi(_ucsra, TXC0);
		return 1;
	}
	uint8_t i = (_tx_buffer_head + 1) % SERIAL_BUFFER_SIZE;
	
	// If the output buffer is full, there's nothing for it other than to
	// wait for the interrupt handler to empty it a bit
	while (i == _tx_buffer_tail) {
		if (bit_is_clear(SREG, SREG_I)) {
			// Interrupts are disabled, so we'll have to poll the data
			// register empty flag ourselves. If it is set, pretend an
			// interrupt has happened and call the handler to free up
			// space for us.
			if(bit_is_set(_ucsra, UDRE0))
			Serial_tx_udr_empty_irq();
			} else {
			// nop, the interrupt handler will free up space for us
		}
	}

	_tx_buffer[_tx_buffer_head] = c;
	_tx_buffer_head = i;
	
	sbi(_ucsrb, UDRIE0);
	_written = true;
	
	return 1;
}



/* default implementation: may be overridden */
uint8_t Serial_writeBuffer(const uint8_t *buffer, size_t size)
{
	uint8_t n = 0;
	while (size--) {
		n += Serial_writeChar(*buffer++);
	}
	return n;
}

uint8_t Serial_writeStr(const char *str) {
	if (str == 0) return 0;
	return Serial_writeBuffer((const uint8_t *)str, strlen(str));
}

uint8_t Serial_printStr(const char str[])
{
	return Serial_writeStr(str);
}

uint8_t Serial_printChar(char c)
{
	return Serial_writeChar(c);
}

uint8_t Serial_printuChar(unsigned char b, int base)
{
	return Serial_printuLong((unsigned long) b, base);
}

uint8_t Serial_printInt(int n, int base)
{
	return Serial_printLong((long) n, base);
}

uint8_t Serial_printuInt(unsigned int n, int base)
{
	return Serial_printuLong((unsigned long) n, base);
}

uint8_t Serial_printLong(long n, int base)
{
	if (base == 0) {
		return Serial_writeChar(n);
		} else if (base == 10) {
		if (n < 0) {
			int t = Serial_printChar('-');
			n = -n;
			return printNumber(n, 10) + t;
		}
		return printNumber(n, 10);
		} else {
		return printNumber(n, base);
	}
}

uint8_t Serial_printlong(unsigned long n, int base)
{
	if (base == 0) return Serial_writeChar(n);
	else return printNumber(n, base);
}

uint8_t Serial_printDouble(double n, int digits)
{
	return printFloat(n, digits);
}


uint8_t Serial_println(void)
{
	uint8_t n = Serial_printChar('\r');
	n += Serial_printChar('\n');
	return n;
}

uint8_t Serial_printlnStr(const char c[])
{
	uint8_t n = Serial_printStr(c);
	n += Serial_println();
	return n;
}

uint8_t Serial_printlnChar(char c)
{
	uint8_t n = Serial_printChar(c);
	n += Serial_println();
	return n;
}

uint8_t Serial_printlnuChar(unsigned char b, int base)
{
	uint8_t n = Serial_printuChar(b, base);
	n += Serial_println();
	return n;
}

uint8_t Serial_printlnInt(int num, int base)
{
	uint8_t n = Serial_printInt(num, base);
	n += Serial_println();
	return n;
}

uint8_t Serial_printlnuInt(unsigned int num, int base)
{
	uint8_t n = Serial_printuInt(num, base);
	n += Serial_println();
	return n;
}

uint8_t Serial_printlnLong(long num, int base)
{
	uint8_t n = Serial_printLong(num, base);
	n += Serial_println();
	return n;
}

uint8_t Serial_printlnuLong(unsigned long num, int base)
{
	uint8_t n = Serial_printuLong(num, base);
	n += Serial_println();
	return n;
}

uint8_t Serial_printlnDouble(double num, int digits)
{
	uint8_t n = Serial_printDouble(num, digits);
	n += Serial_println();
	return n;
}


// Private Methods /////////////////////////////////////////////////////////////

uint8_t printNumber(unsigned long n, uint8_t base) {
	char buf[8 * sizeof(long) + 1]; // Assumes 8-bit chars plus zero byte.
	char *str = &buf[sizeof(buf) - 1];

	*str = '\0';

	// prevent crash if called with base == 1
	if (base < 2) base = 10;

	do {
		unsigned long m = n;
		n /= base;
		char c = m - base * n;
		*--str = c < 10 ? c + '0' : c + 'A' - 10;
	} while(n);

	return Serial_writeStr(str);
}

uint8_t printFloat(double number, uint8_t digits)
{
	uint8_t n = 0;
	
	if (isnan(number)) return Serial_printStr("nan");
	if (isinf(number)) return Serial_printStr("inf");
	if (number > 4294967040.0) return Serial_printStr ("ovf");  // constant determined empirically
	if (number <-4294967040.0) return Serial_printStr ("ovf");  // constant determined empirically
	
	// Handle negative numbers
	if (number < 0.0)
	{
		n += Serial_printChar('-');
		number = -number;
	}

	// Round correctly so that print(1.999, 2) prints as "2.00"
	double rounding = 0.5;
	for (uint8_t i=0; i<digits; ++i)
	rounding /= 10.0;
	
	number += rounding;

	// Extract the integer part of the number and print it
	unsigned long int_part = (unsigned long)number;
	double remainder = number - (double)int_part;
	n += Serial_printChar(int_part);

	// Print the decimal point, but only if there are digits beyond
	if (digits > 0) {
		n += Serial_printChar('.');
	}

	// Extract digits from the remainder one at a time
	while (digits-- > 0)
	{
		remainder *= 10.0;
		int toPrint = (int)(remainder);
		n += Serial_printChar(toPrint);
		remainder -= toPrint;
	}
	
	return n;
}

