#ifndef UART_LIB_H
#define UART_LIB_H

#ifndef F_CPU
/* prevent compiler error by supplying a default */
#warning "F_CPU n'est pas defini pour <util/delay.h>. par defaut : F_CPU = 16000000UL "
#define F_CPU 16000000UL
#endif

#include <avr/interrupt.h>
#include <string.h>


// Define constants and variables for buffering incoming serial data.  We're
// using a ring buffer (I think), in which head is the index of the location
// to which to write the next incoming character and tail is the index of the
// location from which to read.

#define SERIAL_BUFFER_SIZE 64

#define cbi(sfr, bit) (_SFR_BYTE(sfr) &= ~_BV(bit))
#define sbi(sfr, bit) (_SFR_BYTE(sfr) |= _BV(bit))

// Define config for Serial_begin(baud, config);
#define SERIAL_5N1 0x00
#define SERIAL_6N1 0x02
#define SERIAL_7N1 0x04
#define SERIAL_8N1 0x06
#define SERIAL_5N2 0x08
#define SERIAL_6N2 0x0A
#define SERIAL_7N2 0x0C
#define SERIAL_8N2 0x0E
#define SERIAL_5E1 0x20
#define SERIAL_6E1 0x22
#define SERIAL_7E1 0x24
#define SERIAL_8E1 0x26
#define SERIAL_5E2 0x28
#define SERIAL_6E2 0x2A
#define SERIAL_7E2 0x2C
#define SERIAL_8E2 0x2E
#define SERIAL_5O1 0x30
#define SERIAL_6O1 0x32
#define SERIAL_7O1 0x34
#define SERIAL_8O1 0x36
#define SERIAL_5O2 0x38
#define SERIAL_6O2 0x3A
#define SERIAL_7O2 0x3C
#define SERIAL_8O2 0x3E

#define DEC 10
#define HEX 16
#define OCT 8
#define BIN 2


class HardwareSerial
{
	protected:
	volatile uint8_t * const _ubrrh;
	volatile uint8_t * const _ubrrl;
	volatile uint8_t * const _ucsra;
	volatile uint8_t * const _ucsrb;
	volatile uint8_t * const _ucsrc;
	volatile uint8_t * const _udr;
	// Has any byte been written to the UART since begin()
	bool _written;

	volatile uint8_t _rx_buffer_head;
	volatile uint8_t _rx_buffer_tail;
	volatile uint8_t _tx_buffer_head;
	volatile uint8_t _tx_buffer_tail;

	// Don't put any members after these buffers, since only the first
	// 32 bytes of this struct can be accessed quickly using the ldd
	// instruction.
	unsigned char _rx_buffer[SERIAL_BUFFER_SIZE];
	unsigned char _tx_buffer[SERIAL_BUFFER_SIZE];
	
	private:
	uint8_t printNumber(unsigned long, uint8_t);
	uint8_t printFloat(double, uint8_t);

	public:
	inline HardwareSerial(
	volatile uint8_t *ubrrh, volatile uint8_t *ubrrl,
	volatile uint8_t *ucsra, volatile uint8_t *ucsrb,
	volatile uint8_t *ucsrc, volatile uint8_t *udr);
	void begin(unsigned long baud) { begin(baud, SERIAL_8N1); }
	void begin(unsigned long, uint8_t);
	void end();
	int available(void);
	int peek(void);
	uint8_t read(void);
	void flush(void);
	uint8_t write(uint8_t);
	inline uint8_t write(unsigned long n) { return write((uint8_t)n); }
	inline uint8_t write(long n) { return write((uint8_t)n); }
	inline uint8_t write(unsigned int n) { return write((uint8_t)n); }
	inline uint8_t write(int n) { return write((uint8_t)n); }
	
	uint8_t write(const char *str) {
		if (str == 0) return 0;
		return write((const uint8_t *)str, strlen(str));
	}
	uint8_t write(const uint8_t *buffer, uint8_t size);
	uint8_t write(const char *buffer, uint8_t size) {
		return write((const uint8_t *)buffer, size);
	}
	
	uint8_t print(const char[]);
	uint8_t print(char);
	uint8_t print(unsigned char, int = DEC);
	uint8_t print(int, int = DEC);
	uint8_t print(unsigned int, int = DEC);
	uint8_t print(long, int = DEC);
	uint8_t print(unsigned long, int = DEC);
	uint8_t print(double, int = 2);


	uint8_t println(const char[]);
	uint8_t println(char);
	uint8_t println(unsigned char, int = DEC);
	uint8_t println(int, int = DEC);
	uint8_t println(unsigned int, int = DEC);
	uint8_t println(long, int = DEC);
	uint8_t println(unsigned long, int = DEC);
	uint8_t println(double, int = 2);
	uint8_t println(void);
	// Interrupt handlers - Not intended to be called externally
	inline void _rx_complete_irq(void);
	void _tx_udr_empty_irq(void);
};
extern HardwareSerial Serial;

#endif // whole file
