/********************************************************************************
						Programme GCC_Serial_LCD
						Jean-Marie COLLETTE
						21/08/2014
Le programme est crit pour un Atmega8 tournant  8 MHz, sur oscillateur interne
car tout le PortB est occup par le byte  envoyer au LCD.
Ce programme reoit des caractres par l'USART Interrupt.
Les caractres sont d'abord envoys dans un buffer circulaire FIFO puis
envoys un  un sur un LCD de 2 lignes de 16 caractres.
La ligne en cours se termine lors de la rception du caractre '\r' (Carriage Return).
la ligne est complte avec des espaces et le curseur revient en position 1.
En mode "Terminal", si les deux lignes du LCD sont dja crites, le texte suivant
provoque un scroll  de l'afficheur : la deuxime ligne passe  la premire ligne et
le nouveau texte s'affiche sur la deuxime ligne.
La premire ligne est comptabilise comme ligne 0 et correspond  0b10000000.
La deuxime ligne est comptabilise comme ligne 1 et correspond  0b11000000.
Le curseur va de 0  15, soit 0b00000000  0b00001111.
La commande de positionnement est obtenue en additionnant la ligne et le curseur.
Soit pour la ligne 1 : de 0b10000000  0b10001111
	 pour la ligne 2 : de 0b11000000  0b11001111
********************************************************************************/

/*******************************************************************************
***********************  Types, Macros and Defines  ****************************
********************************************************************************/
//Types
typedef unsigned char byte;		//Dfinition du type "byte"

//Macros
#define setbit(value,bit) ((value) |= (1 << (bit)))
#define clearbit(value,bit) ((value) &= ~(1 << (bit)))

//Defines
#define F_CPU 8000000UL			//Frquence du C  8 MHz ( fixer avec les fuses)
#define USART_BAUDRATE 9600
#define BAUD_PRESCALE (((F_CPU/(USART_BAUDRATE*16UL)))-1)
#define LCD_CmdDir		DDRD	//Commandes sur le PortD
#define LCD_CmdPort 	PORTD
#define RS 				5		//Pin pour la commande RS (Register Select)
#define RW				6		//Pin pour la commande RW (Read/Write)
#define E				7		//Pin pour la commande E (Enable)

#define LCD_DataDir 	DDRB	//Donnes sur le PortB
#define LCD_DataPort	PORTB
#define LCD_DataPin		PINB

/*******************************************************************************
****************************  Includes  ****************************************
********************************************************************************/
#include <avr/io.h>
#include <avr/interrupt.h>
#include <stdio.h>
#include <util/delay.h>
#include <util/delay_basic.h>

/********************************************************************************
****************************  Function Prototypes  ******************************
*********************************************************************************/

void usart_init();				//Initialisation de l'USART

void LCD_Epulse();				//Provoque une pulse de Enable
void LCD_datawrite(byte data);	//Ecriture d'un caractre sur le LCD
void LCD_gotoXY(byte );			//Positionnement du curseur
void LCD_scroll();				//Scroll de la ligne 1 sur la ligne 0
void LCD_init();				//Initialisation du LCD


/********************************************************************************
****************************  Global Variables **********************************
*********************************************************************************/

byte L = 0;					//N de Ligne sur laquelle se trouve le curseur
byte ScrollFlag;			//Drapeau pour le dclenchement du scroll
#define databuf_size 64		//Cration d'un buffer de 64 caractres pour
byte databuf[databuf_size];	//la rception de l'USART
byte databuf_tail = 0;		//Index pour enlever un caractre du buffer
byte databuf_head = 0;		//Index pour ajouter un caractre au buffer
byte scrollbuf[16];			//Buffer pour scroller la ligne 1 sur la ligne 0


/********************************************************************************				
****************************  M  A  I  N  ***************************************
*********************************************************************************/
int main(void) 
{
	usart_init();									//Initialisation del'USART
	LCD_init();										//Initialisation du LCD

	while (1)										//Boucle gnrale de la fonction "main"
	{
		while (databuf_tail == databuf_head)		//boucle tant que le buffer est vide 
		{
			_delay_ms(1);
		}
		
		LCD_datawrite(databuf[databuf_tail]);		//criture du caractre
		databuf_tail++;								//Incrmente databuf_tail 
		databuf_tail = databuf_tail % databuf_size;	//l'index du buffer = modulo de la taille du buffer
	}
} 



/********************************************************************************/
void usart_init()	//Initialisation de l'USART
/********************************************************************************/
{
	// Turn on the transmission and reception circuitry
	UCSRB=(1<<RXEN)|(1<<TXEN); 
	// Use 8- bit character size
	UCSRC=(1<<URSEL)|(1<<UCSZ0)|(1<<UCSZ1);
	// Load upper 8- bits of the baud rate value into the high byte of the UBRR register 
	UBRRH=(BAUD_PRESCALE>>8); 
	// Load lower 8- bits of the baud rate value into the low byte of the UBRR register
	UBRRL=BAUD_PRESCALE;
	// Enable the USART Receive Complete interrupt ( USART_RXC )
	UCSRB|=(1<<RXCIE); 
	// Enable the Global Interrupt Enable flag so that interrupts can be processed
	sei();
}

/********************************************************************************/
ISR(USART_RXC_vect)		//Routine d'interruption de rception d'un caractre
/********************************************************************************/
{
	databuf[databuf_head]=UDR;					// Range le byte reu dans le buffer
	databuf_head++ ;							// Incrmente databuf_head 
	databuf_head = databuf_head % databuf_size;	//l'index du buffer = modulo de la taille du buffer
}


/********************************************************************************/
void LCD_Epulse()
/********************************************************************************/
	{
		setbit(LCD_CmdPort,E);		//Enable  1
		_delay_us(20);				//dlai court
		clearbit(LCD_CmdPort,E);	//Enable repasse  0
		_delay_us(20);				//dlai court
	}

/********************************************************************************/
void LCD_datawrite(byte data)
/********************************************************************************/
	{
		_delay_ms(20);
		switch (data)
		{
			case '\r':	//Caractre New Line
						
						ScrollFlag = (L == 1);	//Si c'est la fin de la ligne 1 on positionne le flag
						L++;					//on augmente la ligne
						LCD_gotoXY(0b11000000);	//on se positionne en dbut de ligne 1
						break;
			
			default:	//tous les autres caractres
						if (ScrollFlag) //si c'est un caractre ordinaire et que le flag a t mis
						{
							LCD_scroll();	//on scroll l'cran
							ScrollFlag = 0;	//on remet le flag  zro
							L=1;			//on se positionne sur la ligne 1
						}
						LCD_DataDir = 0xFF;	
						setbit(LCD_CmdPort,RS);		//RS=1 pour une donne
						clearbit(LCD_CmdPort,RW);	//RW=0 pour une criture
						_delay_loop_1(2);			//dlai trs court
						setbit(LCD_CmdPort,E);		//E  1	//spid					
						LCD_DataPort=data;			//Ecriture de la donne
						_delay_loop_1(2);			//dlai trs court
						clearbit(LCD_CmdPort,E);	//E repasse  0 
						_delay_us(50);				//dlai court
		}
	}


/********************************************************************************/
void LCD_gotoXY(byte Position) 
/********************************************************************************/
	{
		LCD_DataDir=0xFF;
		clearbit(LCD_CmdPort,RS);	//RS=0 pour une commande
		clearbit(LCD_CmdPort,RW);	//RW=0 pour une criture
		LCD_DataPort=Position;
		LCD_Epulse();
		_delay_us(40);
	}
	
	
/********************************************************************************/
void LCD_scroll()
/********************************************************************************/
	/* transfert de la ligne 1 dans un buffer, effacement du LCD,
		puis criture du buffer sur la ligne 0 et repositionnement
		du curseur au dbut de la ligne 1 */
	{
		byte x;
		LCD_gotoXY(0xC0);			//Positionnement en ligne 1
		setbit(LCD_CmdPort,RS);		//RS=1 pour une donne
		setbit(LCD_CmdPort,RW);		//RW=1 pour une lecture
		LCD_DataPort=0x00;
		LCD_DataDir=0x00;
		_delay_loop_1(2);			//dlai court
		for (x=0;x<16;x++)
		{
			setbit(LCD_CmdPort,E);
			_delay_loop_1(3);			//dlai trs court
			scrollbuf[x]=LCD_DataPin;	//transfert de la ligne 2 dans buffer
			_delay_loop_1(3);			//dlai trs court
			clearbit(LCD_CmdPort,E);
			_delay_us(50);				//dlai court
		}	
		//Efface l'cran et curseur home
		LCD_DataDir=0xFF;			//Port de donne en sortie
		clearbit(LCD_CmdPort,RS);	//RS=0 pour une commande
		clearbit(LCD_CmdPort,RW);	//RW=0 pour une criture		
		LCD_DataPort=0b00000001;	//Efface l'cran et curseur home
		LCD_Epulse();
		_delay_ms(2);				//dlai plus long
		
		//recopie le buffer sur la ligne 0
		LCD_DataDir=0xFF;			//Port de donne en sortie
		setbit(LCD_CmdPort,RS);		//RS=1 pour une donne
		clearbit(LCD_CmdPort,RW);	//RW=0 pour une criture
		for (x=0;x<16;x++)			
		{
			LCD_DataPort=scrollbuf[x];	//copie scrollbuf sur la ligne 0
			LCD_Epulse();
			_delay_ms(2);
		}
 
		LCD_gotoXY(0xC0);			//Positionnement en dbut de ligne 1
	}	
		
	
/********************************************************************************/
void LCD_init()
/********************************************************************************/
	{
		_delay_ms(60);				//Attente pour la stabilisation de la tension
		LCD_CmdDir = 0xFF;			//Port de commande en sortie
		LCD_DataDir=0xFF;			//Port de donne en sortie
		clearbit(LCD_CmdPort,RS);	//RS=0 pour une commande
		clearbit(LCD_CmdPort,RW);	//RW=0 pour une criture
		
		LCD_DataPort=0b00111100;	//8 bit - 2 lignes
		LCD_Epulse();
		_delay_us(160);
		
		LCD_DataPort=0b00001100;	//Affichage ON / curseur OFF / pas de blink
		LCD_Epulse();
		
		LCD_DataPort=0b00000001;	//Efface l'cran et curseur home
		LCD_Epulse();
				
		LCD_DataPort=0b00010100;	//Incrmente la position. Fentre fixe
		LCD_Epulse();
	}	
