/*********************************************************************
 *
 *                  Tick Manager for PIC18
 *                 (Modified to work with CCS PCH, by CCS)
 *
 *********************************************************************
 * FileName:        Tick.c
 * Dependencies:    stackTSK.h
 *                  Tick.h
 * Processor:       PIC18
 * Complier:        CCS PCH 3.181 or higher
 * Company:         Microchip Technology, Inc.
 *
 * Software License Agreement
 *
 * The software supplied herewith by Microchip Technology Incorporated
 * (the Company) for its PICmicro Microcontroller is intended and
 * supplied to you, the Companys customer, for use solely and
 * exclusively on Microchip PICmicro Microcontroller products. The
 * software is owned by the Company and/or its supplier, and is
 * protected under applicable copyright laws. All rights are reserved.
 * Any use in violation of the foregoing restrictions may subject the
 * user to criminal sanctions under applicable laws, as well as to
 * civil liability for the breach of the terms and conditions of this
 * license.
 *
 * THIS SOFTWARE IS PROVIDED IN AN AS IS CONDITION. NO WARRANTIES,
 * WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED
 * TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT,
 * IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR
 * CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 *
 * Author               Date     Comment
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * Nilesh Rajbharti     6/28/01  Original        (Rev 1.0)
 * Nilesh Rajbharti     2/9/02   Cleanup
 * Nilesh Rajbharti     5/22/02  Rev 2.0 (See version.log for detail)
 * Darren Rook (CCS)    01/09/04 Initial CCS Public Release
 * Darren Rook (CCS)    01/16/04 Intermediate counter vars added
 * R. Shelquist (CCS)   09/23/04 TickGet() temporarily disables Timer0 interrupt
 ********************************************************************/

#ifndef __TICK_C__
#define __TICK_C__

#define TICK_INCLUDE

#include "tcpip/stacktsk.h"
#include "tcpip/tick.h"


TICKTYPE TickCount = 0;  //increment every 100ms

#if defined(__PCH__)
 #define ResetTickTimer() set_timer0(TICK_COUNTER)
#else
 #define ResetTickTimer() set_timer1(TICK_COUNTER)
#endif

/*********************************************************************
 * Function:        void TickInit(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          Tick manager is initialized.
 *
 * Side Effects:    None
 *
 * Overview:        Initializes Timer0 as a tick counter.
 *
 * Note:            None
 ********************************************************************/
void TickInit(void)
{
 #if defined(__PCH__)
    setup_timer_0(RTCC_INTERNAL | RTCC_DIV_16);
 #elif defined(__PCD__)
   setup_timer1(TMR_INTERNAL | TMR_DIV_BY_64);
 #endif
   ResetTickTimer();

 #if !defined(__DISABLE_TICK_INTERRUPT__)
    enable_interrupts(__INT_TICK);
   #if defined(__PCD__)
    enable_interrupts(INTR_GLOBAL);
   #else
    enable_interrupts(GLOBAL);
   #endif
 #endif
}


/*********************************************************************
 * Function:        TICK TickGet(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          Current second value is given
 *
 * Side Effects:    None
 *
 * Overview:        None
 *
 * Note:            None
 ********************************************************************/
#if !defined(__DISABLE_TICK_INTERRUPT__)
TICKTYPE TickGet(void)
{
    TICKTYPE ret;
    disable_interrupts(__INT_TICK); 
    ret=TickCount;
    enable_interrupts(__INT_TICK);
    return ret;
}
#else
TICKTYPE TickGet(void)
{
    return(TickCount);
}
#endif
 
BYTE second_counter=0;  //increment this every 1s
BYTE second_counter_intermediate=0;

#if STACK_USE_PPP
 BYTE ppp_second_counter=0;
 BYTE ppp_second_counter_intermediate=0;
#endif

void _TickUpdate(void)
{
        TickCount++;    //increment this every 100ms

        second_counter_intermediate++;
        if (second_counter_intermediate >= TICKS_PER_SECOND) {
            second_counter++; //increment this ever 1s
            second_counter_intermediate=0;
            //fputc('.', STREAM_UI);
        }


       #if STACK_USE_PPP
        ppp_second_counter_intermediate++;
        if (ppp_second_counter_intermediate >= TICKS_PER_SECOND) {
            ppp_second_counter_intermediate=0;
            ppp_second_counter++;
        }
       #endif

    ResetTickTimer();
}

void TickUpdate(void)
{
   if (interrupt_active(__INT_TICK))
   {
      _TickUpdate();
      clear_interrupt(__INT_TICK);
   }
}

#if !defined(__DISABLE_TICK_INTERRUPT__)
#if (defined(INT_TIMER0) && (__INT_TICK==INT_TIMER0))
#int_timer0
#elif (__INT_TICK==INT_TIMER1)
#int_timer1
#else
 #error Define this interrupt
#endif
void Tick_Isr(void)
{
   _TickUpdate();
}
#endif

#endif
