/*********************************************************************
 *
 *                  Tick Manager for PIC18
 *                 (Modified to work with CCS PCH, by CCS)
 *
 *********************************************************************
 * FileName:        Tick.h
 * Dependencies:    None
 * Processor:       PIC18
 * Complier:        CCS PCH 3.181 or higher
 * Company:         Microchip Technology, Inc.
 *
 * Software License Agreement
 *
 * The software supplied herewith by Microchip Technology Incorporated
 * (the Company) for its PICmicro Microcontroller is intended and
 * supplied to you, the Companys customer, for use solely and
 * exclusively on Microchip PICmicro Microcontroller products. The
 * software is owned by the Company and/or its supplier, and is
 * protected under applicable copyright laws. All rights are reserved.
 * Any use in violation of the foregoing restrictions may subject the
 * user to criminal sanctions under applicable laws, as well as to
 * civil liability for the breach of the terms and conditions of this
 * license.
 *
 * THIS SOFTWARE IS PROVIDED IN AN AS IS CONDITION. NO WARRANTIES,
 * WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED
 * TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT,
 * IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR
 * CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 *
 * Author               Date     Comment
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * Nilesh Rajbharti     6/28/01  Original        (Rev 1.0)
 * Nilesh Rajbharti     2/9/02   Cleanup
 * Nilesh Rajbharti     5/22/02  Rev 2.0 (See version.log for detail)
 * Darren Rook (CCS)    01/09/04 Initial CCS Public Release
 * R. Shelquist (CCS)   09/24/04 TickGetDiff fixed so it isn't off by 1 if a<b
 ********************************************************************/

#ifndef TICK_H
#define TICK_H

#if defined(__PCH__)
 #define __INT_TICK  INT_TIMER0
#elif defined(__PCD__)
 #define __INT_TICK  INT_TIMER1
#endif

typedef unsigned int16 TICKTYPE;

/*
 * This value is used by TCP to implement timeout actions.
 */
#define TICKS_PER_SECOND               10

#if (TICKS_PER_SECOND < 10 || TICKS_PER_SECOND > 255)
#error Invalid TICKS_PER_SECONDS specified.
#endif

//16 is the prescalar for PCH
//64 is the prescalar for PCD
#if defined(__PCH__)
 #define CYCLES_PER_TICKS_PER_SECOND getenv("CLOCK")/(4 * 16 * TICKS_PER_SECOND)
#elif defined(__PCD__)
 #define CYCLES_PER_TICKS_PER_SECOND getenv("CLOCK")/(4 * 64 * TICKS_PER_SECOND)
#endif

#if (CYCLES_PER_TICKS_PER_SECOND > 0xFFFF)
 #error Reduce Clock Speed or Increase TICKS_PER_SECOND
#endif

#if (CYCLES_PER_TICKS_PER_SECOND <= 100)
 #error Timer0 not accurate enough.  Raise Clock Speed or Reduce TICKS_PER_SECOND
#endif


#define TICK_COUNTER 0xFFFF-CYCLES_PER_TICKS_PER_SECOND

/*
#if (getenv("CLOCK")==20000000)
   #define TICK_COUNTER 34250
#elif (getenv("CLOCK")==40000000)
   #define TICK_COUNTER 3035
#elif (getenv("CLOCK")==19600000)
   #define TICK_COUNTER 34910
#else
 #error CALCULATE THIS CLOCK SPEED (INCREMENT TICKCOUNT EVERY 100MS).  SEE TickInit()
#endif
*/

#define TICK_SECOND  TICKS_PER_SECOND

#if defined(__TICKGET_IS_MACRO__)
#define TickGetDiff(a, b)       ((TICKTYPE)(a < b) ? (((TICKTYPE)0xffff - b) + a + 1) : (a - b))
#else
TICKTYPE TickGetDiff(TICKTYPE a, TICKTYPE b)
{
   if (a < b)
      return(((TICKTYPE)0xffff - b) + a + 1);
   else
      return(a - b);
}
#endif


/*********************************************************************
 * Function:        void TickInit(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          Tick manager is initialized.
 *
 * Side Effects:    None
 *
 * Overview:        Initializes Timer0 as a tick counter.
 *
 * Note:            None
 ********************************************************************/
void TickInit(void);



/*********************************************************************
 * Function:        TICK TickGet(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          Current second value is given
 *
 * Side Effects:    None
 *
 * Overview:        None
 *
 * Note:            None
 ********************************************************************/
TICKTYPE TickGet(void);


/*********************************************************************
 * Function:        void TickUpdate(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Internal Tick and Seconds count are updated.
 *
 * Note:            None
 *
 * CCS NOTE:        Normally the CCS version of the stack will handle updating
 *                   the tick in the Timer interrupt - you only need
 *                   to call TickUpdate if you have disabled the interrupt.
 ********************************************************************/
void TickUpdate(void);



#endif
