#include <xc.h>
#include "RTDM.h"

typedef signed int SFRAC16;

#define FCY             39613750
#define FPWM            19531		// 19.5 kHz, so that no audible noise is present.
#define _10MILLISEC	10
#define _50MILLISEC	50
#define _100MILLISEC	100
#define _500MILLISEC	500

#define HALLA	0x01	// RB3
#define HALLB	0x02	// RB4
#define HALLC	0x04	// RB5
#define CW	0	// Actual Hurst Motor speed is CCW
#define CCW	1	// Actual Hurst Motor speed is CW
#define Switch_2	(!PORTAbits.RA8)

#define RPM             3000       // 3000 RPM at 24V dc (6000 RPM at 48v dc)
#define POLEPAIRS       10
#define PERIOD_SCALE    (FCY/(POLEPAIRS*64))*60        //7427578

#define T1PR1           ((FCY/1000)/64)                 // 1ms

#define MAX_DUTY_CYCLE  (FCY/FPWM - 1)*2                //4054
#define MIN_DUTY_CYCLE  (MAX_DUTY_CYCLE/33)	 	// 3% duty cycle P1DC1=P1DC2=P1DC3 = 72

#define MINPERIOD	(int)(((FCY/RPM)*60)/(POLEPAIRS*64))  // For 6000 max rpm and 10 poles motor
#define MAXPERIOD	32767 //MINPERIOD*33 //31250	// For 60 min rpm and 10 poles motor
#define MINABSSPEED	MINPERIOD
#define MAXABSSPEED     MAXPERIOD

// Use this MACRO when using floats to initialize signed 16-bit fractional variables
#define SFloat_To_SFrac16(Float_Value)	(((Float_Value) < 0.0) ? (SFRAC16)(32768 * (Float_Value) - 0.5) : (SFRAC16)(32767 * (Float_Value) + 0.5))

void SixStepComm(int _Current_Sector, int _Required_Voltage);	// This subroutine is used for commutating the motor depending on its rotor position
void InitADC10(void);			// Initialization of ADC used for Speed Command
void InitMCPWM(void);			// Initialization for PWM at 20kHz, Center aligned, Complementary mode with 500 ns of deadtime
void InitTMR1(void);			// Initialization for TIMER1 used for speed control and motor stalled protection
void InitTMR3(void);			// Initialization for TIMER3 used as a timebase for the two input capture channels
void InitUserInt(void);			// This function initializes all ports (inputs and outputs) for the application
void Init_IC_and_CN(void);		// Initializes input captures and change notification, used for the hall sensor inputs
void RunMotor(void);			// This function initializes all variables and interrupts used for starting and running the motor
void StopMotor(void);			// This function clears all flags, and stops anything related to motor control, and also disables PWMs
void ForceCommutation(void);	// When motor is to slow to generate interrupts on halls, this function forces a commutation
void ChargeBootstraps(void);	// At the begining of the motor operation, the bootstrap caps are charged with this function
extern void SpeedControl(void);	// This function contains all ASM and C operations for doing the PID Control loop for the speed
extern void SpeedCalculation(void);

// Flags used for the application
struct 
{
	unsigned MotorRunning	:1;
	unsigned MotorCommInAdv :1;
	unsigned MotorAdvEnabled:1;
	unsigned unused			:13;
}Flags;

unsigned int CurrentRPM;
unsigned int HallValue;					// This variable holds the hall sensor input readings
unsigned int Sector;					// This variables holds present sector value. Which is the rotor position
unsigned int LastSector;				// This variable holds the last sector value. This is critical to filter halls slew rate HW issue
unsigned int MotorStalledCounter = 0;	// This variable gets incremented each 1 ms, and is cleared everytime a new sector is
										// detected. Used for ForceCommutation and MotorStalled protection functions

// 	Motor Terminal		|	MCLV Connection
// ---------------------|-------------------
//	White Phase --------|-- M3
//	Black Phase --------|-- M2
//	Red Phase   --------|-- M1
//  Green Wire  --------|--	G
//  Hall Supply --------|-- +5V
//  Hall Ground --------|-- GND
//	White Hall  --------|-- HA
//	Brown Hall  --------|-- HB
//	Green Hall  --------|-- HC

// This array translates the hall state value read from the digital
// I/O to the proper commutation sector.  Hall values of 0 or 7 
// represent illegal values and therore return -1.
// Motor Connection for this Sector Table:

char SectorTable[] = {-1,4,2,3,0,5,1,-1};

/*************************************************************
	Low side driver table is as below.  The high side driver is
	PWM while the low side driver is either on or off.
*************************************************************/

unsigned int StateLoTable[] = {	0x0204,  /* PWM2L -> 1, PWM1H -> PWM */
							 	0x0210,  /* PWM3L -> 1, PWM1H -> PWM */
                                0x0810,  /* PWM3L -> 1, PWM2H -> PWM */
								0x0801,  /* PWM1L -> 1, PWM2H -> PWM */
								0x2001,  /* PWM1L -> 1, PWM3H -> PWM */
								0x2004}; /* PWM2L -> 1, PWM3H -> PWM */

/* Speed Control Variables */
unsigned int POT;
unsigned char Current_Direction;	// Current mechanical motor direction of rotation Calculated in halls interrupts
unsigned char Required_Direction;	// Required mechanical motor direction of rotation, will have the same sign as the
									// ControlOutput variable from the Speed Controller
unsigned int PastCapture, ActualCapture, Period;	// Variables containing the Period of half an electrical cycle, which is
													// an interrupt each edge of one of the hall sensor input
// Absolute PID gains used by the controller. Position form implementation of a digital PID, see Ogata Page 205
SFRAC16 Kp = SFloat_To_SFrac16(0.10000);
SFRAC16 Ki = SFloat_To_SFrac16(0.01000);
SFRAC16 Kd = SFloat_To_SFrac16(0.00000);
SFRAC16 Speed, RefSpeed;	// Actual and Desired speeds for the PID controller, that will generate the error
SFRAC16 SpeedAbsValue;		// Absolute value of the Speed variable. Used by the phase advance subroutine
SFRAC16 ControlOutput = 0;	// Controller output, used as a voltage output, use its sign for the required direction
// Constants used by the PID controller, since a MAC operation is used, the PID structure is changed (See SpeedControl() Comments
SFRAC16 ControlDifference[3] __attribute__((__space__(xmemory), __aligned__(4)));	// Errors in x memory for MAC instruction
SFRAC16 PIDCoefficients[3] __attribute__((__space__(ymemory), __aligned__(4)));		// PID modified coefficients in y memory for MAC instruction

// Used as a temporal variable to perform a fractional divide operation in assembly
SFRAC16 _MINPERIOD = MINPERIOD - 1;	// Used by fractional divide operation

/********************* Variables to display data using DMCI *********************************/
#define DATA_BUFFER_SIZE 128  //Size in 16-bit Words
unsigned int __attribute__((aligned(DATA_BUFFER_SIZE)))RecorderBuffer1[DATA_BUFFER_SIZE]; 	//Buffer to store the data samples for the DMCI data viewer Graph1
unsigned int __attribute__((aligned(DATA_BUFFER_SIZE)))RecorderBuffer2[DATA_BUFFER_SIZE];	//Buffer to store the data samples for the DMCI data viewer Graph2
unsigned int __attribute__((aligned(DATA_BUFFER_SIZE)))RecorderBuffer3[DATA_BUFFER_SIZE];	//Buffer to store the data samples for the DMCI data viewer Graph3
int __attribute__((aligned(DATA_BUFFER_SIZE)))RecorderBuffer4[DATA_BUFFER_SIZE];			//Buffer to store the data samples for the DMCI data viewer Graph4

unsigned int * PtrRecBuffer1 = &RecorderBuffer1[0];	//Tail pointer for the DMCI Graph1
unsigned int * PtrRecBuffer2 = &RecorderBuffer2[0];	//Tail pointer for the DMCI Graph2
unsigned int * PtrRecBuffer3 = &RecorderBuffer3[0];	//Tail pointer for the DMCI Graph3
int * PtrRecBuffer4 = &RecorderBuffer4[0];	//Tail pointer for the DMCI Graph4
int * RecBuffUpperLimit = RecorderBuffer4 + DATA_BUFFER_SIZE -1;	//Buffer Recorder Upper Limit

typedef struct DMCIFlags{
			unsigned StartStop : 1;
		    unsigned Recorder : 1;
			unsigned unused :	14;  
} DMCIFLAGS;

DMCIFLAGS DMCIFlags;

typedef struct HallSensorsStates{
			unsigned A : 1;
		    unsigned B : 1;
			unsigned C : 1;
			unsigned unused :	13;  
} HALLSENSORS;

HALLSENSORS Hall;
unsigned int I1,I2;
int Ibus;
SFRAC16 DMCIRefSpeed;
//unsigned int DMCIRefSpeed;


/*********************************************************************
	main function of the application. Peripherals are initialized,
	and then, depending on the motor status (running or stopped) and
	if the push button is pressed, the motor is started or stopped.
	All other operations and state machines are performed with
	interrupts.
*********************************************************************/
int main(void)
{
	// Configure Oscillator to operate the device at 30Mhz
	// Fosc= Fin*M/(N1*N2), Fcy=Fosc/2
	// Fosc= 7.3728*43/(2*2)= 79.25Mhz for 7.3728 input clock
	PLLFBD=41;					// M=43
	CLKDIVbits.PLLPOST=0;		// N1=2
	CLKDIVbits.PLLPRE=0;		// N2=2
	OSCTUN=0;					// Tune FRC oscillator, if FRC is used

	// Disable Watch Dog Timer
	RCONbits.SWDTEN=0;

	// Clock switch to incorporate PLL
	__builtin_write_OSCCONH(0x01);		// Initiate Clock Switch to
						// FRC with PLL (NOSC=0b001)
	__builtin_write_OSCCONL(0x01);		// Start clock switching
	while (OSCCONbits.COSC != 0b001);	// Wait for Clock switch to occur

	// Wait for PLL to lock
	while(OSCCONbits.LOCK!=1) {};
	
	//Flags.MotorRunning = 0;	// Indicate that the motor has been stopped

	InitADC10();		// Initialize ADC to be signed fractional
	InitTMR1();			// Initialize TMR1 for 1 ms periodic ISR
	InitTMR3();			// Initialize TMR3 for timebase of capture
	Init_IC_and_CN();	// Initialize Hall sensor inputs ISRs	
	InitMCPWM();		// Initialize PWM @ 20 kHz, center aligned, 500 ns of deadtime
	InitUserInt();		// Initialize User Interface I/Os
    RTDM_Start();  		// Configure the UART module used by RTDM
						// it also initializes the RTDM variables
        DMCIRefSpeed = 200;

	for(;;){
	
 	while((!DMCIFlags.StartStop) && (!Flags.MotorRunning)){RTDM_ProcessMsgs();}
	RunMotor();	// Run motor if push button is pressed and motor is stopped

    while((DMCIFlags.StartStop) && (Flags.MotorRunning)){
        RTDM_ProcessMsgs();
        Ibus = ((int)((I1+I2)-512)/10); 
        }
	StopMotor();	// Stop motor if push button is pressed and motor is running
  
	}
	return 0;
}


/**********************************************************************
	Timer 1 interrupt, In this ISR the Period and Speed are calculated 
    based on the input capture of one of the halls.
	The speed controller is also called in this ISR to generate a new
	output voltage (ControlOutput). The last thing done in this ISR is
	the forced commutation, which happens each time the motor doesn't
	generate a new hall interrupt after a programmed period of time.
	If the timeout for generating hall ISR is too much (i.e. 100 ms)
	the motor is then stopped.
	Note: The Speed Calculation is made in assembly to take advantage of
	the signed fractional division.
**********************************************************************/
void __attribute__((__interrupt__,auto_psv)) _T1Interrupt (void)
{
	IFS0bits.T1IF = 0;
	Period = ActualCapture - PastCapture;	// This is an UNsigned substraction

	if (Period < (unsigned int)MINPERIOD) 	// MINPERIOD corresponds to 6000 rpm
		Period = MINPERIOD;
	else if (Period > (unsigned int)MAXPERIOD) // MAXPERIOD corresponds to 60 rpm
		Period = MAXPERIOD;

	// This subroutine in assembly calculates the Speed using fractional division
	//	                              		MINPERIOD
	// Speed = (Fractional divide) -----------------------
	//                                 			Period

	SpeedCalculation();
	SpeedAbsValue = Speed;
        CurrentRPM = PERIOD_SCALE / Period;
	// Speed sign adjustment based on current motor direction of rotation
	if (Current_Direction == CCW)
		Speed = -Speed;

	// Condition        RPM          SFRAC16      SINT      HEX
	// Max Speed CW  -> 6000 RPM  -> 0.996805  -> 32663  -> 0x7F97
	// Min Speed CW  -> 60 RPM    -> 0.009984  -> 327    -> 0x0147
	// Min Speed CCW -> -60 RPM   -> -0.009984 -> -327   -> 0xFEB9
	// Max Speed CCW -> -6000 RPM -> -0.996805 -> -32663 -> 0x8069

	// Speed PID controller is called here. It will use Speed, RefSpeed, Some buffers
	// and will generate the new voltage for the SVM.
	PIDCoefficients[0] = Kp + Ki + Kd;	// Modified coefficient for using MACs
	PIDCoefficients[1] = -(Kp + 2*Kd);	// Modified coefficient for using MACs
	PIDCoefficients[2] = Kd;			// Modified coefficient for using MACs
	SpeedControl();
//        ControlOutput = DMCIRefSpeed;
	// Set minimum ControlOutput to 10% to keep the motor running

	if (ControlOutput < 70) //MIN_DUTY_CYCLE)
		ControlOutput = 70;//MIN_DUTY_CYCLE;

       	if (ControlOutput > 4056 )//MAX_DUTY_CYCLE)
		ControlOutput = 4056;//MAX_DUTY_CYCLE;

	MotorStalledCounter++;	// This variable is cleared in halls ISRs
    if ((MotorStalledCounter % _10MILLISEC) == 0)
	{
		ForceCommutation();	// Force Commutation if no hall sensor changes have occured in specified timeout.
	}
	else if (MotorStalledCounter >= _100MILLISEC)
	{
		StopMotor(); // Stop motor is no hall changes have occured in specified timeout
	}
	return;
}

/**********************************************************************
	Hall A interrupt, this is done with the input change notification CN5.
	The purpose of this ISR is to Calculate the actual mechanical
	direction of rotation of the motor, and to adjust the Phase variable
	depending on the sector the rotor is in.
	Note: The sector must be validated in order to avoid any spurious
	interrupt due to a slow slew rate on the halls inputs due to hardware
	filtering.
**********************************************************************/
void __attribute__((__interrupt__,auto_psv)) _CNInterrupt (void)
{
	IFS1bits.CNIF = 0;	// Clear interrupt flag
	HallValue = (unsigned int)((PORTB >> 1) & 0x0007);	// Read halls
	Hall.A = PORTBbits.RB1;
	Sector = SectorTable[HallValue];	// Get Sector from table
	if (Sector != LastSector)	// This is a MUST for getting around the HW slow rate
	{
		// Since a new sector is detected, clear variable that would stop the motor if stalled.
		MotorStalledCounter = 0;
		// Motor current direction is computed based on Sector
		if ((Sector == 5) || (Sector == 2))	// If going forward for falling and rising then CW
			Current_Direction = CCW;
		else
			Current_Direction = CW;
	
		SixStepComm(Sector, ControlOutput); // If ControlOutput > 0 will run CW, else will run CCW

		LastSector = Sector;	// If a change in sector, update last sector
	}
	return;
}

/**********************************************************************
	Hall B interrupt, this is done with the input change notification IC7.
	The purpose of this ISR is to Calculate the actual mechanical
	direction of rotation of the motor, and to adjust the Phase variable
	depending on the sector the rotor is in AND ALSO to calculate the
	mechanical Period between hall transition on the same input using
	input capture read.
	Note: The sector must be validated in order to avoid any spurious
	interrupt due to a slow slew rate on the halls inputs due to hardware
	filtering.
**********************************************************************/
void __attribute__((__interrupt__,auto_psv)) _IC7Interrupt (void)
{
	IFS1bits.IC7IF = 0;	// Cleat interrupt flag
	HallValue = (unsigned int)((PORTB >> 1) & 0x0007);	// Read halls
	Hall.B = PORTBbits.RB2;
	Sector = SectorTable[HallValue];	// Get Sector from table
	if (Sector != LastSector)	// This is a MUST for getting around the HW slow rate
	{
		// Calculate Hall period corresponding to half an electrical cycle
		PastCapture = ActualCapture;
		ActualCapture = IC7BUF;
		// Clear all buffers
		IC7BUF;
		IC7BUF;
		IC7BUF;
		// Since a new sector is detected, clear variable that would stop the motor if stalled.
		MotorStalledCounter = 0;
		// Motor current direction is computed based on Sector
		if ((Sector == 3) || (Sector == 0))	// If going forward for falling and rising then CW
			Current_Direction = CCW;
		else
			Current_Direction = CW;

		SixStepComm(Sector, ControlOutput); // If ControlOutput > 0 will run CW, else will run CCW

		LastSector = Sector;	// If a change in sector, update last sector
	}
	return;
}

/**********************************************************************
	Hall C interrupt, this is done with the input change notification IC8.
	The purpose of this ISR is to Calculate the actual mechanical
	direction of rotation of the motor, and to adjust the Phase variable
	depending on the sector the rotor is in.
	Note: The sector must be validated in order to avoid any spurious
	interrupt due to a slow slew rate on the halls inputs due to hardware
	filtering.
**********************************************************************/
void __attribute__((__interrupt__,auto_psv)) _IC8Interrupt (void)
{	
	IFS1bits.IC8IF = 0;	// Cleat interrupt flag
	HallValue = (unsigned int)((PORTB >> 1) & 0x0007);	// Read halls
	Hall.C =PORTBbits.RB3;
	Sector = SectorTable[HallValue];	// Get Sector from table	
	if (Sector != LastSector)	// This is a MUST for getting around the HW slow rate
	{
		// Since a new sector is detected, clear variable that would stop the motor if stalled.
		MotorStalledCounter = 0;
		// Motor current direction is computed based on Sector
		if ((Sector == 1) || (Sector == 4))	// If going forward for falling and rising then CW
			Current_Direction = CCW;
		else
			Current_Direction = CW;
		
		SixStepComm(Sector, ControlOutput); // If ControlOutput > 0 will run CW, else will run CCW

		LastSector = Sector;	// If a change in sector, update last sector
	}
	return;
}

/*********************************************************************
	The ADC interrupt loads the reference speed (RefSpeed) with the 
	respective value of the POT. The value will be a unsigned fractional 
	value.
*********************************************************************/
void __attribute__((__interrupt__,auto_psv)) _ADC1Interrupt (void)
{

	/********************* DMCI Dynamic Data Views  *********************************/
	IFS0bits.AD1IF = 0;
	POT = ADC1BUF0;		//ADC CH0 holds the I3 value
	I1 = ADC1BUF1>>6;	//ADC CH1 holds the I1 value	
	I2 = ADC1BUF2>>6;	//ADC CH2 holds the I2 value

	/********************** RECORDING MOTOR PHASE VALUES & IBUS****************/
	if(DMCIFlags.Recorder){
	*PtrRecBuffer1++ 	= ControlOutput;//Hall.A;
	*PtrRecBuffer2++	= CurrentRPM;//Hall.B;
	*PtrRecBuffer3++	= Period; //Hall.C;
	*PtrRecBuffer4++	= Ibus-50;
	
	if(PtrRecBuffer4 > RecBuffUpperLimit){
		PtrRecBuffer1 = RecorderBuffer1;
		PtrRecBuffer2 = RecorderBuffer2;
        PtrRecBuffer3 = RecorderBuffer3;
        PtrRecBuffer4 = RecorderBuffer4;
        DMCIFlags.Recorder = 0;
        }   
	}

	RefSpeed = (DMCIRefSpeed);//*10);

	return;
}


/*********************************************************************
	In the topology used, it is necessary to charge the bootstrap
	caps each time the motor is energized for the first time after an
	undetermined amount of time. ChargeBootstraps subroutine turns ON
	the lower transistors for 10 ms to ensure voltage on these caps,
	and then it transfers the control of the outputs to the PWM
	module.
*********************************************************************/
void ChargeBootstraps(void)
{
	unsigned int i;
	P1OVDCON = 0x0015;	// Turn ON low side transistors to charge
	for (i = 0; i < 33330; i++) // 10 ms Delay at 20 MIPs
		;
	P1OVDCON = 0x0000;
	return;
}

/*********************************************************************
	Call this subroutine when first trying to run the motor and the
	motor is previously stopped. RunMotor will charge bootstrap caps,
	will initialize application variables, and will enable all ISRs.
*********************************************************************/
void RunMotor(void)
{
	ChargeBootstraps();
	// init variables
	ControlDifference[0] = 0;	// Error at K	(most recent)
	ControlDifference[1] = 0;	// Error at K-1
	ControlDifference[2] = 0;	// Error at K-2	(least recent)
	PIDCoefficients[0] = Kp + Ki + Kd;	// Modified coefficient for using MACs
	PIDCoefficients[1] = -(Kp + 2*Kd);	// Modified coefficient for using MACs
	PIDCoefficients[2] = Kd;			// Modified coefficient for using MACs
	
	TMR1 = 0;			// Reset timer 1 for speed control
	TMR3 = 0;			// Reset timer 3 for speed measurement
	ActualCapture = MAXPERIOD; // Initialize captures for minimum speed (60 RPMs)
	PastCapture = 0;
	Period = MAXPERIOD;
	while (IC7CONbits.ICBNE)
		IC7BUF;

	// Initialize direction with required direction
	// Remember that ADC is not stopped.
	HallValue = (unsigned int)((PORTB >> 1) & 0x0007);	// Read halls
	LastSector = Sector = SectorTable[HallValue];	// Initialize Sector variable

	// RefSpeed's sign will determine if the motor should be run at CW (+RefSpeed) or CCW (-RefSpeed) ONLY
	// at start up, since when the motor has started, the required direction will be set by the
	// control output variable to be able to operate in the four quadrants
	if (RefSpeed > 0)
	{
		ControlOutput = 0;
		Current_Direction = Required_Direction = CW;
		Speed = MINABSSPEED;
	}
	else
	{
		ControlOutput = 0;
		Current_Direction = Required_Direction = CCW;
		Speed = -MINABSSPEED;
	}
	SpeedAbsValue = MINABSSPEED;

	MotorStalledCounter = 0;	// Reset motor stalled protection counter

	// Clear all interrupts flags
	IFS0bits.T1IF = 0;	// Clear timer 1 flag
	IFS1bits.CNIF = 0;	// Clear interrupt flag
	IFS1bits.IC7IF = 0;	// Clear interrupt flag
	IFS1bits.IC8IF = 0;	// Clear interrupt flag

	// enable all interrupts
	__asm__ volatile("DISI #5");
	IEC0bits.T1IE = 1;	// Enable interrupts for timer 1
	IEC1bits.CNIE = 1;	// Enable interrupts on CN5
	IEC1bits.IC7IE = 1;	// Enable interrupts on IC7
	IEC1bits.IC8IE = 1;	// Enable interrupts on IC8

	Flags.MotorRunning = 1;	// Indicate that the motor is running
	return;
}

/*********************************************************************
	Call this subroutine whenever the user want to stop the motor.
	This will clear interrupts properly, and will also turn OFF all
	PWM channels.
*********************************************************************/

void StopMotor(void)
{
	P1OVDCON = 0x0000;	// turn OFF every transistor
	
	// disable all interrupts
	__asm__ volatile("DISI #5");
	IEC0bits.T1IE = 0;	// Disable interrupts for timer 1
	IEC1bits.CNIE = 0;	// Disable interrupts on CN5
	IEC1bits.IC7IE = 0;	// Disable interrupts on IC7
	IEC1bits.IC8IE = 0;	// Disable interrupts on IC8

	Flags.MotorRunning = 0;	// Indicate that the motor has been stopped
	return;
}

/********************************************************************
	This subroutine implements the six-step commutation of a BLDC motor.
	Depending on the Sector the motor is in, the corresponding pair of
	transistors is energized based on a look up table. The duty cycle
	of the transistor with PWM is taken from the second parameter.
*********************************************************************/
void SixStepComm (int _Sector, int _Voltage)
{

	if (_Voltage >= 0)
	{
		// Load the same duty cycle to all the PDC registers.
		// The appropriate pair of transistors will be selected using
		// the overide control logic of the MCPWM module (P1OVDCON)
		PDC1 = PDC2 = PDC3 = (unsigned int)_Voltage; // / 16;
		if (_Sector == -1)
			P1OVDCON = 0x0000; // Invalid sector will shut down all transistors
		else
			P1OVDCON = StateLoTable[_Sector];
	}
	else
	{
		// Load the same duty cycle to all the PDC registers.
		// The appropriate pair of transistors will be selected using
		// the overide control logic of the MCPWM module (P1OVDCON)
		PDC1 = PDC2 = PDC3 = (unsigned int)(-(_Voltage+1)) / 16;
		if (_Sector == -1)
			P1OVDCON = 0x0000; // Invalid sector will shut down all transistors
		else
			P1OVDCON = StateLoTable[(_Sector + 3) % 6];
	}
	return;
}

/*********************************************************************
	This function is called each time the motor doesn't generate
	hall change interrupt, which means that the motor running too slow
	or is stalled. If it is stalled, the motor is stopped, but if it
	is only slow, this function is called and forces a commutation
	based on the actual hall sensor position and the required
	direction of rotation.
*********************************************************************/
void ForceCommutation(void)
{
	HallValue = (unsigned int)((PORTB >> 1) & 0x0007);	// Read halls
	Sector = SectorTable[HallValue];	// Read sector based on halls
	if (Sector != -1)	// If the sector is invalid don't do anything
	{
		// Depending on the required direction, a new phase is fetched
		if (Required_Direction == CW)
		{
			SixStepComm(Sector, ControlOutput);
		}
		else
		{
			SixStepComm(Sector, -(ControlOutput + 1));
		}
	}
	return;
}

/*******************************************************************
	Below is the code required to setup the ADC registers for :
	1. 1 channel conversion (in this case RB2/AN2)
	2. PWM trigger starts conversion
	3. Pot is connected to CH0 and RB2
	4. Manual Stop Sampling and start converting
	5. Manual check of Conversion complete 
	6. The data format will be signed fractional			
*********************************************************************/
void InitADC10(void)
{
	AD1PCFGLbits.PCFG0 = 0; // ensure AN0 is analog  -I1
	AD1PCFGLbits.PCFG1 = 0; // ensure AN1 is analog  - I2
	AD1PCFGLbits.PCFG3 = 1; // ensure AN3is digital  - HALL A
	AD1PCFGLbits.PCFG4 = 1; // ensure AN4 is digital - HALL B
	AD1PCFGLbits.PCFG5 = 1; // ensure AN5 is digital - HALL C
	AD1PCFGLbits.PCFG8 = 0; // ensure AN8 is analog  - POT  AN8/RP18 
	
	AD1CON1 = 0x026C;   //ADC is off
						// Continue module operation in Idle mode
						// 10-bit, 4-channel ADC operation
						// Data Output Format bits Fractional (dddd dddd dd00 0000)
						// 0110 = PWM ends sampling and starts conversion
						// Samples CH0, CH1, CH2, CH3 simultaneously when CHPS<1:0> = 1x
						// Sampling begins immediately after last conversion SAMP bit is auto-set.

	AD1CHS123 = 0x0000; //MUX B CH1, CH2, CH3 negative input is VREF-
						// MUX B CH1 positive input is AN0, CH2 positive input is AN1, CH3 positive input is AN2						
						// MUX A CH1, CH2, CH3 negative input is VREF-
						// MUX A CH1 positive input is AN0, CH2 positive input is AN1, CH3 positive input is AN2

	AD1CHS0 = 0x0008;   //MUX B Channel 0 negative input is VREF-
						// MUX B Channel 0 positive input is AN0
						// MUX A Channel 0 negative input is VREF-
						// MUX A Channel 0 positive input is AN8

	AD1CSSL = 0x0000;	//Skip all ANx channels for input scan

	AD1CON3 = 0x0002;	//ADC Clock derived from system clock
						// Autosample time time bits = 0 TAD sinjce PWM is controlling sampling time
						// TAD = 3*TCY, TAD = 101.7 nSec

	AD1CON2 = 0x0300;	//ADREF+ = AVDD ADREF- = AVSS
						// Do not scan inputs
						// 1x = Converts CH0, CH1, CH2 and CH3
						// A/D is currently filling buffer 0x0-0x7
						// Interrupts at the completion of conversion for each sample/convert sequence
						// Always starts filling buffer from the beginning
						// Always uses channel input selects for Sample A

	_DONE = 0;			//Making sure that there is any conversion in progress
	_AD1IP = 3;			//Assigning ADC ISR priority
	_AD1IF = 0;			//Clearing the ADC Interrupt Flag
	_AD1IE = 1;			//Enabling the ADC conversion complete interrupt
	_ADON = 1;			//Enabling the ADC module 

	return;
}


/********************************************************************
	InitMCPWM, intializes the PWM as follows
	1. FPWM = 20000 hz
	2. Independent PWMs with Edge aligned
	3. Set Duty Cycle to 0 for Independent
	4. Set ADC to be triggered by PWM special trigger
*********************************************************************/
void InitMCPWM(void)
{
	TRISB&=0x3FF;			// PWM pins (RB10~15) as outputs,
	RPINR12bits.FLTA1R= 8;	// remap FLTA1 to RP8,  
	TRISBbits.TRISB8=1; 	// SCL1/RP8/CN22/RB8 as input
	P1TPER = FCY/FPWM - 1;	// Compute Period based on CPU speed and required PWM frequency (see defines)
	P1OVDCON = 0x0000;		// Disable all PWM outputs.
	PWM1CON1 = 0x0777;		// Enable PWM output pins and configure them as complementary mode	 
	P1DC1 = 0;				// Initialize as 0 voltage
	P1DC2 = 0;				// Initialize as 0 voltage
	P1DC3 = 0;				// Initialize as 0 voltage
	SEVTCMP = P1TPER;		// Enable triggering for ADC
	PWM1CON2 = 0x0F00;		// 16 postscale values, for achieving 20 kHz
	P1TCON = 0x8000;		// start PWM as edge aligned mode
	return;				 
}


/********************************************************************
	Configure Hall sensor inputs, one change notification and two
	input captures. on IC7 the actual capture value is used for
	further period calculation
*********************************************************************/
void Init_IC_and_CN(void)
{
	//Hall A -> CN5. Hall A is only used for commutation.
	//Hall B -> IC7. Hall B is used for Speed measurement and commutation.
	//Hall C -> IC8. Hall C is only used for commutation.
	
	AD1PCFGLbits.PCFG3 = 1;          // ensure AN3/RP1/CN5/RB1 is digital
	AD1PCFGLbits.PCFG4 = 1;          // ensure AN4/RP2/CN6/RB2 is digital
	AD1PCFGLbits.PCFG5 = 1;          // ensure AN5/RP3/CN7/RB3 is digital
	TRISB |= 0xE;					// Ensure that hall connections are inputs

	// Init Input change notification 5
	CNPU1 = 0;				// Disable all CN pull ups
	CNEN1bits.CN5IE = 1;	// Enable CN5 (Hall A)
	IFS1bits.CNIF = 0;		// Clear interrupt flag

	// Init Input Capture 7
	RPINR10bits.IC7R=2;		// remap IC7 input to AN4/RP2/CN6/RB2
	IC7CON = 0x0001;		// every edge interrupts, TMR3
	IFS1bits.IC7IF = 0;		// Clear interrupt flag

	// Init Input Capture 8
	RPINR10bits.IC8R=3; 	// remap IC8 input to AN5/RP3/CN7/RB3
	IC8CON = 0x0001;		// every edge interrupts, TMR3
	IFS1bits.IC8IF = 0;		// Clear interrupt flag

	return;
}

/********************************************************************
	Initialization of timer 1 as a periodic interrupt each 1 ms for
	speed control, motor stalled protection which includes: forced
	commutation if the motor is too slow, or motor stopped if the
	motor is stalled.
*********************************************************************/
void InitTMR1(void)
{
	T1CON = 0x0020;			// internal Tcy/64 clock
	TMR1 = 0;
	PR1 = T1PR1;			// 1 ms interrupts for 40 MIPS
	T1CONbits.TON = 1;		// turn on timer 1 
	return;
}

/************************************************************************
	Initialization of timer 3 as the timebase for the capture channels
	for calculating the period of the halls.
*************************************************************************/
void InitTMR3(void)
{
	T3CON = 0x0020;			// internal Tcy/64 clock
	TMR3 = 0;
	PR3 = 0xFFFF;
	T3CONbits.TON = 1;		// turn on timer 3 
	return;
}

/************************************************************************
	Initialization of the IOs used by the application. The IOs for the
	PWM and for the ADC are initialized in their respective peripheral
	initialization subroutine.
*************************************************************************/
void InitUserInt(void)
{
	_TRISA8 = 1;	// S2/RA8 as input
	/*Assigning the TX and RX pins to ports RP7 & RP9 to the dsPIC33FJ32MC204*/
	/************** Code section for the low pin count devices ******/
	__builtin_write_OSCCONL(OSCCON & (~(1<<6))); // clear bit 6 
	RPINR18bits.U1RXR = 7;  // Make Pin RP7 U1RX
	RPOR4bits.RP9R = 3;		// Make Pin RP9 U1TX
	__builtin_write_OSCCONL(OSCCON | (1<<6)); 	 // Set bit 6 
	/****************************************************************/	
	return;
}
/***********************************************************************************************************************
*  2008 Microchip Technology Inc.                                                    									*
*                                                                           											*
* FileName:        ClosedLoopSenBlDC.c                                                     								*
* Dependencies:    Header (.h) files if applicable, see below                                      						*
* Processor:           dsPIC33FJ32MC204                                                                          											*
* SOFTWARE LICENSE AGREEMENT:                                                  								            *
* Microchip Technology Incorporated ("Microchip") retains all ownership and                           				    *
* intellectual property rights in the code accompanying this message and in all                           				*
* derivatives hereto.  You may use this code, and any derivatives created by                            				*
* any person or entity by or on your behalf, exclusively with Microchip's                              					*
* proprietary products.  Your acceptance and/or use of this code constitutes                            				*
* agreement to the terms and conditions of this notice.                                         						*
*                                                                          												*
* CODE ACCOMPANYING THIS MESSAGE IS SUPPLIED BY MICROCHIP "AS IS".  NO                   			                    *
* WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED          		                        *
* TO, IMPLIED WARRANTIES OF NON-INFRINGEMENT, MERCHANTABILITY AND FITNESS FOR A         		                        *
* PARTICULAR PURPOSE APPLY TO THIS CODE, ITS INTERACTION WITH MICROCHIP'S                			                    *
* PRODUCTS, COMBINATION WITH ANY OTHER PRODUCTS, OR USE IN ANY APPLICATION.             		                        *
*                                                                           											*
* YOU ACKNOWLEDGE AND AGREE THAT, IN NO EVENT, SHALL MICROCHIP BE LIABLE, WHETHER         		                        *
* IN CONTRACT, WARRANTY, TORT (INCLUDING NEGLIGENCE OR BREACH OF STATUTORY DUTY),         		                        *
* STRICT LIABILITY, INDEMNITY, CONTRIBUTION, OR OTHERWISE, FOR ANY INDIRECT, SPECIAL,        	                        *
* PUNITIVE, EXEMPLARY, INCIDENTAL OR CONSEQUENTIAL LOSS, DAMAGE, FOR COST OR EXPENSE OF    	                            *
* ANY KIND WHATSOEVER RELATED TO THE CODE, HOWSOEVER CAUSED, EVEN IF MICROCHIP HAS BEEN   	                            *
* ADVISED OF THE POSSIBILITY OR THE DAMAGES ARE FORESEEABLE.  TO THE FULLEST EXTENT         		                    *
* ALLOWABLE BY LAW, MICROCHIP'S TOTAL LIABILITY ON ALL CLAIMS IN ANY WAY RELATED TO        		                        *
* THIS CODE, SHALL NOT EXCEED THE PRICE YOU PAID DIRECTLY TO MICROCHIP SPECIFICALLY TO       	                        *
* HAVE THIS CODE DEVELOPED.                                                     									    *
*                                                                           											*
* You agree that you are solely responsible for testing the code and                                  					*
* determining its suitability.  Microchip has no obligation to modify, test,                              				*
* certify, or support the code.                                                        									*
*                                                                           											*
* ADDITIONAL NOTES:                                                           										    *
* Code Tested on:                                                                										*
* dsPICDEM MCLV development board using the dsPIC33FJ32MC204 device                           					        *
*Revision History                                                               										*
*6/19/05 -- first version                                                            									*
*8/05/05 -- second version                                                          									*
*6/19/08 -- third version                                                           									*
*----------------------------------------------------------------------                                        			*
*OpenLoopSenBlDC.c uses the periphals on the dsPIC to implement a                                 					    *
*sensored open loop BLDC motor control algorithm.                                           							*
*In this program the Hall effect sensors are connected to the                                     						*
*Change Notification pins.  When a transition from one sector to                                   						*
*the next occurs, the value on the CN pins changes and causes an                                  						*
*interrupt.  In this interrupt the Hall sensors are read, and used                                   					*
*as an offset in a lookup table to generate a new 16-bit value                                     						*
*for the OVDCON register.  This value enables windings at PWM, low and                             					    *
*inactive appropriately to drive the BLDC motor.                                             							*
************************************************************************************************************************/