/*******************************************************************************
*  2014 Microchip Technology Inc.
*
* FileName:        uartDriver_solution.c
* Dependencies:    Header (.h) files if applicable, see below
* Processor:       dsPIC33EP512MU810
* Tested on:       EXPLORER16 development board (Microchip) ref DM240001
*                  dsPIC33EP512MU810 : processor module (PIM) ref  MA330025-1
* IDE :            MPLAB X v2.10 or above
* Compiler:        XC16 v1.21 or above
*
*******************************************************************************/

#include "xc.h"
#include "main.h"

/*** Macro definition *********************************************************/
// This macro computes the closest BRG value (rounding) based on BAUDRATE and Fcy
#define round(x) 	(int)(((float)(x))>0?(float)(x)+0.5:(float)(x)-0.5)
#define Fosc            80000000
#define FCY   		Fosc/2
#define BAUDRATE        9600                   
#define BRGVAL          round((FCY/BAUDRATE/(float)16)-1)
/******************************************************************************/

/*** Variables declaration ****************************************************/

//##############################################################################
// STEP1 : Allocate two 8 integer buffers for DMA transfers in DPSRAM space (dma space) (2 lines)
// hint1 : array name : use BufferA & BufferB which contains an array of integers
// hint2 : use __eds__ qualifier to enable compiler EDS management
// hint3 : use eds attribute __attribute__((eds)) so locate buffers in EDS space
// hint4 : alignment directive for XC16 : __attribute__((eds,space(dma),aligned(xxx)));
//##############################################################################
__eds__ unsigned int BufferA[8] __attribute__((eds,space(dma)));//#insert code here //# BufferA[8]
__eds__ unsigned int BufferB[8] __attribute__((eds,space(dma)));//#insert code here //# BufferB[8]

/******************************************************************************/

// UART Configuration
void cfgUart2(void)
{
// Map UART2 pins to RF4(U2RX) and RF5(U2TX) using PPS
    RPINR19bits.U2RXR = 0b1100100;  // Connect U2RX to RF4(RP100) (input)
    RPOR9bits.RP101R = 3;           // Connect RF5(RP101) to U2TX (output)
// Select RX and TX port direction
    TRISFbits.TRISF4 = 1;           // RF4 (U2RX) as input
    TRISFbits.TRISF5 = 0;           // RF5 (U2TX) as output
//  Setup Stop Bits, Parity, Data bits and baudrate according to user choice
// (BAUDRATE definition)
    U2MODEbits.STSEL = 0;	// 1-stop bit
    U2MODEbits.PDSEL = 0;	// No Parity, 8-data bits
    U2MODEbits.ABAUD = 0;	// Autobaud Disabled
    U2BRG = BRGVAL;		// Setup baudrate according to user definition
        
//  Configure UART2
    U2STAbits.UTXISEL1 = 0; // Interrupt after one TX character is transmitted (SEL1)
    U2STAbits.UTXISEL0 = 0; // Interrupt after one TX character is transmitted (SEL0)
    U2STAbits.URXISEL  = 0; // Interrupt after one RX character is received
    U2MODEbits.UARTEN  = 1; // Enable UART2 module
    U2STAbits.UTXEN    = 1; // Enable UART2 TX

    IEC4bits.U2EIE = 0;         // Enable UART2 ERROR interrupt (if error handler used)
}

// DMA0 channel configuration
void cfgDma0UartTx(void)
{
//##############################################################################
//  STEP2 : Setup the Transmitter's DMA operation using channel 0 (9 lines)
//  DMA0REQ :
//      Set DMA channel 0 for automatic mode
//      Associate DMA Channel 0 with UART2 TX
//  DMA0PAD :
//      Source address is U2TXREG peripheral address
//  DMA0CON :
//      Register Indirect with Post-Increment
//      One-Shot mode using single buffer (no ping-pong mode)
//      Transfer data from RAM to UART
//      Set transfer size to word
//  DMA0CNT :
//      Transfer 8 words
//  DMA0STAH/L :
//       Associate BufferA with Channel 0 for one-shot operation
//       hint : use __builtin_dmapage(xxx) to get BufferA page (High address) at linking time
//       hint : use __builtin_dmaoffset(xxx) to get BufferA offset (Low address) at linking time
//##############################################################################
    DMA0REQbits.FORCE = 0;//#insert code here //# Set DMA Channel 0 for Automatic mode
    DMA0REQbits.IRQSEL = 0x001F;//# insert code here //# Select UART2 Transmit interrupt source
    DMA0PAD = (volatile unsigned int) &U2TXREG;//# insert code here //# Point DMA0 to U2TXREG address

    DMA0CONbits.AMODE = 0;//# insert code here //# Set DMA0 channel to Register Indirect with Post-Increment mode
    DMA0CONbits.MODE  = 1;//# insert code here //# Set DMA0 channel to One-Shot, no Ping-Pong mode
    DMA0CONbits.DIR   = 1;//# insert code here //# Set Transfer Direction to read from RAM, write to peripheral
    DMA0CONbits.SIZE  = 0;//# insert code here //# Set Data Transfer Size to word

    DMA0CNT = 7;//# insert code here //# Set the number of DMA0 requests

    DMA0STAH = __builtin_dmapage(&BufferA);// # insert code here //# DMA0STAH : Associate BufferA page to DMA0 (High address)
    DMA0STAL = __builtin_dmaoffset(&BufferA);// # insert code here //# DMA0STAL : Associate BufferA offset to DMA0 (Low address)

    IFS0bits.DMA0IF  = 0; // Clear DMA0 Interrupt Flag
    IEC0bits.DMA0IE  = 1; // Enable DMA0 interrupt
}

// DMA1 channel configuration
void cfgDma1UartRx(void)
{
//##############################################################################
// STEP3 : Setup the Receiver's DMA channel 1 operation (12 lines)
// DMA1REQ :
//      UART2 RX as DMA1 interrupt request source
// DMA1PAD :
//      Affect UART2 RX address as peripheral source address
// DMA1CON :
//      Transfer words
//      Transfer data from UART to RAM
//      Peripheral Indirect mode with post-increment
//      Continuous, ping-pong buffers
// DMA1CNT :
//      8 transfers per buffer
// DMA1STA :
//      Associate BufferA with Channel 1 (for Ping-Pong operation)
//      hint : use __builtin_dmapage(xxx) to get BufferA page (High address) at linking time
//      hint : use __builtin_dmaoffset(xxx) to get BufferA offset (Low address) at linking time
// DMA1STB :
//      Associate BufferB with Channel 1 (for Ping-Pong operation)
//      hint : use __builtin_dmapage(xxx) to get BufferB page (High address) at linking time
//      hint : use __builtin_dmaoffset(xxx) to get BufferB offset (Low address) at linking time
// DMA1CON :
//      Enable DMA Channel 1
//##############################################################################
    DMA1REQbits.IRQSEL = 0x001E;//# insert code here //#Select UART2 Receiver interrupt as DMA1 request
    DMA1PAD = (volatile unsigned int) &U2RXREG;//# insert code here //#  Point DMA1 to U2RXREG address
    //DMA1CON = 0x0002;             // Continuous, Ping-Pong, Post-Inc, Periph-RAM
    DMA1CONbits.SIZE  = 0;//# insert code here //# Set Data Transfer Size
    DMA1CONbits.DIR   = 0;//# insert code here //# Set Transfer Direction
    DMA1CONbits.AMODE = 0;//# insert code here //# Set DMA1 channel addressing mode
    DMA1CONbits.MODE  = 2;//# insert code here //# Set DMA1 channel operating mode

    DMA1CNT = 7;//# insert code here //# Set the number of DMA1 requests

    DMA1STAH = __builtin_dmapage(&BufferA);// # insert code here //# DMA1STAH : Associate BufferA offset to DMA1 (High address)
    DMA1STAL = __builtin_dmaoffset(&BufferA);// # insert code here //# DMA1STAL : Associate BufferA offset to DMA1 (Low address)
    DMA1STBH = __builtin_dmapage(&BufferB);// # insert code here //# DMA1STBH : Associate BufferB offset to DMA1 (High address)
    DMA1STBL = __builtin_dmaoffset(&BufferB);// # insert code here //# DMA1STBL : Associate BufferB offset to DMA1 (Low address)

    IFS0bits.DMA1IF  = 0;// Clear DMA1 interrupt request flag
    IEC0bits.DMA1IE  = 1;// Enable DMA1 interrupts

    DMA1CONbits.CHEN = 1;//# insert code here //#  Enable DMA1 Channel
}

//##############################################################################
//  STEP4 : Setup the DMA0 interrupt handler (1 line)
// IFS0 :
//      clear the DMA0 interrupt request flag
//##############################################################################
void __attribute__((interrupt, no_auto_psv)) _DMA0Interrupt(void)
{
    TX_led_D4 = TX_led_D4 ^ 1;  // toggle transmit activity led D4
    IFS0bits.DMA0IF = 0;//# insert code here //# Clear the DMA0 Interrupt Flag;
}

//##############################################################################
//  STEP5 : Setup the DMA1 interrupt handler (6 lines)
// DMA0STA :
//	Associate BufferA with Channel 0 if Buffercount = 0 to tranmit back 8 characters
//      hint : use __builtin_dmapage(xxx) to get BufferA page (High address) at linking time
//      hint : use __builtin_dmaoffset(xxx) to get BufferA offset (Low address) at linking time
// DMA0STA :
//	Associate BufferB with Channel 0 if Buffercount = 1 to tranmit back 8 characters
//      hint : use __builtin_dmapage(xxx) to get BufferB page (High address) at linking time
//      hint : use __builtin_dmaoffset(xxx) to get BufferB offset (Low address) at linking time
// DMA0CON :
//      Re-Enable DMA Channel 0 (as it is in single shot mode)
// DMA0REQ :
//      Force first TX transfer to generate TX empty interrupts
//##############################################################################
void __attribute__((interrupt, no_auto_psv)) _DMA1Interrupt(void)
{
    static unsigned int BufferCount = 0;  // Keep record of which buffer contains Rx Data

    if(BufferCount == 0)
    {
        DMA0STAH = __builtin_dmapage(&BufferA);// # insert code here //# Point DMA0STAH to BufferA (High address)
        DMA0STAL = __builtin_dmaoffset(&BufferA);// # insert code here //# Point DMA0STAL to BufferA (Low address)
    }
    else
    {
        DMA0STAH = __builtin_dmapage(&BufferB);// # insert code here //# Point DMA0STAH to BufferB (High address)
        DMA0STAL = __builtin_dmaoffset(&BufferB);// # insert code here //# Point DMA0STAL to BufferB (Low address)
    }

    DMA0CONbits.CHEN  = 1;//# insert code here //# Re-enable DMA0 Channel
    DMA0REQbits.FORCE = 1;//# insert code here //# Manual mode: Kick-start (force) the first transfer

    BufferCount = BufferCount ^1;   // Toggle Buffer indicator semaphore
    RX_led_D3 = RX_led_D3 ^ 1;  // toggle receive activity led D3
    IFS0bits.DMA1IF = 0;    // Clear the DMA1 Interrupt Flag
}


void __attribute__((interrupt, no_auto_psv)) _U2ErrInterrupt(void)
{
    U2_error_led_D9 = 1 ;   // we had an error in UART 2, light Led D9
    while(1);               // stop here
//    IFS4bits.U2EIF = 0;     // Clear the UART2 Error Interrupt Flag
}
