volatile unsigned long long __coreTime = 0;

_INTERRUPT_JUMP (_TIMER0_OVF)
{
	__coreTime++;
}

unsigned long long Core::s()
{
	unsigned char tcnt0Save = 0;
	unsigned long long s = 0;
	
	_SREG &= ~0b10000000;
	
	tcnt0Save = _TCNT0;
	
	if ((_TIFR0 & 0b00000001) == true)
	{
		tcnt0Save = _TCNT0;
		_TIFR0 |= 0b00000001;
		
		__coreTime++;
	}
	
	s = ((__coreTime * 256ull) + (unsigned long long)(tcnt0Save)) / 2000000ull;
	
	_SREG |= 0b10000000;
	
	return s;
}

unsigned long long Core::ms()
{
	unsigned char tcnt0Save = 0;
	unsigned long long ms = 0;
	
	_SREG &= ~0b10000000;
	
	tcnt0Save = _TCNT0;
	
	if ((_TIFR0 & 0b00000001) == true)
	{
		tcnt0Save = _TCNT0;
		_TIFR0 |= 0b00000001;
		
		__coreTime++;
	}
	
	ms = ((__coreTime * 256ull) + (unsigned long long)(tcnt0Save)) / 2000ull;
	
	_SREG |= 0b10000000;
	
	return ms;
}

unsigned long long Core::us()
{
	unsigned char tcnt0Save = 0;
	unsigned long long us = 0;
	
	_SREG &= ~0b10000000;
	
	tcnt0Save = _TCNT0;
	
	if ((_TIFR0 & 0b00000001) == true)
	{
		tcnt0Save = _TCNT0;
		_TIFR0 |= 0b00000001;
		
		__coreTime++;
	}
	
	us = ((__coreTime * 256ull) + (unsigned long long)(tcnt0Save)) / 2ull;
	
	_SREG |= 0b10000000;
	
	return us;
}

volatile unsigned char *Core::pinToRegisterPort (const unsigned char PIN)
{
	volatile unsigned char *registerPort = 0;
	
	if (PIN >= 1 && PIN <= 8)
	{
		registerPort = &_PORTB;
	}
	else if (PIN >= 9 && PIN <= 16)
	{
		registerPort = &_PORTD;
	}
	else if (PIN >= 17 && PIN <= 24)
	{
		registerPort = &_PORTC;
	}
	else if (PIN >= 25 && PIN <= 32)
	{
		registerPort = &_PORTA;
	}
	
	return registerPort;
}

volatile unsigned char *Core::pinToRegisterDdr (const unsigned char PIN)
{
	volatile unsigned char *registerDdr = 0;
	
	if (PIN >= 1 && PIN <= 8)
	{
		registerDdr = &_DDRB;
	}
	else if (PIN >= 9 && PIN <= 16)
	{
		registerDdr = &_DDRD;
	}
	else if (PIN >= 17 && PIN <= 24)
	{
		registerDdr = &_DDRC;
	}
	else if (PIN >= 25 && PIN <= 32)
	{
		registerDdr = &_DDRA;
	}
	
	return registerDdr;
}

volatile unsigned char *Core::pinToRegisterPin (const unsigned char PIN)
{
	volatile unsigned char *registerPin = 0;
	
	if (PIN >= 1 && PIN <= 8)
	{
		registerPin = &_PINB;
	}
	else if (PIN >= 9 && PIN <= 16)
	{
		registerPin = &_PIND;
	}
	else if (PIN >= 17 && PIN <= 24)
	{
		registerPin = &_PINC;
	}
	else if (PIN >= 25 && PIN <= 32)
	{
		registerPin = &_PINA;
	}
	
	return registerPin;
}

unsigned char Core::pinToBytePort (const unsigned char PIN)
{
	unsigned char bytePort = 0;
	
	if (PIN == 1 || PIN == 9 || PIN == 17 || PIN == 32)
	{
		bytePort = 0b00000001;
	}
	else if (PIN == 2 || PIN == 10 || PIN == 18 || PIN == 31)
	{
		bytePort = 0b00000010;
	}
	else if (PIN == 3 || PIN == 11 || PIN == 19 || PIN == 30)
	{
		bytePort = 0b00000100;
	}
	else if (PIN == 4 || PIN == 12 || PIN == 20 || PIN == 29)
	{
		bytePort = 0b00001000;
	}
	else if (PIN == 5 || PIN == 13 || PIN == 21 || PIN == 28)
	{
		bytePort = 0b00010000;
	}
	else if (PIN == 6 || PIN == 14 || PIN == 22 || PIN == 27)
	{
		bytePort = 0b00100000;
	}
	else if (PIN == 7 || PIN == 15 || PIN == 23 || PIN == 26)
	{
		bytePort = 0b01000000;
	}
	else if (PIN == 8 || PIN == 16 || PIN == 24 || PIN == 25)
	{
		bytePort = 0b10000000;
	}
	
	return bytePort;
}

unsigned char Core::pinToInterrupt (const unsigned char PIN)
{
	unsigned char interrupt = 0;
	
	if (PIN == 3)
	{
		interrupt = 2;
	}
	else if (PIN == 12)
	{
		interrupt = 1;
	}
	
	return interrupt;
}

volatile unsigned char *Core::pinToRegisterPcmsk (const unsigned char PIN)
{
	volatile unsigned char *registerPcmsk = 0;
	
	if (PIN >= 1 && PIN <= 8)
	{
		registerPcmsk = &_PCMSK1;
	}
	else if (PIN >= 9 && PIN <= 16)
	{
		registerPcmsk = &_PCMSK3;
	}
	else if (PIN >= 17 && PIN <= 24)
	{
		registerPcmsk = &_PCMSK2;
	}
	else if (PIN >= 25 && PIN <= 32)
	{
		registerPcmsk = &_PCMSK0;
	}
	
	return registerPcmsk;
}

unsigned char Core::pinToBytePcie (const unsigned char PIN)
{
	unsigned char bytePcie = 0;
	
	if (PIN >= 1 && PIN <= 8)
	{
		bytePcie = 0b00000010;
	}
	else if (PIN >= 9 && PIN <= 16)
	{
		bytePcie = 0b00001000;
	}
	else if (PIN >= 17 && PIN <= 24)
	{
		bytePcie = 0b00000100;
	}
	else if (PIN >= 25 && PIN <= 32)
	{
		bytePcie = 0b00000001;
	}
	
	return bytePcie;
}

bool Core::pinIsChannelTimer1 (const unsigned char PIN)
{
	bool is = false;
	
	if (PIN == 13 || PIN == 14)
	{
		is = true;
	}
	
	return is;
}

unsigned char Core::pinToByteCom1 (const unsigned char PIN)
{
	unsigned char byteCom1 = 0;
	
	if (PIN == 13)
	{
		byteCom1 = 0b00100000;
	}
	else if (PIN == 14)
	{
		byteCom1 = 0b10000000;
	}
	
	return byteCom1;
}

volatile unsigned int *Core::pinToRegisterOcr1 (const unsigned char PIN)
{
	volatile unsigned int *registerOcr1 = 0;
	
	if (PIN == 13)
	{
		registerOcr1 = &_OCR1BLH;
	}
	else if (PIN == 14)
	{
		registerOcr1 = &_OCR1ALH;
	}
	
	return registerOcr1;
}

unsigned char Core::pinToByteAdmux (const unsigned char PIN)
{
	return 0b01000000 | (32 - PIN);
}

void Core::startGlobalInterrupt()
{
	_SREG |= 0b10000000;
}

void Core::startInterruptTimer0()
{
	_TCCR0B = 0b00000010;
	_TIMSK0 = 0b00000001;
}

void Core::modeTimer1()
{
	_TCCR1A = 0b00000010;
	_TCCR1B = 0b00011000;
}

unsigned int Core::frequencyToPrescalerTimer1 (const unsigned int FREQUENCY)
{
	unsigned int prescaler = 1024;
	
	if (((16000000.0 / 1.0) / float (FREQUENCY)) - 0.5 <= 65535)
	{
		prescaler = 1;
	}
	else if (((16000000.0 / 8.0) / float (FREQUENCY)) - 0.5 <= 65535)
	{
		prescaler = 8;
	}
	else if (((16000000.0 / 64.0) / float (FREQUENCY)) - 0.5 <= 65535)
	{
		prescaler = 64;
	}
	else if (((16000000.0 / 256.0) / float (FREQUENCY)) - 0.5 <= 65535)
	{
		prescaler = 256;
	}
	
	return prescaler;
}

void Core::prescalerTimer1 (const unsigned int PRESCALER)
{
	if (PRESCALER == 1)
	{
		_TCCR1B |= 0b00000001;
	}
	else if (PRESCALER == 8)
	{
		_TCCR1B |= 0b00000010;
	}
	else if (PRESCALER == 64)
	{
		_TCCR1B |= 0b00000011;
	}
	else if (PRESCALER == 256)
	{
		_TCCR1B |= 0b00000100;
	}
	else if (PRESCALER == 1024)
	{
		_TCCR1B |= 0b00000101;
	}
}

void Core::inputCaptureTimer1 (const unsigned int VALUE)
{
	_ICR1LH = VALUE;
}

void Core::outputCompareTimer1 (const unsigned int VALUE)
{
	_OCR1ALH = VALUE;
}

void Core::startInterruptTimer1()
{
	_TIMSK1 = 0b00000011;
}

void Core::startOutputCompareMatchTimer1 (const unsigned char BYTE_COM1)
{
	_TCCR1A |= BYTE_COM1;
}

void Core::stopOutputCompareMatchTimer1 (const unsigned char BYTE_COM1)
{
	_TCCR1A &= ~BYTE_COM1;
}

void Core::modeTimer2()
{
	_TCCR2A = 0b00000010;
}

unsigned int Core::frequencyToPrescalerTimer2 (const unsigned int FREQUENCY)
{
	unsigned int prescaler = 1024;
	
	if (((16000000.0 / 1.0) / float (FREQUENCY)) - 0.5 <= 255)
	{
		prescaler = 1;
	}
	else if (((16000000.0 / 8.0) / float (FREQUENCY)) - 0.5 <= 255)
	{
		prescaler = 8;
	}
	else if (((16000000.0 / 32.0) / float (FREQUENCY)) - 0.5 <= 255)
	{
		prescaler = 32;
	}
	else if (((16000000.0 / 64.0) / float (FREQUENCY)) - 0.5 <= 255)
	{
		prescaler = 64;
	}
	else if (((16000000.0 / 128.0) / float (FREQUENCY)) - 0.5 <= 255)
	{
		prescaler = 128;
	}
	else if (((16000000.0 / 256.0) / float (FREQUENCY)) - 0.5 <= 255)
	{
		prescaler = 256;
	}
	
	return prescaler;
}

void Core::prescalerTimer2 (const unsigned int PRESCALER)
{
	if (PRESCALER == 1)
	{
		_TCCR2B = 0b00000001;
	}
	else if (PRESCALER == 8)
	{
		_TCCR2B = 0b00000010;
	}
	else if (PRESCALER == 32)
	{
		_TCCR2B = 0b00000011;
	}
	else if (PRESCALER == 64)
	{
		_TCCR2B = 0b00000100;
	}
	else if (PRESCALER == 128)
	{
		_TCCR2B = 0b00000101;
	}
	else if (PRESCALER == 256)
	{
		_TCCR2B = 0b00000110;
	}
	else if (PRESCALER == 1024)
	{
		_TCCR2B = 0b00000111;
	}
}

void Core::outputCompareTimer2 (const unsigned char VALUE)
{
	_OCR2A = VALUE;
}

void Core::startInterruptTimer2()
{
	_TIMSK2 = 0b00000010;
}

void Core::stopInterruptTimer2()
{
	_TIMSK2 = 0b00000000;
}

void Core::startInterrupt (const unsigned char INTERRUPT, const bool RISING, const bool FALLING)
{
	if (INTERRUPT == 0)
	{
		if (RISING == true && FALLING == false)
		{
			_EICRA |= 0b00000011;
		}
		else if (RISING == false && FALLING == true)
		{
			_EICRA |= 0b00000010;
		}
		else
		{
			_EICRA |= 0b00000001;
		}
		
		_EIMSK |= 0b00000001;
	}
	else if (INTERRUPT == 1)
	{
		if (RISING == true && FALLING == false)
		{
			_EICRA |= 0b00001100;
		}
		else if (RISING == false && FALLING == true)
		{
			_EICRA |= 0b00001000;
		}
		else
		{
			_EICRA |= 0b00000100;
		}
		
		_EIMSK |= 0b00000010;
	}
	else if (INTERRUPT == 2)
	{
		if (RISING == true && FALLING == false)
		{
			_EICRA |= 0b00110000;
		}
		else if (RISING == false && FALLING == true)
		{
			_EICRA |= 0b00100000;
		}
		else
		{
			_EICRA |= 0b00010000;
		}
		
		_EIMSK |= 0b00000100;
	}
}

void Core::stopInterrupt (const unsigned char INTERRUPT)
{
	if (INTERRUPT == 0)
	{
		_EIMSK &= ~0b00000001;
	}
	else if (INTERRUPT == 1)
	{
		_EIMSK &= ~0b00000010;
	}
	else if (INTERRUPT == 2)
	{
		_EIMSK &= ~0b00000100;
	}
}

void Core::startPinChangeInterrupt (const unsigned char BYTE_PCIE)
{
	_PCICR = BYTE_PCIE;
}

void Core::stopPinChangeInterrupt()
{
	_PCICR = 0b00000000;
}

unsigned char Core::digitToByte (const unsigned char DIGIT, const bool COMMA)
{
	unsigned char byte = 0;
	
	if (DIGIT == 0)
	{
		byte = 0b01111110;
	}
	else if (DIGIT == 1)
	{
		byte = 0b00110000;
	}
	else if (DIGIT == 2)
	{
		byte = 0b01101101;
	}
	else if (DIGIT == 3)
	{
		byte = 0b01111001;
	}
	else if (DIGIT == 4)
	{
		byte = 0b00110011;
	}
	else if (DIGIT == 5)
	{
		byte = 0b01011011;
	}
	else if (DIGIT == 6)
	{
		byte = 0b01011111;
	}
	else if (DIGIT == 7)
	{
		byte = 0b01110000;
	}
	else if (DIGIT == 8)
	{
		byte = 0b01111111;
	}
	else if (DIGIT == 9)
	{
		byte = 0b01111011;
	}
	
	if (COMMA == true)
	{
		byte |= 0b10000000;
	}
	
	return byte;
}

unsigned char Core::characterToByte (const char CHARACTER)
{
	unsigned char byte = 0;
	
	if (CHARACTER == 0b01110001)
	{
		byte = 0b01111110;
	}
	else if (CHARACTER == 0b01110111)
	{
		byte = 0b00111110;
	}
	else if (CHARACTER == 0b01100101)
	{
		byte = 0b01001111;
	}
	else if (CHARACTER == 0b01110010)
	{
		byte = 0b01000110;
	}
	else if (CHARACTER == 0b01110100)
	{
		byte = 0b00000111;
	}
	else if (CHARACTER == 0b01111001)
	{
		byte = 0b00110011;
	}
	else if (CHARACTER == 0b01110101)
	{
		byte = 0b00111110;
	}
	else if (CHARACTER == 0b01101001)
	{
		byte = 0b00110000;
	}
	else if (CHARACTER == 0b01101111)
	{
		byte = 0b01111110;
	}
	else if (CHARACTER == 0b01110000)
	{
		byte = 0b01100111;
	}
	else if (CHARACTER == 0b01100001)
	{
		byte = 0b01110111;
	}
	else if (CHARACTER == 0b01110011)
	{
		byte = 0b01011011;
	}
	else if (CHARACTER == 0b01100100)
	{
		byte = 0b01111000;
	}
	else if (CHARACTER == 0b01100110)
	{
		byte = 0b01000111;
	}
	else if (CHARACTER == 0b01100111)
	{
		byte = 0b01011110;
	}
	else if (CHARACTER == 0b01101000)
	{
		byte = 0b00110111;
	}
	else if (CHARACTER == 0b01101010)
	{
		byte = 0b00111000;
	}
	else if (CHARACTER == 0b01101011)
	{
		byte = 0b00110111;
	}
	else if (CHARACTER == 0b01101100)
	{
		byte = 0b00001110;
	}
	else if (CHARACTER == 0b01111010)
	{
		byte = 0b01101101;
	}
	else if (CHARACTER == 0b01111000)
	{
		byte = 0b00110111;
	}
	else if (CHARACTER == 0b01100011)
	{
		byte = 0b01001110;
	}
	else if (CHARACTER == 0b01110110)
	{
		byte = 0b00111110;
	}
	else if (CHARACTER == 0b01100010)
	{
		byte = 0b01111111;
	}
	else if (CHARACTER == 0b01101110)
	{
		byte = 0b01110110;
	}
	else if (CHARACTER == 0b01101101)
	{
		byte = 0b01110110;
	}
	else if (CHARACTER == 0b00101110)
	{
		byte = 0b10000000;
	}
	else if (CHARACTER == 0b00101101)
	{
		byte = 0b00000001;
	}
	else if (CHARACTER == 0b00111101)
	{
		byte = 0b00001001;
	}
	else if (CHARACTER == 0b00110000)
	{
		byte = 0b01111110;
	}
	else if (CHARACTER == 0b00110001)
	{
		byte = 0b00110000;
	}
	else if (CHARACTER == 0b00110010)
	{
		byte = 0b01101101;
	}
	else if (CHARACTER == 0b00110011)
	{
		byte = 0b01111001;
	}
	else if (CHARACTER == 0b00110100)
	{
		byte = 0b00110011;
	}
	else if (CHARACTER == 0b00110101)
	{
		byte = 0b01011011;
	}
	else if (CHARACTER == 0b00110110)
	{
		byte = 0b01011111;
	}
	else if (CHARACTER == 0b00110111)
	{
		byte = 0b01110000;
	}
	else if (CHARACTER == 0b00111000)
	{
		byte = 0b01111111;
	}
	else if (CHARACTER == 0b00111001)
	{
		byte = 0b01111011;
	}
	
	return byte;
}

void Core::startUsart (const bool RECEIVER, const bool TRANSMITTER)
{
	_UCSR0A = 0b00000010;
	
	if (RECEIVER == true && TRANSMITTER == false)
	{
		_UCSR0B = 0b10010000;
	}
	else if (RECEIVER == false && TRANSMITTER == true)
	{
		_UCSR0B = 0b00001000;
	}
	else
	{
		_UCSR0B = 0b10011000;
	}
	
	_UCSR0C = 0b00000110;
}

signed long Core::readUsart (const signed char LENGTH)
{
	unsigned char shift = 0;
	signed long data = 0;
	
	if (LENGTH < 0)
	{
		for (shift = 0; shift != -LENGTH; shift += 8)
		{
			while ((_UCSR0A & 0b10000000) == false)
			{
			}
			
			if (LENGTH == -8)
			{
				data |= (signed char)(_UDR0) << shift;
			}
			else if (LENGTH == -16)
			{
				data |= (signed int)(_UDR0) << shift;
			}
			else
			{
				data |= (signed long)(_UDR0) << shift;
			}
		}
	}
	else
	{
		for (shift = LENGTH; shift != 0; shift -= 8)
		{
			while ((_UCSR0A & 0b10000000) == false)
			{
			}
			
			if (LENGTH == 8)
			{
				data |= (signed char)(_UDR0) << (shift - 8);
			}
			else if (LENGTH == 16)
			{
				data |= (signed int)(_UDR0) << (shift - 8);
			}
			else
			{
				data |= (signed long)(_UDR0) << (shift - 8);
			}
		}
	}
	
	return data;
}

void Core::writeUsart (const signed long DATA, const signed char LENGTH)
{
	unsigned char shift = 0;
	
	if (LENGTH < 0)
	{
		for (shift = 0; shift != -LENGTH; shift += 8)
		{
			_UDR0 = DATA >> shift;
			
			while ((_UCSR0A & 0b00100000) == false)
			{
			}
		}
	}
	else
	{
		for (shift = LENGTH; shift != 0; shift -= 8)
		{
			_UDR0 = DATA >> (shift - 8);
			
			while ((_UCSR0A & 0b00100000) == false)
			{
			}
		}
	}
}

void Core::startSpi (const unsigned char PIN_SS)
{
	_DDRB |= 0b10110000;
	_SPCR0 = 0b01010000;
	
	if (PIN_SS != 0)
	{
		_countSpiDevice [PIN_SS - 1]++;
	}
}

signed long Core::readSpi (const unsigned char ADDRESS, const signed char LENGTH)
{
	unsigned char shift = 0;
	signed long data = 0;
	
	_SPDR0 = ADDRESS;
	
	while (!(_SPSR0 & 0b10000000) == true)
	{
	}
	
	if (LENGTH < 0)
	{
		for (shift = 0; shift != -LENGTH; shift += 8)
		{
			_SPDR0 = 0b11111111;
			
			while (!(_SPSR0 & 0b10000000) == true)
			{
			}
			
			if (LENGTH == -8)
			{
				data |= (signed char)(_SPDR0) << shift;
			}
			else if (LENGTH == -16)
			{
				data |= (signed int)(_SPDR0) << shift;
			}
			else
			{
				data |= (signed long)(_SPDR0) << shift;
			}
		}
	}
	else
	{
		for (shift = LENGTH; shift != 0; shift -= 8)
		{
			_SPDR0 = 0b11111111;
			
			while (!(_SPSR0 & 0b10000000) == true)
			{
			}
			
			if (LENGTH == 8)
			{
				data |= (signed char)(_SPDR0) << (shift - 8);
			}
			else if (LENGTH == 16)
			{
				data |= (signed int)(_SPDR0) << (shift - 8);
			}
			else
			{
				data |= (signed long)(_SPDR0) << (shift - 8);
			}
		}
	}
	
	return data;
}

signed long Core::readSpiPush (const signed char LENGTH)
{
	unsigned char shift = 0;
	signed long data = 0;
	
	if (LENGTH < 0)
	{
		for (shift = 0; shift != -LENGTH; shift += 8)
		{
			_SPDR0 = 0b11111111;
			
			while (!(_SPSR0 & 0b10000000) == true)
			{
			}
			
			if (LENGTH == -8)
			{
				data |= (signed char)(_SPDR0) << shift;
			}
			else if (LENGTH == -16)
			{
				data |= (signed int)(_SPDR0) << shift;
			}
			else
			{
				data |= (signed long)(_SPDR0) << shift;
			}
		}
	}
	else
	{
		for (shift = LENGTH; shift != 0; shift -= 8)
		{
			_SPDR0 = 0b11111111;
			
			while (!(_SPSR0 & 0b10000000) == true)
			{
			}
			
			if (LENGTH == 8)
			{
				data |= (signed char)(_SPDR0) << (shift - 8);
			}
			else if (LENGTH == 16)
			{
				data |= (signed int)(_SPDR0) << (shift - 8);
			}
			else
			{
				data |= (signed long)(_SPDR0) << (shift - 8);
			}
		}
	}
	
	return data;
}

void Core::writeSpi (const unsigned char ADDRESS, const signed long DATA, const signed char LENGTH)
{
	unsigned char shift = 0;
	
	_SPDR0 = ADDRESS;
	
	while (!(_SPSR0 & 0b10000000) == true)
	{
	}
	
	if (LENGTH < 0)
	{
		for (shift = 0; shift != -LENGTH; shift += 8)
		{
			_SPDR0 = DATA >> shift;
			
			while (!(_SPSR0 & 0b10000000) == true)
			{
			}
		}
	}
	else
	{
		for (shift = LENGTH; shift != 0; shift -= 8)
		{
			_SPDR0 = DATA >> (shift - 8);
			
			while (!(_SPSR0 & 0b10000000) == true)
			{
			}
		}
	}
}

void Core::writeSpiPush (const signed long DATA, const signed char LENGTH)
{
	unsigned char shift = 0;
	
	if (LENGTH < 0)
	{
		for (shift = 0; shift != -LENGTH; shift += 8)
		{
			_SPDR0 = DATA >> shift;
			
			while (!(_SPSR0 & 0b10000000) == true)
			{
			}
		}
	}
	else
	{
		for (shift = LENGTH; shift != 0; shift -= 8)
		{
			_SPDR0 = DATA >> (shift - 8);
			
			while (!(_SPSR0 & 0b10000000) == true)
			{
			}
		}
	}
}

void Core::writeSpiCascaded (const unsigned char PIN_SS, const unsigned char DEVICE, const unsigned char ADDRESS, const signed long DATA, const signed char LENGTH)
{
	unsigned char offset = 2;
	const unsigned char COUNT_SPI_DEVICE = _countSpiDevice [PIN_SS - 1];
	unsigned char shift = 0;
	
	for (offset = 2; offset < (COUNT_SPI_DEVICE - (DEVICE - 1)) * 2; offset++)
	{
		_SPDR0 = 0b00000000;
		
		while (!(_SPSR0 & 0b10000000) == true)
		{
		}
	}
	
	_SPDR0 = ADDRESS;
	
	while (!(_SPSR0 & 0b10000000) == true)
	{
	}
	
	if (LENGTH < 0)
	{
		for (shift = 0; shift != -LENGTH; shift += 8)
		{
			_SPDR0 = DATA >> shift;
			
			while (!(_SPSR0 & 0b10000000) == true)
			{
			}
		}
	}
	else
	{
		for (shift = LENGTH; shift != 0; shift -= 8)
		{
			_SPDR0 = DATA >> (shift - 8);
			
			while (!(_SPSR0 & 0b10000000) == true)
			{
			}
		}
	}
	
	for (offset = 2; offset < DEVICE * 2; offset++)
	{
		_SPDR0 = 0b00000000;
		
		while (!(_SPSR0 & 0b10000000) == true)
		{
		}
	}
}

void Core::startAnalog()
{
	_ADCSRA = 0b10000011;
}

unsigned int Core::readAnalog (const unsigned char BYTE_ADMUX)
{
	_ADMUX = BYTE_ADMUX;
	_ADCSRA |= 0b01000000;
	
	while (!(_ADCSRA & 0b01000000) == false)
	{
	}
	
	_ADCSRA |= 0b01000000;
	
	while (!(_ADCSRA & 0b01000000) == false)
	{
	}
	
	return _ADCLH;
}

void Core::startTwi()
{
	_TWBR = 0b00001100;
}

bool Core::scanTwi (const unsigned char DEVICE)
{
	unsigned int n = 0;
	unsigned char check = 0;
	unsigned char error = 0;
	bool foundDevice = false;
	
	while (n == 0 && error < 16)
	{
		_TWCR = 0b10100100;
		
		for (n = 512; !(_TWCR & 0b10000000) == true && n > 0; n--)
		{
		}
		
		if (n != 0)
		{
			check = _TWSR & 0b11111000;
			
			if (check != 0b00001000 && check != 0b00010000)
			{
				n = 0;
			}
		}
		
		if (n != 0)
		{
			_TWDR = DEVICE << 1;
			_TWCR = 0b10000100;
			
			for (n = 512; !(_TWCR & 0b10000000) == true && n > 0; n--)
			{
			}
		}
		
		if (n != 0)
		{
			check = _TWSR & 0b11111000;
			
			if (check != 0b00011000 && check != 0b01000000)
			{
				n = 0;
			}
		}
		
		if (n != 0)
		{
			_TWCR = 0b10010100;
			
			for (n = 512; !(_TWCR & 0b00010000) == true && n > 0; n--)
			{
			}
		}
		
		if (n == 0)
		{
			_TWCR = 0b00000000;
			error++;
		}
	}
	
	if (error < 16)
	{
		foundDevice = true;
	}
	
	return foundDevice;
}

signed long Core::readTwi (const unsigned char DEVICE, const unsigned char ADDRESS, const signed char LENGTH)
{
	unsigned int n = 0;
	unsigned char check = 0;
	unsigned char error = 0;
	unsigned char shift = 0;
	signed long data = 0;
	
	while (n == 0 && error < 16)
	{
		_TWCR = 0b10100100;
		
		for (n = 512; !(_TWCR & 0b10000000) == true && n > 0; n--)
		{
		}
		
		if (n != 0)
		{
			check = _TWSR & 0b11111000;
			
			if (check != 0b00001000 && check != 0b00010000)
			{
				n = 0;
			}
		}
		
		if (n != 0)
		{
			_TWDR = DEVICE << 1;
			_TWCR = 0b10000100;
			
			for (n = 512; !(_TWCR & 0b10000000) == true && n > 0; n--)
			{
			}
		}
		
		if (n != 0)
		{
			check = _TWSR & 0b11111000;
			
			if (check != 0b00011000 && check != 0b01000000)
			{
				n = 0;
			}
		}
		
		if (n != 0)
		{
			_TWDR = ADDRESS;
			_TWCR = 0b10000100;
			
			for (n = 512; !(_TWCR & 0b10000000) == true && n > 0; n--)
			{
			}
		}
		
		if (n != 0)
		{
			check = _TWSR & 0b11111000;
			
			if (check != 0b00101000)
			{
				n = 0;
			}
		}
		
		if (n != 0)
		{
			_TWCR = 0b10100100;
			
			for (n = 512; !(_TWCR & 0b10000000) == true && n > 0; n--)
			{
			}
		}
		
		if (n != 0)
		{
			check = _TWSR & 0b11111000;
			
			if (check != 0b00001000 && check != 0b00010000)
			{
				n = 0;
			}
		}
		
		if (n != 0)
		{
			_TWDR = (DEVICE << 1) + 0b00000001;
			_TWCR = 0b10000100;
			
			for (n = 512; !(_TWCR & 0b10000000) == true && n > 0; n--)
			{
			}
		}
		
		if (n != 0)
		{
			check = _TWSR & 0b11111000;
			
			if (check != 0b00011000 && check != 0b01000000)
			{
				n = 0;
			}
		}
		
		if (n != 0)
		{
			if (LENGTH < 0)
			{
				for (shift = 0; shift != -LENGTH && n != 0; shift += 8)
				{
					if (shift == -LENGTH - 8)
					{
						_TWCR = 0b10000100;
					}
					else
					{
						_TWCR = 0b11000100;
					}
					
					for (n = 512; !(_TWCR & 0b10000000) == true && n > 0; n--)
					{
					}
					
					if (LENGTH == -8)
					{
						data |= (signed char)(_TWDR) << shift;
					}
					else if (LENGTH == -16)
					{
						data |= (signed int)(_TWDR) << shift;
					}
					else
					{
						data |= (signed long)(_TWDR) << shift;
					}
				}
			}
			else
			{
				for (shift = LENGTH; shift != 0 && n != 0; shift -= 8)
				{
					if (shift == 8)
					{
						_TWCR = 0b10000100;
					}
					else
					{
						_TWCR = 0b11000100;
					}
					
					for (n = 512; !(_TWCR & 0b10000000) == true && n > 0; n--)
					{
					}
					
					if (LENGTH == 8)
					{
						data |= (signed char)(_TWDR) << (shift - 8);
					}
					else if (LENGTH == 16)
					{
						data |= (signed int)(_TWDR) << (shift - 8);
					}
					else
					{
						data |= (signed long)(_TWDR) << (shift - 8);
					}
				}
			}
		}
		
		if (n != 0)
		{
			_TWCR = 0b10010100;
			
			for (n = 512; !(_TWCR & 0b00010000) == true && n > 0; n--)
			{
			}
		}
		
		if (n == 0)
		{
			_TWCR = 0b00000000;
			data = 0;
			error++;
		}
	}
	
	return data;
}

void Core::writeTwi (const unsigned char DEVICE, const unsigned char ADDRESS, const signed long DATA, const signed char LENGTH)
{
	unsigned int n = 0;
	unsigned char check = 0;
	unsigned char error = 0;
	unsigned char shift = 0;
	
	while (n == 0 && error < 16)
	{
		_TWCR = 0b00000000;
		_TWCR = 0b10100100;
		
		for (n = 512; !(_TWCR & 0b10000000) == true && n > 0; n--)
		{
		}
		
		if (n != 0)
		{
			check = _TWSR & 0b11111000;
			
			if (check != 0b00001000 && check != 0b00010000)
			{
				n = 0;
			}
		}
		
		if (n != 0)
		{
			_TWDR = DEVICE << 1;
			_TWCR = 0b10000100;
			
			for (n = 512; !(_TWCR & 0b10000000) == true && n > 0; n--)
			{
			}
		}
		
		if (n != 0)
		{
			check = _TWSR & 0b11111000;
			
			if (check != 0b00011000 && check != 0b01000000)
			{
				n = 0;
			}
		}
		
		if (n != 0)
		{
			_TWDR = ADDRESS;
			_TWCR = 0b10000100;
			
			for (n = 512; !(_TWCR & 0b10000000) == true && n > 0; n--)
			{
			}
		}
		
		if (n != 0)
		{
			check = _TWSR & 0b11111000;
			
			if (check != 0b00101000)
			{
				n = 0;
			}
		}
		
		if (n != 0)
		{
			if (LENGTH < 0)
			{
				for (shift = 0; shift != -LENGTH && n != 0; shift += 8)
				{
					_TWDR = DATA >> shift;
					
					if (shift == -LENGTH - 8)
					{
						_TWCR = 0b10000100;
					}
					else
					{
						_TWCR = 0b11000100;
					}
					
					for (n = 512; !(_TWCR & 0b10000000) == true && n > 0; n--)
					{
					}
				}
			}
			else
			{
				for (shift = LENGTH; shift != 0 && n != 0; shift -= 8)
				{
					_TWDR = DATA >> (shift - 8);
					
					if (shift == 8)
					{
						_TWCR = 0b10000100;
					}
					else
					{
						_TWCR = 0b11000100;
					}
					
					for (n = 512; !(_TWCR & 0b10000000) == true && n > 0; n--)
					{
					}
				}
			}
		}
		
		if (n != 0)
		{
			check = _TWSR & 0b11111000;
			
			if (check != 0b00101000)
			{
				n = 0;
			}
		}
		
		if (n != 0)
		{
			_TWCR = 0b10010100;
			
			for (n = 512; !(_TWCR & 0b00010000) == true && n > 0; n--)
			{
			}
		}
		
		if (n == 0)
		{
			_TWCR = 0b00000000;
			error++;
		}
	}
}

unsigned char Core::readEeprom (const unsigned int ADDRESS)
{
	const unsigned char SREG_SAVE = _SREG;
	
	_SREG &= ~0b10000000;
	
	_EEARLH = ADDRESS;
	_EECR |= 0b00000001;
	
	while (!(_EECR & 0b00000010) == false)
	{
	}
	
	_SREG = SREG_SAVE;
	
	return _EEDR;
}

void Core::writeEeprom (const unsigned int ADDRESS, const unsigned char DATA)
{
	const unsigned char SREG_SAVE = _SREG;
	
	_SREG &= ~0b10000000;
	
	_EEARLH = ADDRESS;
	_EEDR = DATA;
	_EECR |= 0b00000100;
	_EECR |= 0b00000010;
	
	while (!(_EECR & 0b00000010) == false)
	{
	}
	
	_SREG = SREG_SAVE;
}

signed long Core::floor (const float VALUE)
{
	signed long value = 0;
	
	if (VALUE <= 0)
	{
		value = VALUE - 1.0;
	}
	else
	{
		value = VALUE;
	}
	
	return value;
}

signed long Core::round (const float VALUE)
{
	signed long value = 0;
	
	if (VALUE <= 0)
	{
		value = VALUE - 0.5;
	}
	else
	{
		value = VALUE + 0.5;
	}
	
	return value;
}

signed long Core::ceil (const float VALUE)
{
	signed long value = 0;
	
	if (VALUE <= 0)
	{
		value = VALUE;
	}
	else
	{
		value = VALUE + 1.0;
	}
	
	return value;
}

float Core::curve (const float POSITION_START, const float POSITION_CURRENT, const float POSITION_END, const float INTERPOLATION_START, const float INTERPOLATION_END, const unsigned char CURVE)
{
	float value = 0;
	
	if ((POSITION_START < POSITION_END && POSITION_CURRENT <= POSITION_START) || (POSITION_START > POSITION_END && POSITION_CURRENT >= POSITION_START))
	{
		value = INTERPOLATION_START;
	}
	else if ((POSITION_START < POSITION_END && POSITION_CURRENT >= POSITION_END) || (POSITION_START > POSITION_END && POSITION_CURRENT <= POSITION_END))
	{
		value = INTERPOLATION_END;
	}
	else
	{
		if (INTERPOLATION_END > INTERPOLATION_START)
		{
			if (CURVE == 0)
			{
				value = INTERPOLATION_START + ((INTERPOLATION_END - INTERPOLATION_START) * ((POSITION_CURRENT - POSITION_START) / (POSITION_END - POSITION_START)));
			}
			else if (CURVE == 1)
			{
				value = INTERPOLATION_START + (((INTERPOLATION_END - INTERPOLATION_START) * ((POSITION_CURRENT - POSITION_START) / (POSITION_END - POSITION_START))) * ((POSITION_CURRENT - POSITION_START) / (POSITION_END - POSITION_START)));
			}
			else if (CURVE == 2)
			{
				value = (INTERPOLATION_START + ((INTERPOLATION_END - INTERPOLATION_START) * ((POSITION_CURRENT - POSITION_START) / (POSITION_END - POSITION_START)))) + ((INTERPOLATION_END - (INTERPOLATION_START + ((INTERPOLATION_END - INTERPOLATION_START) * ((POSITION_CURRENT - POSITION_START) / (POSITION_END - POSITION_START))))) * ((POSITION_CURRENT - POSITION_START) / (POSITION_END - POSITION_START)));
			}
			else if (CURVE == 3)
			{
				if (POSITION_CURRENT - POSITION_START < (POSITION_END - POSITION_START) / 2.0)
				{
					value = (INTERPOLATION_START + (((INTERPOLATION_START + ((INTERPOLATION_END - INTERPOLATION_START) / 2.0)) - INTERPOLATION_START) / (((POSITION_END - POSITION_START) / 2.0) / (POSITION_CURRENT - POSITION_START)))) + (((INTERPOLATION_START + ((INTERPOLATION_END - INTERPOLATION_START) / 2.0)) - (INTERPOLATION_START + (((INTERPOLATION_START + ((INTERPOLATION_END - INTERPOLATION_START) / 2.0)) - INTERPOLATION_START) / (((POSITION_END - POSITION_START) / 2.0) / (POSITION_CURRENT - POSITION_START))))) / (((POSITION_END - POSITION_START) / 2.0) / (POSITION_CURRENT - POSITION_START)));
				}
				else if (POSITION_CURRENT - POSITION_START == (POSITION_END - POSITION_START) / 2.0)
				{
					value = INTERPOLATION_START + ((INTERPOLATION_END - INTERPOLATION_START) / 2.0);
				}
				else
				{
					value = (INTERPOLATION_START + ((INTERPOLATION_END - INTERPOLATION_START) / 2.0)) + (((INTERPOLATION_END - (INTERPOLATION_START + ((INTERPOLATION_END - INTERPOLATION_START) / 2.0))) / (((POSITION_END - POSITION_START) / 2.0) / ((POSITION_CURRENT - POSITION_START) - ((POSITION_END - POSITION_START) / 2.0)))) / (((POSITION_END - POSITION_START) / 2.0) / ((POSITION_CURRENT - POSITION_START) - ((POSITION_END - POSITION_START) / 2.0))));
				}
			}
			else if (CURVE == 4)
			{
				if (POSITION_CURRENT - POSITION_START < (POSITION_END - POSITION_START) / 2.0)
				{
					value = INTERPOLATION_START + (((INTERPOLATION_END - INTERPOLATION_START) / (((POSITION_END - POSITION_START) / 2.0) / (POSITION_CURRENT - POSITION_START))) * ((POSITION_CURRENT - POSITION_START) / (POSITION_END - POSITION_START)));
				}
				else if (POSITION_CURRENT - POSITION_START == (POSITION_END - POSITION_START) / 2.0)
				{
					value = INTERPOLATION_START + ((INTERPOLATION_END - INTERPOLATION_START) / 2.0);
				}
				else
				{
					value = (INTERPOLATION_START + ((INTERPOLATION_END - INTERPOLATION_START) / (((POSITION_END - POSITION_START) / 2.0) / ((POSITION_CURRENT - POSITION_START) - ((POSITION_END - POSITION_START) / 2.0))))) + ((INTERPOLATION_END - (INTERPOLATION_START + ((INTERPOLATION_END - INTERPOLATION_START) / (((POSITION_END - POSITION_START) / 2.0) / ((POSITION_CURRENT - POSITION_START) - ((POSITION_END - POSITION_START) / 2.0)))))) * ((POSITION_CURRENT - POSITION_START) / (POSITION_END - POSITION_START)));
				}
			}
		}
		else
		{
			if (CURVE == 0)
			{
				value = INTERPOLATION_START - ((INTERPOLATION_START - INTERPOLATION_END) * ((POSITION_CURRENT - POSITION_START) / (POSITION_END - POSITION_START)));
			}
			else if (CURVE == 1)
			{
				value = INTERPOLATION_START - (((INTERPOLATION_START - INTERPOLATION_END) * ((POSITION_CURRENT - POSITION_START) / (POSITION_END - POSITION_START))) * ((POSITION_CURRENT - POSITION_START) / (POSITION_END - POSITION_START)));
			}
			else if (CURVE == 2)
			{
				value = (INTERPOLATION_START - ((INTERPOLATION_START - INTERPOLATION_END) * ((POSITION_CURRENT - POSITION_START) / (POSITION_END - POSITION_START)))) - (((INTERPOLATION_START - ((INTERPOLATION_START - INTERPOLATION_END) * ((POSITION_CURRENT - POSITION_START) / (POSITION_END - POSITION_START)))) - INTERPOLATION_END) * ((POSITION_CURRENT - POSITION_START) / (POSITION_END - POSITION_START)));
			}
			else if (CURVE == 3)
			{
				if (POSITION_CURRENT - POSITION_START < (POSITION_END - POSITION_START) / 2.0)
				{
					value = (INTERPOLATION_START - ((INTERPOLATION_START - (INTERPOLATION_END + ((INTERPOLATION_START - INTERPOLATION_END) / 2.0))) / (((POSITION_END - POSITION_START) / 2.0) / (POSITION_CURRENT - POSITION_START)))) - (((INTERPOLATION_START - ((INTERPOLATION_START - (INTERPOLATION_END + ((INTERPOLATION_START - INTERPOLATION_END) / 2.0))) / (((POSITION_END - POSITION_START) / 2.0) / (POSITION_CURRENT - POSITION_START)))) - (INTERPOLATION_END + ((INTERPOLATION_START - INTERPOLATION_END) / 2.0))) / (((POSITION_END - POSITION_START) / 2.0) / (POSITION_CURRENT - POSITION_START)));
				}
				else if (POSITION_CURRENT - POSITION_START == (POSITION_END - POSITION_START) / 2.0)
				{
					value = INTERPOLATION_START - ((INTERPOLATION_START - INTERPOLATION_END) / 2.0);
				}
				else
				{
					value = (INTERPOLATION_START - ((INTERPOLATION_START - INTERPOLATION_END) / 2.0)) - ((((INTERPOLATION_START - ((INTERPOLATION_START - INTERPOLATION_END) / 2.0)) - INTERPOLATION_END) / (((POSITION_END - POSITION_START) / 2.0) / ((POSITION_CURRENT - POSITION_START) - ((POSITION_END - POSITION_START) / 2.0)))) / (((POSITION_END - POSITION_START) / 2.0) / ((POSITION_CURRENT - POSITION_START) - ((POSITION_END - POSITION_START) / 2.0))));
				}
			}
			else if (CURVE == 4)
			{
				if (POSITION_CURRENT - POSITION_START < (POSITION_END - POSITION_START) / 2.0)
				{
					value = INTERPOLATION_START - (((INTERPOLATION_START - INTERPOLATION_END) / (((POSITION_END - POSITION_START) / 2.0) / (POSITION_CURRENT - POSITION_START))) * ((POSITION_CURRENT - POSITION_START) / (POSITION_END - POSITION_START)));
				}
				else if (POSITION_CURRENT - POSITION_START == (POSITION_END - POSITION_START) / 2.0)
				{
					value = INTERPOLATION_START - ((INTERPOLATION_START - INTERPOLATION_END) / 2.0);
				}
				else
				{
					value = (INTERPOLATION_START - ((INTERPOLATION_START - INTERPOLATION_END) / (((POSITION_END - POSITION_START) / 2.0) / ((POSITION_CURRENT - POSITION_START) - ((POSITION_END - POSITION_START) / 2.0))))) - (((INTERPOLATION_START - ((INTERPOLATION_START - INTERPOLATION_END) / (((POSITION_END - POSITION_START) / 2.0) / ((POSITION_CURRENT - POSITION_START) - ((POSITION_END - POSITION_START) / 2.0))))) - INTERPOLATION_END) * ((POSITION_CURRENT - POSITION_START) / (POSITION_END - POSITION_START)));
				}
			}
		}
	}
	
	return value;
}

float Core::sqrt (const float SQUARED)
{
	unsigned char n = 0;
	float searchDecimal = 0.1;
	float foundDecimal = 0;
	unsigned long searchInteger = 1000;
	float sr1 = 0;
	float sr2 = 0;
	float sr3 = 0;
	float value = 0;
	
	if (SQUARED < 1)
	{
		while (searchDecimal * searchDecimal <= SQUARED)
		{
			searchDecimal += 0.1;
		}
		
		for (n = 0; n < 2; n++)
		{
			foundDecimal = ((SQUARED / searchDecimal) + searchDecimal) / 2.0;
			searchDecimal = foundDecimal;
		}
		
		value = foundDecimal;
	}
	else if (SQUARED == 1)
	{
		value = 1;
	}
	else
	{
		while (searchInteger * searchInteger <= SQUARED)
		{
			searchInteger += 1000;
		}
		
		searchInteger -= 1000;
		
		while (searchInteger * searchInteger <= SQUARED)
		{
			searchInteger += 100;
		}
		
		searchInteger -= 100;
		
		while (searchInteger * searchInteger <= SQUARED)
		{
			searchInteger += 10;
		}
		
		searchInteger -= 10;
		
		while (searchInteger * searchInteger <= SQUARED)
		{
			searchInteger++;
		}
		
		searchInteger--;
		
		sr1 = SQUARED - (float (searchInteger) * float (searchInteger));
		sr2 = sr1 / (2.0 * float (searchInteger));
		sr3 = float (searchInteger) + sr2;
		
		value = sr3 - ((sr2 * sr2) / (2.0 * sr3));
	}
	
	return value;
}

float Core::pow (const float NUMBER, const signed long EXPONENT)
{
	float value = 0;
	signed long n = 0;
	
	if (EXPONENT < 0)
	{
		value = NUMBER;
		
		for (n = -1; n > EXPONENT; n--)
		{
			value *= NUMBER;
		}
		
		value = 1.0 / value;
	}
	else if (EXPONENT == 0)
	{
		if (NUMBER < 0)
		{
			value = -1;
		}
		else
		{
			value = 1;
		}
	}
	else
	{
		value = NUMBER;
		
		for (n = 1; n < EXPONENT; n++)
		{
			value *= NUMBER;
		}
	}
	
	return value;
}

float Core::sin (const float ANGLE)
{
	float angle = 0;
	float alpha = 0;
	unsigned long n = 0;
	float sinus = 0;
	bool quickSearch = false;
	unsigned long foundMultiply = 0;
	float alpha2 = 0;
	float alpha3 = 0;
	float alpha5 = 0;
	
	if (ANGLE >= -180 && ANGLE <= 180)
	{
		angle = ANGLE;
	}
	else if (ANGLE < -180)
	{
		while (quickSearch == false)
		{
			for (n = 1; ANGLE + (360.0 * (float (foundMultiply) + float (n))) < -180; n *= 10)
			{
			}
			
			if (n == 1)
			{
				quickSearch = true;
			}
			else
			{
				foundMultiply += float (n) / 10.0;
			}
		}
		
		foundMultiply++;
		
		angle = ANGLE + (360.0 * float (foundMultiply));
	}
	else
	{
		while (quickSearch == false)
		{
			for (n = 1; ANGLE - (360.0 * (float (foundMultiply) + float (n))) > 180; n *= 10)
			{
			}
			
			if (n == 1)
			{
				quickSearch = true;
			}
			else
			{
				foundMultiply += float (n) / 10.0;
			}
		}
		
		foundMultiply++;
		
		angle = ANGLE - (360.0 * float (foundMultiply));
	}
	
	if (angle == -90 || angle == 270)
	{
		sinus = -1;
	}
	else if (angle == -270 || angle == 90)
	{
		sinus = 1;
	}
	else if (angle != -360 && angle != -180 && angle != 0 && angle != 180 && angle != 360)
	{
		alpha = angle * _DEGREE_TO_RADIAN;
		
		alpha2 = alpha * alpha;
		alpha3 = alpha2 * alpha;
		alpha5 = alpha2 * alpha3;
		
		sinus = alpha - (alpha3 / 6.0) + (alpha5 / 120.0) - ((alpha2 * alpha5) / 5040.0);
	}
	
	return sinus;
}

float Core::cos (const float ANGLE)
{
	float angle = 0;
	float alpha = 0;
	unsigned long n = 0;
	float cosinus = 0;
	bool quickSearch = false;
	unsigned long foundMultiply = 0;
	float alpha2 = 0;
	float alpha4 = 0;
	
	if (ANGLE >= -180 && ANGLE <= 180)
	{
		angle = ANGLE;
	}
	else if (ANGLE < -180)
	{
		while (quickSearch == false)
		{
			for (n = 1; ANGLE + (360.0 * (float (foundMultiply) + float (n))) < -180; n *= 10)
			{
			}
			
			if (n == 1)
			{
				quickSearch = true;
			}
			else
			{
				foundMultiply += float (n) / 10.0;
			}
		}
		
		foundMultiply++;
		
		angle = ANGLE + (360.0 * float (foundMultiply));
	}
	else
	{
		while (quickSearch == false)
		{
			for (n = 1; ANGLE - (360.0 * (float (foundMultiply) + float (n))) > 180; n *= 10)
			{
			}
			
			if (n == 1)
			{
				quickSearch = true;
			}
			else
			{
				foundMultiply += float (n) / 10.0;
			}
		}
		
		foundMultiply++;
		
		angle = ANGLE - (360.0 * float (foundMultiply));
	}
	
	if (angle == -180 || angle == 180)
	{
		cosinus = -1;
	}
	else if (angle == -360 || angle == 0 || angle == 360)
	{
		cosinus = 1;
	}
	else if (angle != -270 && angle != -90 && angle != 90 && angle != 270)
	{
		alpha = angle * _DEGREE_TO_RADIAN;
		
		alpha2 = alpha * alpha;
		alpha4 = alpha2 * alpha2;
		
		cosinus = 1.0 - ((alpha2 / 2.0) - (alpha4 / 24.0) + ((alpha2 * alpha4) / 720.0) - ((alpha4 * alpha4) / 40320.0));
	}
	
	return cosinus;
}

float Core::tan (const float ANGLE)
{
	float angle = 0;
	float alpha = 0;
	unsigned long n = 0;
	float tangent = 0;
	bool quickSearch = false;
	unsigned long foundMultiply = 0;
	float alpha2 = 0;
	float alpha3 = 0;
	float alpha5 = 0;
	
	if (ANGLE >= -90 && ANGLE <= 90)
	{
		angle = ANGLE;
	}
	else if (ANGLE < -90)
	{
		while (quickSearch == false)
		{
			for (n = 1; ANGLE + (180.0 * (float (foundMultiply) + float (n))) < -90; n *= 10)
			{
			}
			
			if (n == 1)
			{
				quickSearch = true;
			}
			else
			{
				foundMultiply += float (n) / 10.0;
			}
		}
		
		foundMultiply++;
		
		angle = ANGLE + (180.0 * float (foundMultiply));
	}
	else
	{
		while (quickSearch == false)
		{
			for (n = 1; ANGLE - (180.0 * (float (foundMultiply) + float (n))) > 90; n *= 10)
			{
			}
			
			if (n == 1)
			{
				quickSearch = true;
			}
			else
			{
				foundMultiply += float (n) / 10.0;
			}
		}
		
		foundMultiply++;
		
		angle = ANGLE - (180.0 * float (foundMultiply));
	}
	
	if (angle == -45)
	{
		tangent = -1;
	}
	else if (angle == -90 || angle == 0 || angle == 90)
	{
		tangent = 0;
	}
	else if (angle == 45)
	{
		tangent = 1;
	}
	else
	{
		if (angle > -45 && angle < 45)
		{
			alpha = angle * _DEGREE_TO_RADIAN;
		}
		else
		{
			if (angle < 0)
			{
				alpha = (-90.0 - angle) * _DEGREE_TO_RADIAN;
			}
			else
			{
				alpha = (90.0 - angle) * _DEGREE_TO_RADIAN;
			}
		}
		
		alpha2 = alpha * alpha;
		alpha3 = alpha2 * alpha;
		alpha5 = alpha2 * alpha3;
		
		if (angle > -45 && angle < 45)
		{
			tangent = alpha + ((alpha3 / 3.0) + ((2.0 * alpha5) / 15.0) + ((17.0 * (alpha5 * alpha2)) / 315.0));
		}
		else
		{
			tangent = 1.0 / (alpha + ((alpha3 / 3.0) + ((2.0 * alpha5) / 15.0) + ((17.0 * (alpha5 * alpha2)) / 315.0)));
		}
	}
	
	return tangent;
}

float Core::arcsin (const float SINUS)
{
	float angle = 10;
	float anglePrevious = 0;
	float increment = 10;
	float alpha = 0;
	float sinus = 0;
	float sinusPositive = 0;
	bool quickSearch = false;
	unsigned char decimal = 0;
	unsigned char error = 0;
	float alpha2 = 0;
	float alpha3 = 0;
	float alpha5 = 0;
	
	if (SINUS < 0)
	{
		sinusPositive = -SINUS;
	}
	else
	{
		sinusPositive = SINUS;
	}
	
	if (SINUS <= -1)
	{
		angle = -90;
	}
	else if (SINUS == 0)
	{
		angle = 0;
	}
	else if (SINUS >= 1)
	{
		angle = 90;
	}
	else
	{
		while (quickSearch == false)
		{
			alpha = angle * _DEGREE_TO_RADIAN;
			
			alpha2 = alpha * alpha;
			alpha3 = alpha2 * alpha;
			alpha5 = alpha2 * alpha3;
			
			sinus = alpha - (alpha3 / 6.0) + (alpha5 / 120.0) - ((alpha2 * alpha5) / 5040.0);
			
			if (sinus < sinusPositive && increment >= 1)
			{
				anglePrevious = angle;
				angle += increment;
			}
			else if (sinus < sinusPositive && increment < 1 && error < 9)
			{
				anglePrevious = angle;
				angle += increment;
				
				error++;
			}
			else if (sinus == sinusPositive || decimal == 4 + 1)
			{
				quickSearch = true;
			}
			else
			{
				increment /= 10;
				angle = anglePrevious + increment;
				
				decimal++;
				error = 0;
			}
			
			if (angle > 90.0001)
			{
				angle = 89.9999;
				quickSearch = true;
			}
		}
		
		if (SINUS < 0)
		{
			angle *= -1;
		}
	}
	
	return angle;
}

float Core::arccos (const float COSINUS)
{
	float angle = 0;
	float anglePrevious = 10;
	float increment = 10;
	float alpha = 0;
	float cosinus = 0;
	float cosinusPositive = 0;
	bool quickSearch = false;
	unsigned char decimal = 0;
	unsigned char error = 0;
	float alpha2 = 0;
	float alpha4 = 0;
	
	if (COSINUS < 0)
	{
		cosinusPositive = -COSINUS;
	}
	else
	{
		cosinusPositive = COSINUS;
	}
	
	if (COSINUS <= -1)
	{
		angle = 180;
	}
	else if (COSINUS == 0)
	{
		angle = 90;
	}
	else if (COSINUS >= 1)
	{
		angle = 0;
	}
	else
	{
		while (quickSearch == false)
		{
			alpha = angle * _DEGREE_TO_RADIAN;
			
			alpha2 = alpha * alpha;
			alpha4 = alpha2 * alpha2;
			
			cosinus = 1.0 - ((alpha2 / 2.0) - (alpha4 / 24.0) + ((alpha2 * alpha4) / 720.0) - ((alpha4 * alpha4) / 40320.0));
			
			if (cosinus > cosinusPositive && increment >= 1)
			{
				anglePrevious = angle;
				angle += increment;
			}
			else if (cosinus > cosinusPositive && increment < 1 && error < 9)
			{
				anglePrevious = angle;
				angle += increment;
				
				error++;
			}
			else if (cosinus == cosinusPositive || decimal == 4 + 1)
			{
				quickSearch = true;
			}
			else
			{
				increment /= 10;
				angle = anglePrevious + increment;
				
				decimal++;
				error = 0;
			}
			
			if (angle > 90.0001)
			{
				angle = 89.9999;
				quickSearch = true;
			}
		}
		
		if (COSINUS < 0)
		{
			angle = 180.0 - angle;
		}
	}
	
	return angle;
}

float Core::arctan (const float TANGENT)
{
	float angle = 10;
	float anglePrevious = 0;
	float increment = 10;
	float alpha = 0;
	float sinus = 0;
	float cosinus = 0;
	float tangent = 0;
	float tangentPositive = 0;
	bool quickSearch = false;
	unsigned char decimal = 0;
	unsigned char error = 0;
	float alpha2 = 0;
	float alpha3 = 0;
	float alpha4 = 0;
	float alpha5 = 0;
	
	if (TANGENT < 0)
	{
		tangentPositive = -TANGENT;
	}
	else
	{
		tangentPositive = TANGENT;
	}
	
	if (TANGENT <= -9999999999999999)
	{
		angle = -90;
	}
	else if (TANGENT == -1)
	{
		angle = -45;
	}
	else if (TANGENT == 0)
	{
		angle = 0;
	}
	else if (TANGENT == 1)
	{
		angle = 45;
	}
	else if (TANGENT >= 9999999999999999)
	{
		angle = 90;
	}
	else
	{
		while (quickSearch == false)
		{
			alpha = angle * _DEGREE_TO_RADIAN;
			
			alpha2 = alpha * alpha;
			alpha3 = alpha2 * alpha;
			alpha4 = alpha2 * alpha2;
			alpha5 = alpha2 * alpha3;
			
			sinus = alpha - (alpha3 / 6.0) + (alpha5 / 120.0) - ((alpha2 * alpha5) / 5040.0);
			cosinus = 1.0 - ((alpha2 / 2.0) - (alpha4 / 24.0) + ((alpha2 * alpha4) / 720.0) - ((alpha4 * alpha4) / 40320.0));
			tangent = sinus / cosinus;
			
			if (tangent < tangentPositive && increment >= 1)
			{
				anglePrevious = angle;
				angle += increment;
			}
			else if (tangent < tangentPositive && increment < 1 && error < 9)
			{
				anglePrevious = angle;
				angle += increment;
				
				error++;
			}
			else if (tangent == tangentPositive || decimal == 4 + 1)
			{
				quickSearch = true;
			}
			else
			{
				increment /= 10;
				angle = anglePrevious + increment;
				
				decimal++;
				error = 0;
			}
			
			if (angle > 90.0001)
			{
				angle = 89.9999;
				quickSearch = true;
			}
		}
		
		if (TANGENT < 0)
		{
			angle *= -1;
		}
	}
	
	return angle;
}

void Core::power (const bool POWER)
{
	if (POWER == false)
	{
		_PRR0 = 0b11111111;
		_SMCR = 0b00000101;
		
		_MCU_SLEEP();
	}
	else
	{
		_SMCR = 0b00000000;
		_PRR0 = 0b00000000;
	}
}
