/* 
 * File:   AD7793.c
 * Author: anthony
 *
 * Created on May 7, 2018, 7:53 PM
 */

#include <pic16f1777.h>
#include <string.h>
#include "AD7793.h"
#include "data.h"


static volatile uint8_t counter;
static volatile uint8_t dataReceived[BUF_SIZE];
static struct AppData* mAppData;


void AD7793_init(struct AppData* const appData){

	mAppData = appData;
	counter = 0;

	memset(dataReceived, 0, BUF_SIZE);
    
    //init SPI communication

    SDO = 0;
    SCK = 0;
    SDI = 1;


    SSP1CON1bits.CKP = 1; //idle state is high
    SSP1CON1bits.SSPM = 1; //SPI master mode, clock Fosc/16
    
    SSPSTATbits.CKE = 0; //if CKP = 1, data transmitted on rising edge of SCK
    SSPSTATbits.SMP = 1; //input data sampled at the end of data output time

    SSP1ADD = 10; //SCL frequency = Fosc / (4*(SSP1ADD+1)) = 200 kbauds
    
    SSP1CON1bits.SSPEN = 1; //enable SPI

    //end init SPI

    //init AD7793
    AD7793_config();
    
}

void AD7793_config(){

	uint8_t clearBuf;
	uint8_t data[2] = {0};
	clearBuf = SSPBUF;

//setup MODE register

	//setup communication register
	data[0] = 0b00001000;
	AD7793_sendData(data, 1);
	//end setup communication register

	//setup MODE register (16 bits) with 50Hz update rate, continuous mode conversion and internal clock
	data[0] = 0b00000101;
	AD7793_sendData(data, 2);
//end setup MODE register


//setup configuration register
	data[0] = 0b00010000; //setup communication register
	AD7793_sendData(data, 1);
	
	//setup configuration register (16 bits)
	data[0] = 0b00010000 | GAIN_ADC_AIN1; //first 8 bits
	data[1] = 0b10010000; //external reference, buffered mode, channel 1
	AD7793_sendData(data, 2); //first 8 bits sent first
//end setup configuration register


}

void AD7793_readADC(AD7793_CHANNEL channel) {

	//setup communication register to setup adc channel
	uint8_t config = 0b00010000;
	AD7793_sendData(&config, 1);

	uint8_t data[2] = {0};
	if(channel == AIN1){

		data[0] = GAIN_ADC_AIN1;
		data[1] = 0x80; //select channel 1 and internal reference
	}
	else{

		data[0] = GAIN_ADC_AIN2;
		data[1] = 0x81; //select channel 2 and internal reference
	}

	//read 24 bits (3 bytes) of adc in continuous read mode
	//send 24 bits of empty data is needed

	uint8_t dataRead[BUF_SIZE] = {0};

	//config communication register to read ADC data register
	config = 0b01011000;
	AD7793_sendData(&config, 1);

	//read ADC data register
	AD7793_readData(dataRead, BUF_SIZE);

	if(channel == AIN1)
		memcpy(&mAppData->AD7793_ADC_value, dataRead, BUF_SIZE);
	else{

		//calculate temperature
		float temperature;
		memcpy(&temperature, dataRead, BUF_SIZE);

		temperature = temperature * V_REF / (16777216*GAIN_ADC_AIN2); //2^24 = 16777216

		//LM35 -->10mV/°C
		temperature = temperature*1000 / 10; 

		mAppData->LM35_temperature = temperature;
	}
}

void AD7793_sendData(const uint8_t* data, const uint8_t len){

	//use this function when nothing expected to be read

	char clearBuf;
	clearBuf = SSPBUF;

	for(uint8_t i = 0; i < len; i++){
		SSPBUF = data[i];
		
		while(SSPSTATbits.BF != 1); //wait buffer completion

		clearBuf = SSPBUF;
	}
}


void AD7793_readData(uint8_t* dataOut, const uint8_t lenToRead){

	for(uint8_t i = 0; i < lenToRead; i++){
		SSPBUF = 0;
		
		while(SSPSTATbits.BF != 1); //wait buffer completion

		dataOut[i] = SSPBUF;
	}
}


void AD7793_reset(){
	
	const uint8_t data[] = {0xFF, 0xFF, 0xFF, 0x7F};
	AD7793_sendData(data, 4);
}