//Sample using LiquidCrystal library
#include <LiquidCrystal.h>

// make : PATH=$PATH:/usr/share/arduino/ make upload && cat /dev/ttyUSB0
 
/*******************************************************************

    Pin Assignments

    Pin Capabilities    Assignment
    
     0  RX
     1  TX
     2                  
     3  PWM             FAN_1 PWM
     4                  LCD D4
     5  PWM             LCD D5
     6  PWM             LCD D6
     7                  LCD D7
     8                  LCD RS
     9  PWM             LCD EN
    10  PWM             LCD Backlight : INPUT = ON, OUTPUT LOW = OFF, OUTPUT HIGH = Burn Transistor
    11  PWM             FAN_0 PWM
    12
    13                  onboard LED (active high)
    
    14  analog 0        LCD Button (analog value)
    15  analog 1        DHT11_0
    16  analog 2        DHT11_1
    17  analog 3        
    18  analog 4        
    19  analog 5        
    
*******************************************************************/

static const int PIN_LCD_BACKLIGHT  = 10;
static const int PIN_FAN1_PWM       = 3;
static const int PIN_FAN0_PWM       = 11;

static const float humidity_fan_start = 55.;
static const float humidity_fan_max   = 75.;
static const float fan_minimum_speed  = 0.3;

// humidity value (0-100) to fan speed (0-1)

float fan_func( float humi )
{
    float v = (humi-humidity_fan_start) / (humidity_fan_max-humidity_fan_start);
    
    // fan off ?
    if( v < 0. )
        return 0.;
    
    // if on, enforce minimum speed
    v = fan_minimum_speed + v * (1.-fan_minimum_speed);
    
    if( v > 1. )
        return 1.;
    
    return v;
}

 // select the pins used on the LCD panel
LiquidCrystal lcd(8, 9, 4, 5, 6, 7);
 
#define btn_RIGHT  0
#define btn_UP     1
#define btn_DOWN   2
#define btn_LEFT   3
#define btn_SELECT 4
#define btn_NONE   5


/*******************************************************************

    LCD, Buttons, LEDs...

*******************************************************************/
 
 
// read the buttons on LCD
int lcd_read_buttons()
{
    int adc_key_in = analogRead(0);      // read the value from the sensor 
    
    if (adc_key_in > 1000) return btn_NONE; // We make this the 1st option for speed reasons since it will be the most likely result
    if (adc_key_in < 50)   return btn_RIGHT;  
    if (adc_key_in < 250)  return btn_UP; 
    if (adc_key_in < 450)  return btn_DOWN; 
    if (adc_key_in < 650)  return btn_LEFT; 
    if (adc_key_in < 850)  return btn_SELECT;  

    return btn_NONE;  // when all others fail, return this...
}

// control LCD backlight
void lcd_backlight_set( bool on )
{
    digitalWrite( PIN_LCD_BACKLIGHT, 0 );
    pinMode( PIN_LCD_BACKLIGHT, on ? INPUT : OUTPUT );
}

// Control LED on pin 13
void led_set( bool on )
{
    pinMode( 13, OUTPUT );
    digitalWrite( 13, on );
}

/*******************************************************************

    Timeouts

*******************************************************************/

class Timeout
{
    public:

        long _remaining; // millis
        long _last_time;

        void start( long ms )
        {
            _remaining = ms;
            _last_time = millis();
        }

        void stop()
        {
            _remaining = 0;
        }

        bool finished()
        {
            if( _remaining <= 0 )
                return true;
            
            long t = millis();
            long elapsed = t-_last_time;    // time elapsed since last call
            _last_time = millis();
            _remaining -= elapsed;
            
            return _remaining <= 0;
        }
};

/*******************************************************************

    DHT 

*******************************************************************/

#define MAXTIMINGS 85

class DHT
{
    public:
        uint8_t data[6];
        uint8_t _pin, _count;
        unsigned long _lastreadtime;
        boolean firstreading;

bool read()
{
    unsigned long currenttime;

    if( firstreading )
    {
        // pull the pin high and wait 250 milliseconds
        digitalWrite(_pin, HIGH);
        delay(250);
    }

    currenttime = millis();
    if( currenttime < _lastreadtime ) 
    {
        // ie there was a rollover
        _lastreadtime = 0;
    }
    
    if( !firstreading && ((currenttime - _lastreadtime) < 2000) )
    {
        //~ Serial.print( firstreading );
        //~ Serial.print( " ");
        //~ Serial.print( currenttime );
        //~ Serial.print( " ");
        //~ Serial.print( _lastreadtime );
        //~ Serial.print( " ");
        //~ Serial.print( currenttime - _lastreadtime );
        //~ Serial.println( " not ready" );
        return false; // not ready
    }
    firstreading = false;

    _lastreadtime = millis();

    data[0] = data[1] = data[2] = data[3] = data[4] = 0;

    // now pull it low for ~20 milliseconds
    pinMode(_pin, OUTPUT);
    digitalWrite(_pin, LOW);
    delay(20);
    cli();
    digitalWrite(_pin, HIGH);
    delayMicroseconds(40);
    pinMode(_pin, INPUT);

    // read in timings
    uint8_t laststate = HIGH, state;
    uint8_t counter = 0;
    uint8_t pos = 0, i;
    for ( i=0; i< MAXTIMINGS; i++) 
    {
        counter = 0;
        while( (state = digitalRead(_pin)) == laststate) 
        {
            counter++;
            delayMicroseconds(1);
            if (counter == 255)
                goto end;
        }
        laststate = state;

        // ignore first 3 transitions
        if ((i >= 4) && (i%2 == 0)) 
        {
            // shove each bit into the storage bytes
            data[pos/8] = (data[pos/8]<<1) | (counter > _count);
            pos++;
        }
    }
end:

    sei();

    /*
    Serial.println(pos, DEC);
    Serial.print(data[0], HEX); Serial.print(", ");
    Serial.print(data[1], HEX); Serial.print(", ");
    Serial.print(data[2], HEX); Serial.print(", ");
    Serial.print(data[3], HEX); Serial.print(", ");
    Serial.print(data[4], HEX); Serial.print(" =? ");
    Serial.println(data[0] + data[1] + data[2] + data[3], HEX);
    */

    // check we read 40 bits and that the checksum matches
    if ((pos >= 40) && (data[4] == ((data[0] + data[1] + data[2] + data[3]) & 0xFF)) )
        return true;

    Serial.println( "bad checksum" );
    return false;
}
    
    
    public:
        
DHT(uint8_t pin, uint8_t count=6)
{
    _pin = pin;
    _count = count;
    firstreading = true;
}
            
void begin(void)
{
    // set up the pins!
    pinMode(_pin, INPUT);
    digitalWrite(_pin, HIGH);
    _lastreadtime = 0;
}

bool readAll( float& humi, float& temp )
{
        if( !read() )
            return false;
        
        temp = data[2];
        humi = data[0];
        return true;
}

};

DHT sensor_dht0( /*pin*/ 15 );
DHT sensor_dht1( /*pin*/ 16 );



/*******************************************************************

    Per-Room Code

*******************************************************************/

typedef enum { FAN_MODE_AUTO=0, FAN_MODE_TIME, FAN_MODE_ALWAYS_ON } FAN_MODE;
const char* FAN_MODE_STR[] = {"AUTO","TIME","ON"};

struct Room
{
    DHT*        _dht;
    byte        _room_id, _fan_pin;
    FAN_MODE    _fan_mode;
    Timeout     _fan_force_time;
    float       _humidity, _temperature, _fan_speed;
    
    
    void init( byte room_id, DHT* dht, byte fan_pin )
    {
        _room_id = room_id;
        _dht     = dht;
        _fan_pin = fan_pin;
        
        _humidity    = 0;
        _temperature = 0;
        _fan_mode    = FAN_MODE_AUTO;
        _fan_force_time.stop();
        
        fan_set( fan_minimum_speed );
    }
    
    void fan_set( float value /* 0-1 */ )
    {        
        pinMode( _fan_pin, OUTPUT );
        if( value < fan_minimum_speed )
        {
            _fan_speed = 0.;
            digitalWrite( _fan_pin, 0 );
        }
        else if( value >= 0.99 )
        {
            _fan_speed = 1.;
            digitalWrite( _fan_pin, 1 );
        }
        else
        {            
            _fan_speed = value;
            analogWrite( _fan_pin, value*255 );
        }
    }
    
    void run_fan()
    {
        switch( _fan_mode )
        {
            
            case FAN_MODE_ALWAYS_ON:
                fan_set( 1. );
                break;
            
            case FAN_MODE_TIME:
                if( !_fan_force_time.finished() )
                {
                    fan_set( 1. );
                    break;
                }
                else
                    _fan_mode = FAN_MODE_AUTO;
                    // fallthrough
                    
            case FAN_MODE_AUTO:
            default:
                fan_set( fan_func( _humidity ) );
                break;
        }
    }
    
    void run( bool read=true )
    {
        //~ Serial.print( "read sensor\t" ); 
        //~ Serial.print( _room_id ); 
        //~ Serial.print( ":\t" ); 
        
        if( read )
            if( !_dht->readAll( _humidity, _temperature ) )
                return;

        // setup fan
        run_fan();
        
        // display stuff
        print_serial();
        print_lcd();
    }
    
    void print_serial()
    {
        Serial.print( "ROOM:\t" ); 
        Serial.print( _room_id );
        Serial.print( "\tHR%:\t" ); 
        Serial.print( _humidity );
        Serial.print( "\tTC:\t" ); 
        Serial.print( _temperature );    
        Serial.print( "\tFAN:\t" ); 
        Serial.print( _fan_speed*100. );    
        Serial.print( "\t" );
        Serial.print( FAN_MODE_STR[ _fan_mode ] );
        Serial.print( "\t" );
        Serial.println( _fan_force_time._remaining/1000 );    
    }
    
    void print_lcd()
    {
        lcd.setCursor(0,_room_id);
        lcd.print( "    " );
        lcd.setCursor(0,_room_id);
        lcd.print( (int)_humidity );
        lcd.print( "% " );
        
        lcd.setCursor(4,_room_id);
        lcd.print( "            " );
        lcd.setCursor(4,_room_id);
        
        switch( _fan_mode )
        {
            case FAN_MODE_TIME:
                {
                    int t = _fan_force_time._remaining/1000;
                    lcd.print( t / 3600 );
                    lcd.print( ":" );
                    lcd.print( (t / 60)%60 );
                    lcd.print( ":" );
                    lcd.print( t%60 );
                }
                break;
            
            case FAN_MODE_ALWAYS_ON:
                lcd.print( "force." );
                break;
            
            default:
                lcd.print( "auto " );
                if( _fan_speed )
                {
                    lcd.print( (int)(_fan_speed*100.0) );
                    lcd.print( "%" );
                }
                else
                    lcd.print( "OFF " );
                break;
        }
    }
    
    void button()
    {
        switch( _fan_mode )
        {
            case FAN_MODE_TIME:
                if( _fan_force_time._remaining < 59*60000 )
                    _fan_force_time.start( 60*60000 );
                else if( _fan_force_time._remaining < 119*60000 )
                    _fan_force_time.start( 120*60000 );
                else if( _fan_force_time._remaining < 239*60000 )
                    _fan_force_time.start( 240*60000 );
                else if( _fan_force_time._remaining < 479*60000 )
                    _fan_force_time.start( 480*60000 );
                else
                    _fan_mode = FAN_MODE_ALWAYS_ON;
                break;
            
            case FAN_MODE_ALWAYS_ON:
                _fan_mode = FAN_MODE_AUTO;
                break;
            
            default:
                _fan_mode = FAN_MODE_TIME;
                _fan_force_time.start(60*60000);
                break;
        }
        run( false );
    }
};

Room rooms[2];

/*******************************************************************

    Main code

*******************************************************************/

Timeout lcd_timeout;
void light()
{
    lcd_backlight_set( 1 );
    lcd_timeout.start( 30000 );
}

void setup()
{
    // init serial port
    Serial.begin(9600); 
    Serial.println("# Humidity fan control");
    
    // init lcd
    lcd.begin(16, 2);              // start the library
    lcd.setCursor(0,0);
    lcd.print("v1.0"); // print a simple message
    
    // init sensors
    sensor_dht0.begin();
    sensor_dht1.begin();
    
    // init room states
    rooms[0].init( 0, &sensor_dht0, PIN_FAN0_PWM );
    rooms[1].init( 1, &sensor_dht1, PIN_FAN1_PWM );
    
    light();
}

static bool wait_for_button_release = false;

void loop()
{    
    if( lcd_timeout.finished() )
        lcd_backlight_set( 0 );
    
    rooms[1].run();
    rooms[0].run();
        
    // debouncing : act only if we have a stable reading for a certain time
    int debounce;
    int lcd_key = lcd_read_buttons();
    int lcd_key_last = btn_NONE;

    // do not autofire button presses
    if( lcd_key == btn_NONE )
    {
        wait_for_button_release = false;
    }
    else if( wait_for_button_release )
        return;
    
    for( debounce=10; debounce>0; debounce-- )
    {
        lcd_key = lcd_read_buttons();  // read the buttons
        if( lcd_key == btn_NONE )
            break;
        
        if( lcd_key_last == btn_NONE )
            lcd_key_last = lcd_key;
        else if( lcd_key_last != lcd_key )
            break;
    }
        
    if( debounce==0 )
    {
        wait_for_button_release = true;
        
        Serial.print( "Button:" );
        Serial.println( lcd_key );
        
        
        switch (lcd_key)               // depending on which button was pushed, we perform an action
        {
            case btn_RIGHT:
            {
                light();
                break;
            }
            case btn_LEFT:
            {
                light();
                break;
            }
            case btn_UP:
            {
                light();
                rooms[0].button();
                break;
            }
            case btn_DOWN:
            {
                light();
                rooms[1].button();
                break;
            }
            case btn_SELECT:
            {
                light();
                break;
            }
            case btn_NONE:
            {
                break;
            }
        }
    }
}
    











