; *********************************************************
; * TV_sync processor                                     *
; * for ATtiny45 on external clock                        *
; * Version 1.1 by JRV  03/22                             *
; *********************************************************
;
.INCLUDE "tn45def.inc"
;
;====================================================================================================================
;                                           Hardware connections
;====================================================================================================================
;                                              ___   ___
;                  (PCINT5/RESET/ADC0/dW) PB5 |1  |_|  8| VCC
;   16MHz  (PCINT3/XTAL1/CLKI/#OC1B/ADC3) PB3 |2   A   7| PB2 (SCK/USCK/SCL/ADC1/T0/INT0/PCINT2)        SYNC_IN
;   XTAL    (PCINT4/XTAL2/CLKO/OC1B/ADC2) PB4 |3   T   6| PB1 (MISO/DO/AIN1/OC0B/OC1A/PCINT1)           SYNC_OUT
;                                         GND |4   M   5| PB0 (MOSI/DI/SDA/AIN0/OC0A/#OC1A/AREF/PCINT0) SYNC_SW
;                                             |_________|
;                            
;
;====================================================================================================================
;                                           Constants definitions
;====================================================================================================================
;
;define processing clock frequency
#define _FREQ_ACT  16;in MHz
#define _FREQ_REG  8;in MHz

;define usefull constants
.EQU _NB_LINE                 = 625;
.EQU _RESET                   = 5; port B
.EQU _SYNC_IN                 = 2; port B sync from extractor
.EQU _SYNC_OUT                = 1; port B sync generator output
.EQU _SYNC_SW                 = 0; port B switch control 0 = internal 1 = external
.EQU _POL                     = 0; sync polarity 0 = neg 1 = pos

;port configuration
.EQU _DDRB                    = 1<<_SYNC_OUT|1<<_SYNC_SW;
.IF _POL == 0
.EQU _PORTB                   = 1<<_RESET|1<<_SYNC_OUT; pull up 
.ELSE
.EQU _PORTB                   = 1<<_RESET; pull up 
.ENDIF

;Timer0 configuration
.EQU _MODE0                   = 7; fast PWM 
.EQU _CLK0                    = 0<<CS02|1<< CS01|0<< CS00; divide by 8 clock is 2MHz
.EQU _TCCR0A                  = 0<<COM0A1|0<<COM0A0|0<<COM0B1|0<<COM0B0|(_MODE0 & 0x3); 
.EQU _TCCR0B0                 = 0<<FOC0A|0<<FOC0B|(_MODE0 & 4)<< 1; clock off
.EQU _TCCR0B1                 = 0<<FOC0A|0<<FOC0B|(_MODE0 & 4)<< 1|_CLK0; clock on
.EQU _OCR0A                   = 45; 30µS from start
.EQU _OCR0B                   = 10; 10µS from start

;Timer1 configuration  
.EQU _MODE1                   = 0<< CTC1|1<< PWM1A; pwm & ctc
.EQU _CLK1                    = 0<< CS13|0<< CS12|1<< CS11|0<< CS10 ;divide by 2
.IF _POL == 0
.EQU _TCCR10                  = 1<<COM1A1|1<<COM1A0|_MODE1|_CLK1; active
.ELSE
.EQU _TCCR10                  = 1<<COM1A1|0<<COM1A0|_MODE1|_CLK1; active
.ENDIF
.EQU _TCCR11                  = 0<<COM1A1|0<<COM1A0|_MODE1|_CLK1; inactive
.EQU _GTCCR0                  = 0<< TSM|1<< PWM1B|0<< COM1B1|0<< COM1B0|0<< FOC1B|0<< FOC1A|0<< PSR1|0<< PSR0;
.EQU _GTCCR1                  = 1<< TSM|1<< PWM1B|0<< COM1B1|0<< COM1B0|0<< FOC1B|0<< FOC1A|0<< PSR1|1<< PSR0;
.EQU _OCR1A0                  = 218;  equalizing pulse wide
.EQU _OCR1A1                  = 19;   equalizing pulse narrow
.EQU _OCR1A2                  = 38;   sync pulse
.EQU _OCR1B                   = 128;  interrupt for timing generator
.EQU _OCR1C                   = 255;  => 31250 Hz

;common values
.EQU _TIMSK                   = 0<< OCIE1A|1<< OCIE1B|1<< OCIE0A|1<< OCIE0B|0<< TOIE1|0<< TOIE0; 
.EQU _TIFR                    = 0<< OCF1A |1<< OCF1B |1<< OCF0A |1<< OCF0B |0<< TOV1 |0<< TOV0;

;interrupt
.EQU _GIMSK                   = 1<<INT0;
.EQU _GIFR                    = 1<<INTF0;
.EQU _MCUCR                   = 1<<ISC01|0<<ISC00; sensing of external composite sync on falling edge

;====================================================================================================================
;                                          Register definitions
;====================================================================================================================
;

.DEF rmp0                    = R16 ;  Multi-purpose register 
.DEF rmp1                    = R17 ;  Multi-purpose register 
.DEF rmp2                    = R18 ;  Multi-purpose register 
.DEF rmp3                    = R19 ;  Multi-purpose register 
.DEF rmp4                    = R20 ;  Multi-purpose register 
.DEF rmp5                    = R21;   Multi-purpose register 
.DEF rmp6                    = R22 ;  Multi-purpose register 
.DEF rmp7                    = R23 ;  Multi-purpose register 
.DEF rDelL                   = R24 ;  delay counter LSB
.DEF rDelH                   = R25 ;  delay counter MSB
            
.DEF rcp0                    = R0  ;  computation registers
.DEF rcp1                    = R1  ;  computation registers
.DEF rcp2                    = R2  ;  computation registers
.DEF rcp3                    = R3  ;  computation registers
.DEF rcp4                    = R4  ;  computation registers
.DEF rcp5                    = R5  ;  computation registers
.DEF rcp6                    = R6  ;  computation registers
.DEF rcp7                    = R7  ;  computation registers
.DEF rcp8                    = R8  ;  computation registers
.DEF rcp9                    = R9  ;  computation registers
.DEF rcp10                   = R10 ;  computation registers
.DEF rcp11                   = R11 ;  computation registers
.DEF rcp12                   = R12 ;  computation registers
.DEF rcp13                   = R13 ;  computation registers
.DEF rcp14                   = R14 ;  computation registers
.DEF rcp15                   = R15 ;  computation registers

;====================================================================================================================
;                                           Macro  definitions
;====================================================================================================================

.MACRO xpush0
    push rmp0; 
    in rmp0,SREG ; 
    push rmp0; 
.ENDMACRO

.MACRO xpop0
    pop rmp0; 
    out SREG,rmp0;  
    pop rmp0; 
.ENDMACRO

.MACRO xdelay ;max is 65536µs*FREQ_REG/FREQ_ACT
    ldi rDelH,HIGH((@0*_FREQ_ACT)/_FREQ_REG);
    ldi rDelL,LOW((@0*_FREQ_ACT)/_FREQ_REG);
    rcall DelayZ;
.ENDMACRO



;====================================================================================================================
;                                                  SRAM definitions
;====================================================================================================================
.DSEG
.ORG Sram_Start
sField                       : .BYTE 1; field number
sFrame                       : .BYTE 1; frame sync
sFlag1                       : .BYTE 1; sync flag for field extraction
sFlag2                       : .BYTE 1; sync flag for timing generator synchronization
sState                       : .BYTE 1; timing machine state 
sMono                        : .BYTE 1; > 32µS monostable flag
sDetect                      : .BYTE 1; line detect
sLockedl                     : .BYTE 2; line lock
sLockedf                     : .BYTE 1; frame lock
sWindowf                     : .BYTE 1; capture window for frame sync

;
.ESEG
.ORG $0000



;====================================================================================================================
;                               Reset and Interrupt Vectors starting here
;====================================================================================================================
;
.CSEG
.ORG $0000
;
; Reset/Interrupt vectors
;
  rjmp RESET ; Reset
  rjmp INT_0;reti ; INT_0;
  reti ; PCINT;
  reti ; TIMER1_COMPA; Timer/Counter1 Compare Match A
  reti ; TIMER1_OVF
  reti ; TIMER0_OVF;
  reti ; EE_RDY EEPROM; Ready
  reti ; ANA_COMP; Analog Comparator
  reti ; ADC_INT; ADC Conversion Complete
  rjmp TIMER1_COMPB;reti ; TIMER1_COMPB; Timer/Counter1 Compare Match B
  rjmp TIMER0_COMPA;reti ; TIMER0_COMPA;Timer/Counter0 Compare Match A
  rjmp TIMER0_COMPB;reti ; TIMER0_COMPB; Timer/Counter0 Compare Match B
  reti ; WDT; Watchdog Time-out
  reti ; USI_START; USI START
  reti ; USI_OVF;
  
;
;====================================================================================================================
;                                     Interrupt Service Routines
;====================================================================================================================
;===================================================================================================================
;                                                   horizontal sync processor
;===================================================================================================================
  
  INT_0:;external sync proccessing use  XL:XH,rcp4 only XL & SREG saved
    in XL,SREG ; 
    push XL; 
    ldi XL,14; value to lock horizontal counter if unlocked
    lds XH,sLockedl;
    tst XH;
  breq set_sync; -1
    in XL,TCNT1;+1
    cpi XL,6;+1
  brlo ignore;+1
    cpi XL,10;+1
  brsh ignore;+1
    ldi XL,17;+1        value to lock horizontal counter if locked
  set_sync:
    inc XH; count good lines
    sbrs XH,7;
    sts sLockedl,XH;
    sbrc XH,7;
    sts sLockedl+1,XH; 128 consecutives good lines
    in XH,GTCCR;
    ori XH,1<< PSR1; reset prescaler
    out GTCCR,XH;
    out TCNT1,XL; load horizontal counter
    lds XL,sMono; output of 40µS monostable
    lds XH,sFlag1; 
    sbrs XH,_SYNC_IN; skip next instruction if falling edge of vsync
  rjmp set_mono;
    and XH,XL;
    sts sField,XH; set field according to monostable output
    sts sFlag2,XH; flag for timing generator resync ,1 if rising edge of field 
    sbrc XH,_SYNC_IN; skip next instruction not vertical resync
    sts sLockedf,XH;
  ignore:
    clr XH;
    sts sFlag1,XH; erase resync flag
    ;cbi PORTB,_SYNC_SW; switch on incomming video
  set_mono:  
    lds XL,sMono; output of 40µS monostable
    sbrs XL,0; skip if monostable still active, this removes equalizing pulses
    com XL;
    sts sMono,XL;
    ldi XL,_GTCCR1; activate 10µS monostable
    out GTCCR,XL;
    clr XL;
    out TCNT0,XL; clear counter
    ldi XL,1<< OCF0A|1<< OCF0B;
    out TIFR,XL;
    ldi XL,_TCCR0B1;
    out TCCR0B ,XL; start counting
    ldi XL,_GTCCR0;
    out GTCCR,XL;
    ldi XL,0x07;
    mov rcp4,XL; flag to detect loss of vidéo
;    sts sDetect,XL; to detect loss of vidéo
    clr XL;
    out GIMSK,XL; kill interrupt
    pop XL; 
    out SREG,XL ; 
  reti;
  
;===================================================================================================================
;                                                   horizontal sync filter
;===================================================================================================================
   
  TIMER0_COMPA:; monostable 30µs to filter sync_in;
    xpush0;
    ldi rmp0,_TCCR0B0;
    out TCCR0B,rmp0; stop counting
    ldi rmp0,_GIMSK;
    out GIMSK,rmp0;
    ldi rmp0,_GIFR;
    out GIFR,rmp0; réactivate interrupt
    xpop0;
  reti;
;===================================================================================================================
;                                                   vertical sync extractor
;===================================================================================================================
  
  TIMER0_COMPB:; monostable 10µs to detect Vsync use rmp0, rcp0..1 only rmp0 & SREG are saved
    xpush0;
    in rmp0,PINB;
    andi rmp0,1<<_SYNC_IN; check state of _SYNC_IN when monostable elapse if 0 then Vsync occurs
  brne set_frame; not frame sync
    lds rcp0,sFrame;
    tst rcp0;
  breq set_frame; already in frame sync
    lds rcp1,sWindowf; check if in synchronisation allowed window
    tst rcp1;
  breq set_frame; no
    lds rcp1,sLockedf; check status of frame synchronization 
    tst rcp1;
  breq set_flag1; first sync
    cpi YH,0
  brne set_frame;
    cpi YL,2;
  brsh set_frame;
    sbi PORTB,_SYNC_SW; switch on incomming video
  set_flag1:
    eor rcp0,rmp0; detect falling edge
    sts sFlag1,rcp0; set flag for resync on next sync pulse
  set_frame:  
    sts sFrame,rmp0;memorize state of Vsync
    xpop0;  
  reti;
  
;===================================================================================================================
;                                                   sync generator
;===================================================================================================================

  TIMER1_COMPB: ; video timing generator use rmp0, YL:YH,ZL,ZH,rcp2,rcp3,rcp4 ,registers only rmp0 & SREG are saved
    xpush0;                                     
    sei; reactivate interrupt to maintain stable timing 
    lds rmp0,sFlag2; test resync flag
    tst rmp0;
  breq no_sync;
    ldi rmp0,_TCCR10; reactivate OC1A output                         
    out TCCR1,rmp0;                               
    ldi rmp0,_OCR1A0;
    out OCR1A,rmp0; update pulse size             
    ldi YL,2;  update half line counter                                   
    clr YH; 
    clr rmp0;
    sts sWindowf,rmp0; close capture window after sync done
    sts sFlag2,rmp0; erase flag
 rjmp no_change;
  no_sync:
    lds rmp0,sState;  get state of timing generator                             
    ldi ZL,LOW(2*tab_line);  base address of timing parameters table                     
    ldi ZH,HIGH(2*tab_line);                      
    mov rcp2,rmp0;                                
    lsl rcp2;                                     
    lsl rcp2; 2 words by entry                    
    clr rcp3;                                     
    add ZL,rcp2;                                  
    adc ZH,rcp3;                                  
    lpm rcp2,Z+;  read current boundary                                
    lpm rcp3,Z+;                                  
    cp YL,rcp2;                                   
    cpc YH,rcp3;                                  
  brlo no_change; boundary not reached                                    
    lpm rcp2,Z+;  read next pulse size                                
    out OCR1A,rcp2; update pulse size             
    inc rmp0; update state                                    
    andi rmp0,7; limit to 7                                 
  brne no_change;                                 
    clr YL; end of picture reinit counter         
    clr YH;  
  no_change:
    sts sState,rmp0;
    cpi rmp0,7;
  brne next;
    cpi YL,LOW(_NB_LINE*2-1); capture window activation one line before Vsync
  brne next;
    lds rcp3,sWindowf;
    tst rcp3;
  breq set_window;
    clr rcp3;  if window still opened then unlock capture process
    sts sLockedf,rcp3;
  set_window:
    lds rcp2,sLockedl+1;
    sts sWindowf,rcp2; set vertical sync window if horizontal sync done
  next:
    cpi rmp0,2;                                   
  breq switch; first field active video period ,supress one pulse out of two                                   
    cpi rmp0,6;                                   
  breq switch; second field active video period , supress one pulse out of two                                     
    cpi rmp0,3;                                   
  breq switch1;                                   
  rjmp update_line;                               
  switch:
    sbrc YL,0;                                    
    ldi rmp0,_TCCR11; off                         
    sbrs YL,0;                                    
  switch1:  
    ldi rmp0,_TCCR10; on                          
    out TCCR1,rmp0;                               
  update_line:                                    
    adiw YH:YL,1;  update line count
    lsr rcp4;
  brne end_timer1_compb;
    cbi PORTB,_SYNC_SW; switch on incomming video
    sts sLockedl,rmp0;
    sts sLockedl+1,rmp0;
    ldi rmp0,_GIMSK;
    out GIMSK,rmp0; réactivate interrupt
    ldi rmp0,_GIFR;
    out GIFR,rmp0; réactivate interrupt
  end_timer1_compb:  
    xpop0;                                      
  reti;
  
;===================================================================================================================
;                                   Delays execution for 1 microseconds @ 8MHz
;===================================================================================================================
  DelayZ:
    nop;1
    nop;1
    nop;1
    nop;1
    sbiw rDelL,1 ; 2
    brne DelayZ ; 1/2
  ret;
  
  
;===================================================================================================================
; Main Program Start
;===================================================================================================================

RESET:
    ; init ports
    ldi rmp0,_PORTB;
    out PORTB,rmp0;
    ldi rmp0,_DDRB;
    out DDRB,rmp0;
    
    ; set stack pointer
    ldi ZH,HIGH(RAMEND);
    out SPH,ZH;
    ldi ZL,LOW(RAMEND);
    out SPL,ZL;
    
    ; init whole memory to zero
    ldi YH,HIGH(SRAM_START); Y point ram location
    ldi YL,LOW(SRAM_START);
    ldi rmp0,0;
  init_ram:
    st Y+,rmp0
    cp YL,ZL;
    cpc YH,ZH;
  brne init_ram;
  
    ldi YL,1; init line counter
    clr YH;
    
    ;init Timer0
    ldi rmp0,_TCCR0A; 
    out TCCR0A,rmp0;
    ldi rmp0,_TCCR0B0;
    out TCCR0B,rmp0;
    ldi rmp0,_OCR0A;
    out OCR0A,rmp0;
    ldi rmp0,_OCR0B;
    out OCR0B,rmp0;
    
    ;init Timer1
    ldi rmp0,_TCCR10; active
    out TCCR1,rmp0;
    ldi rmp0,_OCR1A0; equalizing pulse wide
    out OCR1A,rmp0;
    ldi rmp0,_OCR1B;
    out OCR1B,rmp0;
    ldi rmp0,_OCR1C;
    out OCR1C,rmp0;
    
  ;common
    ldi rmp0,_GTCCR0;
    out GTCCR,rmp0;
  
   ;init interrupts
    ldi rmp0,_TIFR;
    out TIFR,rmp0;
    ldi rmp0,_TIMSK;
    out TIMSK,rmp0;
    ldi rmp0,_MCUCR;
    out MCUCR,rmp0;
    ldi rmp0,0;
    out GIMSK,rmp0; external interrupt disabled
    ldi rmp0,_GIFR;
    out GIFR,rmp0;
    
  ; enable interrupt 
    sei;
  ; Here is the infinite loop
  loop_main:
    lds rmp0,sMono;
    sbrs rmp0,0;
  rjmp loop_main;
    xdelay 32; equalization pulses filter
    clr rmp0;
    sts sMono,rmp0;
  rjmp loop_main;
  
  
  ; tab_line is the timing parameters table, two values by state: first one is line count limit for current state,second is next state pulse width
  ;.DW 5,_OCR1A1,10,_OCR1A2,_NB_LINE-5,_OCR1A1 ,_NB_LINE,_OCR1A0,_NB_LINE+5,_OCR1A1,_NB_LINE+10,_OCR1A2,_NB_LINE*2-5,_OCR1A1 ,_NB_LINE*2,_OCR1A0;
  tab_line:
  ;state 0 = first field Vsync
  .DW 5,_OCR1A1;
  ;state 1 = first field post equalization
  .DW 10,_OCR1A2;
  ;state 2 = first field active video  
  .DW _NB_LINE-5,_OCR1A1;
  ;state 3 = second field pre equalization
  .DW _NB_LINE,_OCR1A0;
  ;state 4 = second field Vsync
  .DW _NB_LINE+5,_OCR1A1;
  ;state 5 = second field post equalization
  .DW _NB_LINE+10,_OCR1A2;
  ;state 6 = second field active video
  .DW _NB_LINE*2-5,_OCR1A1;
  ;state 7 = second field pre equalization
  .DW _NB_LINE*2,_OCR1A0;
; End of source code
