; *********************************************************
; * cheap sinus generator                                       *
; * for ATtiny45 on internal clock                        *
; * Version 1.0 by JRV                                    *
; *********************************************************
;
.INCLUDE "tn45def.inc"
.

;====================================================================================================================
;           Hardware connections
;====================================================================================================================
;                                              ___   ___
;                  (PCINT5/RESET/ADC0/dW) PB5 |1  |_|  8| VCC
;    LED   (PCINT3/XTAL1/CLKI/#OC1B/ADC3) PB3 |2   A   7| PB2 (SCK/USCK/SCL/ADC1/T0/INT0/PCINT2)        SWC
;    PWM    (PCINT4/XTAL2/CLKO/OC1B/ADC2) PB4 |3   T   6| PB1 (MISO/DO/AIN1/OC0B/OC1A/PCINT1)           SWB
;                                         GND |4   M   5| PB0 (MOSI/DI/SDA/AIN0/OC0A/#OC1A/AREF/PCINT0) SWA
;                                             |_________|
;                            
;
;====================================================================================================================
;            constant definitions
;====================================================================================================================
;
;define processing clock frequency
#define _FREQ_ACT  8;in MHz
#define _FREQ_REG  8;in MHz
;define usefull constants
.EQU _RESET                   = 5; port B
.EQU _PWM                     = 4; port B
.EQU _LED                     = 3; port B
.EQU _SWC                     = 2; port B
.EQU _SWB                     = 1; port B
.EQU _SWA                     = 0; port B
;port configuration
.EQU _DDRB                    = 1<<_PWM|1<<_LED; 
.EQU _PORTB                   = 1<<_RESET; pull up 

;Timer0 configuration
.EQU _MODE0                   = 0<<WGM02|1<<WGM01|1<<WGM00; fast PWM
.EQU _CLK0                    = 1<<CS02|0<< CS01|0<< CS00; divide by 1024
.EQU _TCCR0A                  = 0<<COM0A1|0<<COM0A0|0<<COM0B1|0<<COM0B0|(_MODE0 & 0x3); 
.EQU _TCCR0B                  = 0<<FOC0A|0<<FOC0B|(_MODE0 & 4)<<1|_CLK0; 
.EQU _OCR0A                   = 10; 
.EQU _OCR0B                   = 0;

;Timer1 configuration  
.EQU _MODE1                   = 0<< CTC1|0<< PWM1A; pwm
.EQU _CLK1                    = 0<< CS13|0<< CS12|0<< CS11|1<< CS10 ;divide by 1
.EQU _TCCR1                   = 0<<COM1A1|0<<COM1A0|_MODE1|_CLK1; 
.EQU _GTCCR                   = 0<< TSM|1<< PWM1B|1<< COM1B1|0<< COM1B0|0<< FOC1B|0<< FOC1A|0<< PSR1|0<< PSR0;
.EQU _OCR1A                   = 64;50% 
.EQU _OCR1B                   = 64;50%
.EQU _OCR1C                   = 127; => 500Khz

;common values
.EQU _TIMSK                   = 0<< OCIE1A|0<< OCIE1B|0<< OCIE0A|0<< OCIE0B|1<< TOIE1|0<< TOIE0; 
.EQU _TIFR                    = 0<< OCF1A |0<< OCF1B|0<< OCF0A|0<< OCF0B|1<< TOV1|0<< TOV0;
.EQU _PLLCSR                  = 1<<PCKE|1<<PLLE;
;external interrupts
.EQU _GIMSK                   = 1<<PCIE;
.EQU _GIFR                    = 1<<PCIF;
.EQU _PCMSK                   = 1<<PCINT2|1<<PCINT0;
;====================================================================================================================
;                                          Register definitions
;====================================================================================================================
;

.DEF rmp0          = R16 ;  Multi-purpose register 
.DEF rmp1          = R17 ;  Multi-purpose register 
.DEF rmp2          = R18 ;  Multi-purpose register 
.DEF rmp3          = R19 ;  Multi-purpose register 
.DEF rmp4          = R20 ;  Multi-purpose register 
.DEF rmp5          = R21;   Multi-purpose register 
.DEF rmp6          = R22 ;  Multi-purpose register 
.DEF rmp7          = R23 ;  Multi-purpose register 
.DEF rDelL         = R24 ;  delay counter LSB
.DEF rDelH         = R25 ;  delay counter MSB

.DEF rcp0          = R0  ;  computation registers
.DEF rcp1          = R1  ;  computation registers
.DEF rcp2          = R2  ;  computation registers
.DEF rcp3          = R3  ;  computation registers
.DEF rcp4          = R4  ;  computation registers
.DEF rcp5          = R5  ;  computation registers
.DEF rcp6          = R6  ;  computation registers
.DEF rcp7          = R7  ;  computation registers
.DEF rcp8          = R8  ;  computation registers
.DEF rcp9          = R9  ;  computation registers
.DEF rcp10         = R10 ;  computation registers
.DEF rcp11         = R11 ;  computation registers
.DEF rcp12         = R12 ;  computation registers
.DEF rcp13         = R13 ;  computation registers
.DEF rcp14         = R14 ;  computation registers
.DEF rcp15         = R15 ;  computation registers

;====================================================================================================================
;                                           Macro  definitions
;====================================================================================================================

.MACRO xpush0123
    push rmp0;
    push rmp1;
    push rmp2;
    push rmp3;
    in rmp0,SREG ; save SREG
    push rmp0;
.ENDMACRO

.MACRO xpop0123
    pop rmp0;
    out SREG,rmp0; restore SREG
    pop rmp3;
    pop rmp2;
    pop rmp1;
    pop rmp0;
.ENDMACRO


.MACRO xdelay ;max is 65536µs*FREQ_REG/FREQ_ACT
    ldi rDelH,HIGH((@0*_FREQ_ACT)/_FREQ_REG)
    ldi rDelL,LOW((@0*_FREQ_ACT)/_FREQ_REG)
    rcall DelayZ
.ENDMACRO


;====================================================================================================================
;                                                  SRAM definitions
;====================================================================================================================
.DSEG
.ORG Sram_Start
  sOld_state:     .BYTE 1; frequency knob & switch mode 
  sSpeed:         .BYTE 1; speed variable 0 linear 1 exponential 2 wobulation
  sMono:          .BYTE 1; timing for push button
;

;====================================================================================================================
;                               Reset and Interrupt Vectors starting here
;====================================================================================================================
;
.CSEG
.ORG $0000
;
; Reset/Interrupt vectors
;
  rjmp RESET ; Reset
  reti ; INT_0;
  rjmp PC_INT0;reti ; PC_INT0;
  reti ; TIMER1_COMPA; Timer/Counter1 Compare Match A
  rjmp TIMER1_OVF;reti;TIMER1_OVF
  reti ; TIMER0_OVF;
  reti ; EE_RDY EEPROM; Ready
  reti ; ANA_COMP; Analog Comparator
  reti ; ADC_INT; ADC Conversion Complete
  reti ; TIMER1_COMPB; Timer/Counter1 Compare Match B
  reti ; TIMER0_COMPA;Timer/Counter0 Compare Match A
  reti ; TIMER0_COMPB; Timer/Counter0 Compare Match B
  reti ; WDT; Watchdog Time-out
  reti ; USI_START; USI START
  reti ; USI_OVF;
  
;
;====================================================================================================================
;                                     Interrupt Service Routines
;====================================================================================================================
  PC_INT0:
    sei; to maintain clean output
    xpush0123;
    in rmp0,PINB;
    andi rmp0,1<<_SWA|1<<_SWB|1<<_SWC;
    lds rmp1,sOld_state;
    sts sOld_state,rmp0;
    eor rmp1,rmp0;
    mov rmp2,rmp1;
    and rmp2,rmp0; rising only for speed /sweep control
    sbrc rmp2,_SWC;
    rcall set_speed;
    com rmp0;
    and rmp1,rmp0; falling only for frequency control
    sbrc rmp1,_SWC;
    rcall set_time;
    sbrc rmp1,_SWA;
    rcall set_inc;
    xpop0123;
  reti;
  
   
  set_inc:
    lds rmp2,sSpeed;
    andi rmp2,1;
  breq set_inc0;
    movw rmp2,rcp2;
    lsr rmp3;
    ror rmp2;
    lsr rmp3;
    ror rmp2;
    lsr rmp3;
    ror rmp2;
    lsr rmp3;
    ror rmp2;
    lsr rmp3;
    ror rmp2;
  brne set_inc1;
    tst rmp3;
  brne set_inc1;
  
  set_inc0:
    ldi rmp2,1; at least 1
    clr rmp3;
  set_inc1:
    sbrs rmp0,_SWB;
  rjmp set_inc3;
    add rmp2,rcp2;
    adc rmp3,rcp3;
    cpi rmp3,8;
  brlo set_inc2;
    ldi rmp3,8;
    clr rmp2;
    lds rmp0,sSpeed;
    sbrs rmp0,1; sweep ?
  rjmp set_inc2;
    clr rmp3;
    ldi rmp2,1;
    sbi PORTB,_SWB; dec
  set_inc2:
    cli; to avoid conflicts
    movw rcp2,rmp2;
  ret;
  set_inc3:
    movw rmp0,rmp2;
    movw rmp2,rcp2;
    sub rmp2,rmp0;
    sbc rmp3,rmp1;
  brmi set_inc4;
  breq set_inc4;
  rjmp set_inc2;
  set_inc4:
    ldi rmp2,1;
    clr rmp3;
    lds rmp0,sSpeed;
    sbrc rmp0,1; sweep ?
    cbi PORTB,_SWB; inc
  rjmp set_inc2;
    
  set_time:; to check push length
    lds rmp2,sMono;
    tst rmp2;
  brne set_time0;
    ldi rmp2,1;
    sts sMono,rmp2;
  set_time0:
  ret;
  
  set_speed:
    lds rmp2,sSpeed;
    lds rmp3,sMono;
    sbrs rmp3,0; skip if short push   
  rjmp set_sweep;
    inc rmp2;
    cpi rmp2,2;
  brlo set_speed0;
  rcall clr_sweep;
    clr rmp2;
  set_speed0:
    sts sSpeed,rmp2;
    sbrc rmp2,0;
    sbi PORTB,_LED;
  ret;
  
  set_sweep:
    ldi rmp2,2;
    sts sSpeed,rmp2;
    sbi DDRB,_SWA;
    sbi DDRB,_SWB;
    cbi PORTB,_SWB;
    cbi DDRB,_LED;
    ldi rmp3, _TCCR0A|1<<COM0A1
    out TCCR0A,rmp3;
  ret;
  
  clr_sweep:
    cbi DDRB,_SWA;
    cbi DDRB,_SWB;
    sbi DDRB,_LED;
    cbi PORTB,_LED;
    ldi rmp3, _TCCR0A;
    out TCCR0A,rmp3;
  ret;  
  
    
  ; dds timing is critical only SREG is saved 
    
  TIMER1_OVF:
    in rcp15,SREG ; save SREG
    push rcp15;
    mov ZL,rcp1;
    lpm rcp15,Z;pwm value
    out OCR1B,rcp15;update register
    add rcp0,rcp2;
    adc rcp1,rcp3;
    cp rcp1,rcp4;max index value
  brlo end_timer1_ovf;
    sub rcp1,rcp4;
  end_timer1_ovf:
    pop rcp15;
    out SREG,rcp15 ; restore REG
  reti;
  


;===================================================================================================================
;                                   Delays execution for 1 microseconds @ 8MHz
;===================================================================================================================
  DelayZ:
    nop;1
    nop;1
    nop;1
    nop;1
    sbiw rDelL,1 ; 2
    brne DelayZ ; 1/2
  ret;
;===================================================================================================================
; Main Program Start
;===================================================================================================================

RESET:
    ; init ports
    ldi rmp0,_PORTB;
    out PORTB,rmp0;
    ldi rmp0,_DDRB;
    out DDRB,rmp0;
    ; set stack pointer
    ldi ZH,HIGH(RAMEND);
    out SPH,ZH;
    ldi ZL,LOW(RAMEND);
    out SPL,ZL;

    ; init whole memory to zero
    ldi YH,HIGH(SRAM_START); Y point ram location
    ldi YL,LOW(SRAM_START);
    ldi rmp0,0;
  init_ram:
    st Y+,rmp0
    cp YL,ZL;
    cpc YH,ZH;
  brne init_ram;
    clr rcp0;lsb of phase accumulator
    clr rcp1;msb of phase accumulator
    ldi rmp0,98;
    mov rcp2,rmp0; lsb of phase increment => 1000Hz
    clr rcp3; msb of phase increment => 1000Hz
    ldi rmp0,192;
    mov rcp4,rmp0; max of msb of phase increment 
    ldi ZH, HIGH(tab_sin_s*2); init sinus table pointer
    ldi rmp0,1<<_SWA|1<<_SWB|1<<_SWC;
    sts sOld_state,rmp0; init switch status
    ldi rmp0,1;
    sts sSpeed,rmp0;start with fast speed
    sbi PORTB,_LED;
    
    ;calibrate clock
    ldi rmp0,0x4D;
    out OSCCAL,rmp0;
    ldi rmp0,_PLLCSR;
    out PLLCSR,rmp0;
    
    ;init Timer0 will generate pulses for frequency sweep
    ldi rmp0,_TCCR0A; 
    out TCCR0A,rmp0;
    ldi rmp0,_TCCR0B;
    out TCCR0B,rmp0;
    ldi rmp0,_OCR0A;
    out OCR0A,rmp0;
    ldi rmp0,_OCR0B;
    out OCR0B,rmp0;
    
    ;init Timer1 PWM generator
    ldi rmp0,_TCCR1;
    out TCCR1,rmp0;
    ldi rmp0,_OCR1A;
    out OCR1A,rmp0;
    ldi rmp0,_OCR1B;
    out OCR1B,rmp0;
    ldi rmp0,_OCR1C;
    out OCR1C,rmp0;
    ldi rmp0,_GTCCR;
    out GTCCR,rmp0;
     
   ;init interrupts
    ldi rmp0,_TIFR;
    out TIFR,rmp0;
    ldi rmp0,_TIMSK;
    out TIMSK,rmp0;
    ldi rmp0,_GIMSK;
    out GIMSK,rmp0;
    ldi rmp0,_GIFR;
    out GIFR,rmp0;
    ldi rmp0,_PCMSK;
    out PCMSK,rmp0;
    
    ; enable interrupt 
    ldi rmp0,0x80;
    out SREG,rmp0;
 
  ; Here is the infinite loop
  loop_main:
    lds rmp0,sMono;
    tst rmp0;
  breq loop_main;
    ldi rmp0,10;
  loop_mono:; 1second
    xdelay 10000;10ms
    dec rmp0;
  brne loop_mono;
    sts sMono, rmp0;
  rjmp loop_main;
  
 .CSEG
 .ORG $0700

  tab_sin_s:;ARRONDI(62,5*SIN(i*2*PI()/192)) +64 0<=i<=191
  .DB 64 ,66 ,68 ,70 ,72 ,74 ,76 ,78 ,80 ,82 ,84 ,86 ,88 ,90 ,92 ,93
  .DB 95 ,97 ,99 ,100,102,104,105,107,108,110,111,112,114,115,116,117
  .DB 118,119,120,121,122,122,123,124,124,125,125,126,126,126,126,126
  .DB 127,126,126,126,126,126,125,125,124,124,123,122,122,121,120,119
  .DB 118,117,116,115,114,112,111,110,108,107,105,104,102,100,99 ,97
  .DB 95 ,93 ,92 ,90 ,88 ,86 ,84 ,82 ,80 ,78 ,76 ,74 ,72 ,70 ,68 ,66
  .DB 64 ,62 ,60 ,58 ,56 ,54 ,52 ,50 ,48 ,46 ,44 ,42 ,40 ,38 ,36 ,35
  .DB 33 ,31 ,29 ,28 ,26 ,24 ,23 ,21 ,20 ,18 ,17 ,16 ,14 ,13 ,12 ,11
  .DB 10 ,9  ,8  ,7  ,6  ,6  ,5  ,4  ,4  ,3  ,3  ,2  ,2  ,2  ,2  ,2
  .DB 1  ,2  ,2  ,2  ,2  ,2  ,3  ,3  ,4  ,4  ,5  ,6  ,6  ,7  ,8  ,9
  .DB 10 ,11 ,12 ,13 ,14 ,16 ,17 ,18 ,20 ,21 ,23 ,24 ,26 ,28 ,29 ,31
  .DB 33 ,35 ,36 ,38 ,40 ,42 ,44 ,46 ,48 ,50 ,52 ,54 ,56 ,58 ,60 ,62

; End of source code
