;*****************************************************************************
;   Ce fichier est la base de dpart pour une programmation avec             *
;   le PIC 16F676. Il contient  les informations de  base pour               *
;   dmarrer.                                                                *  
;   ecrit grace  Bigonoff par gilles.chevalerias@free.fr                    *
;*****************************************************************************
;                                                                            *
;    NOM: Autocal_f676                                                       *
;    Date:   30/03/2009                                                      *
;    Version:  1                                                             *
;    Circuit:                                                                *
;    Auteur:  Gilles Chevalerias                                             *
;                                                                            *
;*****************************************************************************
;                                                                            *
;    Fichier requis: P16F676.inc                                             *
;                                                                            *
;                                                                            *
;                                                                            *
;*****************************************************************************
;                                                                            *
;    Notes:                                                                  *
;    Permet l'autocalibration d'OSCCAL                                       *
;    avec la methode de la note d'application                                *
;    AN250 de Microchip                                                      *
;    Entre du signal 5kHz sur RA0                                           *
;    Entre BP TEST sur RA1                                                  *
;    Entre BP Calibrate sur RA2                                             *
;    Sortie du signal 5kHz gener par le pic pour verif RA5                  *
;    Sortie led RA4                                                          *
;                                                                            *
;*****************************************************************************

; Ce modele est un peut particulier dans la serie des 16F, il peut travailler avec une tension 
; faible mme sans version LF, sa memmoire est mappe sur les deux banques comme un 16F84,
; il comporte un registre OSCCAL comme les 12F avec une valeur de calibration  sauver, 
; il se programme sur un support reserv au PIC 8 pattes comme un 12F, il a deux ports le A et le C
; qui n'ont que 6 I/O chacun, l'interruption RB0/int et sur RA2, les interruptions de changement d'etat
; sur le portB sont transferes sur le port A et individuellement reglable, idem pour les resistances de
; pullup.


	ERRORLEVEL -302			; suppression du message bank select 
	LIST      p=16F676            ; Dfinition de processeur
	#include <p16F676.inc>        ; fichier include

	__CONFIG   _CP_OFF & _CPD_OFF & _BODEN_OFF & _MCLRE_OFF & _PWRTE_ON & _WDT_OFF & _INTRC_OSC_NOCLKOUT 

; '__CONFIG' prcise les paramtres encods dans le processeur au moment de
; la programmation du processeur. Les dfinitions sont dans le fichier include.
; Choisir une valeur par groupe.
; Voici les valeurs et leurs dfinitions :

; Protection du programme
; -----------------------
;_CP_ALL		protection totale (attention relecture d'OSCCAL impossible)
;_CP_OFF		Pas de protection


; Protection de l'EEprom
; -----------------------
;_CPD_ON		Mmoire EEprom protge
;_CPD_OFF               Mmoire EEprom dprotge


; Reset de la PIC si tension <4V
; ------------------------------
; _BODEN		Reset tension en service
;			Valide PWRTE_ON automatiquement
; _BODEN_OFF		Reset tension hors service

;Utilisation de la pin MCLR
; ------------------------------
;_MCLRE_ON		RA5/MCLR est utilise pour le reset
;
;_MCLRE_OFF		RA5/MCLR est utilise comme une entree/sortie

; Retard  la mise sous tension
; -----------------------------

;_PWRTE_OFF		Dmarrage rapide
;_PWRTE_ON		Dmarrage temporis

; Watchdog
; --------

;_WDT_ON		Watchdog en service
;_WDT_OFF		Watchdog hors service

; Oscillateur
; -----------
;_EC_OSC 		(ECIO) oscillateur externe sur RA5 avec RA4 en I/O               
;_INTRC_OSC_CLKOUT	oscillateur interne clkout sur RA4, I/O sur RA5 (4MHz)    
;_INTRC_OSC_NOCLKOUT	oscillateur interne I/O sur RA4 et RA5 (4MHz)        
;_EXTRC_OSC_NOCLKOUT	(RCIO) RC externe sur RA5 avec RA4 en I/O
;_EXTRC_OSC_CLKOUT      (RC) RC externe sur RA5 avec Fosc/4 sur RA4         
;_LP_OSC 		Oscillateur basse vitesse (?<F<200Khz)        
;_XT_OSC                Oscilateur moyenne vitesse (0,1MHz<F<4Mhz)     
;_HS_OSC 		Oscillateur haute vitesse (1Mhz<F<20Mhz)


;*****************************************************************************
;                               ASSIGNATIONS SYSTEME                         *
;*****************************************************************************

; REGISTRE OPTION_REG (configuration)
; -----------------------------------
OPTIONVAL	EQU	B'01001000'	; pullup, front montant, prediviseur 1/1 assign  WDT
			; RAPU      b7 : 1= Rsistance rappel +5V hors service
			; INTEDG    b6 : 1= Interrupt sur front montant de RA2
			;                0= Interrupt sur front descendant de RA2
			; TOCS      b5 : 1= source clock = transition sur RA2
			;                0= horloge interne
			; TOSE      b4 : 1= Slection front descendant RA2(si B5=1)
			;                0= Slection front montant RA2
			; PSA       b3 : 1= Assignation prdiviseur sur Watchdog
			;                0= Assignation prdiviseur sur Tmr0
			; PS2/PS0   b2/b0 valeur du prdiviseur
                        ;           000 =  1/1 (watchdog) ou 1/2 (tmr0)
			;           001 =  1/2               1/4
			;           010 =  1/4		     1/8
			;           011 =  1/8		     1/16
			;           100 =  1/16		     1/32
			;           101 =  1/32		     1/64
			;           110 =  1/64		     1/128
			;           111 =  1/128	     1/256


; REGISTRE INTCON (contrle interruptions standard)
; -------------------------------------------------
INTCONVAL	EQU	B'00000000'
			; GIE       b7 : masque autorisation gnrale interrupt
                        ;                ne pas mettre ce bit  1 ici
                        ;                sera mis en temps utile
			; PEIE      b6 : masque autorisation gnrale priphriques
			; T0IE      b5 : masque interruption tmr0
			; INTE      b4 : masque interuption RA2/Int
			; RAIE      b3 : masque interruption change PORTA (IOCA doit tre activ)
			; T0IF      b2 : flag tmr0
			; INTF      b1 : flag RA2/Int
			; RAIF      b0 : flag interruption change PORTA


; REGISTRE PIE1 (contrle interruptions priphriques)
; ----------------------------------------------------
PIE1VAL		EQU	B'00000000'
			; EEIE      b7 : masque interrupt criture EEPROM
			; ADIE      b6 : masque interrupt convertisseur A/D
			; RESERVED  b5 : rserv, laisser  0
			; RESERVED  b4 : rserv, laisser  0
			; CMIE      b3 : masque interrupt comparateur
			; RESERVED  b2 : rserv, laisser  0
			; RESERVED  b1 : rserv, laisser  0
			; TMR1IE    b0 : masque interrupt dbordement tmr1


; REGISTRE IOCA (contrle individuel des interruptions sur le PORTA)
; ------------------------------------------------------------------
IOCAVAL		EQU	B'00000000'
			; RESERVED  b7 : rserv, laisser  0 
			; RESERVED  b6 : rserv, laisser  0
			; IOCA5     b5 : masque interruption de changement sur RA5
			; IOCA4     b4 :        "             "     "       "  RA4 
			; IOCA3     b3 :        "             "     "       "  RA3
			; IOCA2     b2 :        "             "     "       "  RA2
			; IOCA1     b1 :        "             "     "       "  RA1
			; IOCA0     b0 :        "             "     "       "  RA0


; REGISTRE WPUA (contrle individuel des resistances de pull up)
; --------------------------------------------------------------
WPUAVAL	EQU		B'00000110'
			; RESERVED  b7 : rserv, laisser  0 
			; RESERVED  b6 : rserv, laisser  0
			; WPUA5     b5 : mettre  1 pour activ resistance sur RA5
			; WPUA4     b4 :        "           "       "       "  RA4 
			; WPUA3     b3 :        "           "       "       "  RA3
			; WPUA2     b2 :        "           "       "       "  RA2
			; WPUA1     b1 :        "           "       "       "  RA1
			; WPUA0     b0 :        "           "       "       "  RA0
	; pour utiliser les resistances de pull up, RAPU de OPTION doit tre  0


; REGISTRE CMCON (COMPARATEURS)
; ------------------------------------
CMCONVAL	EQU	B'00000111' 
			; RESERVED  b7 : rserv, laisser  0
			; COUT      b6 :sortie comparateur
			; RESERVED  b5 : rserv, laisser  0
			; CINV      b4 :inverseur comparateur
			; CIS       b3 :selection entree du comparateur
			; CM2-CM0   b2-0 :mode des comparateurs
			; configurer le mode 111 pour utiliser RA0 RA1 RA2 en I/O


; REGISTRE VRCON (voltage reference module)
; ------------------------------------
VRCONVAL	EQU	B'00000000'
			; VREN      b7 :Validation du module
			; RESERVED  b6 : rserv, laisser  0
			; VRR       b5 :Choix de la plage 
			; RESERVED  b4 : rserv, laisser  0
			; VR3-VR0   b3-0 :Choix de la valeur dans la plage 
					; si VRR=1 VREF=(VR[3:0]/24)*Vdd
					; si VRR=0 VREF=Vdd/4(VR[3:0]/32)*Vdd


; REGISTRE ANSEL (contrle du convertisseur A/D)
; ----------------------------------------------
ANSELVAL	EQU	B'00000000'
			; ANS7	    b7 : 1=AN7/RC3 analogique, 0=AN7/RC3 digital
			; ANS6	    b6 : 1=AN6/RC2 analogique, 0=AN6/RC2 digital
			; ANS5	    b5 : 1=AN5/RC1 analogique, 0=AN5/RC1 digital
			; ANS4	    b4 : 1=AN4/RC0 analogique, 0=AN4/RC0 digital
			; ANS3	    b3 : 1=AN3/RA4 analogique, 0=AN3/RA4 digital
			; ANS2	    b2 : 1=AN2/RA2 analogique, 0=AN2/RA2 digital
			; ANS1	    b1 : 1=AN1/RA1 analogique, 0=AN1/RA1 digital
			; ANS0	    b0 : 1=AN0/RA0 analogique, 0=AN0/RA0 digital


; DIRECTION DES PORTS I/O
; -----------------------

DIRPORTA	EQU	B'00001111'	; Direction PORTA  RA5  RA0 (1=entre)
DIRPORTC	EQU	B'00111111'	; Direction PORTC  RC5  RC0


;*****************************************************************************
;                           ASSIGNATIONS PROGRAMME                           *
;*****************************************************************************

; exemple
; -------
;MASQUE		EQU	H'00FF'


;*****************************************************************************
;                                  DEFINE                                    *
;*****************************************************************************

; exemple
; -------
;PARAMETRES DE MESURE
#define	REFERENCEPERIOD		.200	;Reference Period In Microseconds
#define	TOLERANCE		.1	;Tolerance Can Be Tweaked For Oscillator Accuracy
#define	CALIBRATIONTIME		.10	;Number Of Times To Measure Signal During Calibration

;definition des I/O
#define	INPUT0		PORTA,0		;Input Reference Waveform Of 5KHz
#define	INPUT1		PORTA,1		;Test Button
#define INPUT2		PORTA,2		;Calibrate Button
#define	OUTPUT		PORTA,5		;Outputs 5KHz Calibrated Waveform

;EEPROM Definitions
#define	CALFLAGADR	0x7E		;Value of 0xA5 => Calibration Has Not Been Performed
	                            	;Value of 0x5A => Calibration Has Been Performed
#define	CALVALADR	0X7F	;	Calibration Value

;DEBUG
#define	LED		PORTA,4	;Debug LED

;*****************************************************************************
;                             MACRO                                          *
;*****************************************************************************

			; Changement de banques
			; ----------------------

BANK0	macro				; passer en banque0
		bcf	STATUS,RP0
	endm

BANK1	macro				; passer en banque1
		bsf	STATUS,RP0
	endm

	
;*****************************************************************************
;             VARIABLES BANQUE0 et BANQUE1 sont mappes (communes)           *
;*****************************************************************************

; Zone de 64 bytes
; ----------------

	CBLOCK	0x20			; Dbut de la zone (0x20  0x5F)
	COUNTER	: 1


        ENDC				; Fin de la zone                        


;*****************************************************************************
;                      DEMARRAGE SUR RESET                                   *
;*****************************************************************************

	ORG     0x000            
	call	0x3FF      	; Retrieve Factory Calibration Value
	BANK1	    		; BANK1
	movwf   OSCCAL      	; Load OSCCAL
	goto    INIT            


; ////////////////////////////////////////////////////////////////////////////

;                           P R O G R A M M E

; ////////////////////////////////////////////////////////////////////////////

;*****************************************************************************
;                          INITIALISATIONS                                   *
;*****************************************************************************
INIT

			; initialisation PORTS (banque 0 et 1)
			; ------------------------------------
	BANK0				; slectionner banque0
	clrf	PORTA			; Sorties PORTA  0
	clrf	PORTC			; sorties PORTC  0
	BANK1				; passer en banque1
	movlw	DIRPORTA		; Direction PORTA
	movwf	TRISA			; criture dans registre direction
	movlw	DIRPORTC		; Direction PORTC
	movwf	TRISC			; criture dans registre direction

			; Registre d'options (banque 1)
			; -----------------------------
	movlw	OPTIONVAL		; charger masque
	movwf	OPTION_REG		; initialiser registre option

			; registres interruptions (banque 1)
			; ----------------------------------
	movlw	INTCONVAL		; charger valeur registre interruption
	movwf	INTCON			; initialiser interruptions
	movlw	PIE1VAL			; Initialiser registre 
	movwf	PIE1			; interruptions priphriques 1
	movlw	VRCONVAL		; charger valeur registre interruption
	movwf	VRCON			; initialiser interruptions
	movlw	IOCAVAL			; charger valeur registre interruption
	movwf	IOCA			; initialiser interruptions
	movlw	ANSELVAL		; charger valeur registre interruption
	movwf	ANSEL			; initialiser registre
	movlw	WPUAVAL			; charger valeur registre interruption
	movwf	WPUA			; initialiser registre
	BANK0				; slectionner banque 0
	movlw	CMCONVAL		; charger valeur registre interruption
	movwf	CMCON			; initialiser interruptions


;*****************************************************************************
;                      PROGRAMME PRINCIPAL                                   *
;*****************************************************************************

;*************************************************************************************
;Main Program - This routine watches button INPUT1 & INPUT2
;             - Calls CALIBRATE() or Calls TEST()
;*************************************************************************************
MAIN
	btfss	INPUT1		;Check Test Button
	goto	ONE
	btfsc	INPUT2		;Check Calibration Button
	goto	MAIN
	
TWO
	bsf	LED
	call	CALIBRATE
	bcf	LED
	goto	MAIN	

ONE
	bsf	LED
	call	TEST
	bcf	LED
	goto	MAIN
	

;*******************************************************************************************
;Subroutines
;*******************************************************************************************
;TEST() - This routine is used to test the OSCCAL value in the 12F6xx
;	- Checks If Calbibration Was Performed
;       - Updates OSCCAL Value If Calibration Was Performed
;       - Outputs A 5kHZ 50% Square Wave On OUTPUT Until Pushbutton Is Released
;*******************************************************************************************
TEST
	movlw	CALFLAGADR
	call	EEREAD
	sublw	0x5A
	btfss	STATUS,Z	;Was Calibration Flag Set?
	goto	STARTTEST	;No Don't Change Osccal
	
	movlw	CALVALADR
	call	EEREAD		;Yes Change The Osccal
	BANK1			;BANK1
	movwf	OSCCAL
	BANK0			;BANK0
	
STARTTEST			;The instructions below make a 5kHZ 50% Square Wave Output On GP1,
					;if the device is calibrated
	bsf	OUTPUT		;1us
	movlw	.31		;Delay 99us
	call	DELAYUS		;99+1 = 100us
	bcf	OUTPUT		;1us
	movlw	.30		;Delay 96us
	call	DELAYUS
	btfss	INPUT1		;1us
	goto	STARTTEST	;2us  =>1+96+1+2 = 100us
	return

;**********************************************************************************************
;Calibrate() - Measures A Period From The Input(GP0) Reference Signal
;            - Updates Osccal Value
;            - Updates E^2 
;**********************************************************************************************
CALIBRATE
	movlw	CALIBRATIONTIME
	movwf	COUNTER		;Calibration Counter

LOW0
	btfsc	INPUT0		;Wait To Sample Low Edge #0 (Makes Sure We Are Synchronized First Time Through)
	goto	LOW0		

HIGH1
	btfss	INPUT0		;Wait To Sample High Edge #1
	goto	HIGH1	
	clrf	TMR0		;Start Timer  (Timer Will Be Behind By 5us After This Instruction)

LOW1
	btfsc	INPUT0		;Wait To Sample Low Edge #1
	goto	LOW1
	
HIGH2
	btfss	INPUT0
	goto	HIGH2
	movf	TMR0,W		;Stop Timer (Timer Will Be Stopped 3us Late)
	addlw	.2		;Timer Is Behind By 2us Total From Start To Stop	
	call	CHECKPERIOD	;See If Osccal Needs To Be Adjusted
	
LOW2
	btfsc	INPUT0		;Wait To Sample Low Edge #2
	goto	LOW2
	decfsz	COUNTER,F	;Decrement The Calibration Counter
	goto	HIGH1
	call	UPDATE_EE	;Update E^2
	return

;*****************************************************************************************
;Update_EE - This routine Updates Calibration Flag & Calibration Value
;*****************************************************************************************
UPDATE_EE
	BANK1			;BANK1
	movlw	0x5A		;Update Calibration Flag
	movwf	EEDATA
	movlw	CALFLAGADR
	call	EEWRITE
	
	movf	OSCCAL,W
	movwf	EEDATA
	movlw	CALVALADR
	call	EEWRITE 	;Update Calibration Value
	BANK0			;BANK0
	return
	
;****************************************************************************************************
;CheckPeriod(W) - This routine computes the difference between the REFERENCEPERIOD and MEASUREDPERIOD
;          	- The MEASUREDPERIOD is contained in W when this routine is called
;               - The Osccal Is Adjusted Up or Down If The Period Is Outside The Specified Tolerance
;****************************************************************************************************
CHECKPERIOD
	sublw	REFERENCEPERIOD
	btfsc	STATUS,Z	;If (ReferencePeriod - MeasuredPeriod = 0) Don't Change Osccal
	return
	
	btfsc	STATUS,C	;If (ReferencePeriod - MeasuredPeriod > 0) Oscillator Could Be Too Fast
	goto	RUNNINGSLOW	;Else Oscillator Could Be Too Slow

RUNNINGFAST
	xorlw	0xFF		;Two's Complement Value
	addlw	.1
	sublw	TOLERANCE	;If (Tolerance - (ReferencePeriod - MeasuredPeriod) = 0 ) Don't Change Osccal
	btfsc	STATUS,Z
	return
	goto	ADJUSTDOWN	;Else Adjust Osccal Down
	
RUNNINGSLOW
	sublw	TOLERANCE	;If (Tolerance - (ReferencePeriod - MeasuredPeriod) = 0) Don't Change Osccal
	btfsc	STATUS,Z
	return
	goto	ADJUSTUP	;Else Adjust Osccal Up
	
ADJUSTDOWN
	BANK1			;BANK1
	movlw	.4
	subwf	OSCCAL,F	;Adjust Osccal Down
	BANK0			;BANK0
	return
	
ADJUSTUP
	BANK1			;BANK1
	movlw	.4
	addwf	OSCCAL,F	;Adjust Osccal Up
	BANK0			;BANK0
	return	
	
;*************************************************************************************************************************	
;EEREAD(W) - Address To Read Is Contained In W When This Function Is Called
;***************************************************************************************************************
EEREAD
	BANK1			;BANK1
	movwf	EEADR
	bsf	EECON1,RD
	movf	EEDATA,W
	BANK0			;BANK0
	return	

;**********************************************************************************************************
;EEWRITE(W) - Address To Read Is Contained In W When This Function Is Called
;           - EEDATA Is Loaded Prior To This Function Call
;           - BANK1 must be selected before this function is called
;****************************************************************************************************	
EEWRITE
	movwf	EEADR		
	bsf	EECON1,WREN
	bcf	INTCON,GIE
	movlw	0x55
	movwf	EECON2
	movlw	0xAA
	movwf	EECON2
	bsf		EECON1,WR
EECOMPLETE
	btfsc	EECON1,WR
	goto	EECOMPLETE
	bcf		EECON1,WREN	
	return
	
;****************************************************************************************************
;****************************************************************************************************
;DELAYUS(W) - Delay Microseconds
;           - TotalTime(W) = [(1)+(2)+(1)+(W*3-1)+1+2] * (4/OSC) (This includes the movlw & the call)
;	    - Max Time When W=0xFF, [ 771 Cycles * (4/OSC) ] 
; 	    - Must Declare COUNTER AS GPR
;           - W > 0
;****************************************************************************************************
DELAYUS
	movwf	COUNTER
LOOP1
	decfsz	COUNTER,F
	goto	LOOP1	
	nop
	return

;********************************************************************************************
;EEPROM - CALFLAGADR  - Contains Calibration Flag Value  
;       - CALVALADR - Contains Calibration Value
;********************************************************************************************
	ORG	(0X2100+CALFLAGADR) 	

	DE	0A5H	;Initialize Calibration Flag

	DE	000H	;Initialize Calibration Value



	END 				; directive fin de programme

