function [out,out1,out2]=integrules(varargin);

% This function is used to generate integration constants used by elements
%
% integrules(TopologyName,Type)  generates integration rules. The second
% optional argument type lets you select between various integration rules
%
% Supported topologies are
%  bar1 (1D linear)
%  beam1 (1D cubic)
%  quad4 (2D bi-linear), quadb (2d quadratic)
%  tria3 (2D affine), tria6 (2D quadratic)
%  tetra4, tetra10
%  penta6, penta15
%  hexa8, hexa20, hexa27
%
% Supported Gauss point families can be listed using 
%  integrules('gauss')
% For a given rule family, available rules and ID are listed with
%  integrules('gauss h3d')
%
% integrules(EltName,RSTPositions,XYZPositions)

%	Etienne Balmes, Jean Michel Leclere
%       Copyright (c) 2001-2006 by INRIA and SDTools,All Rights Reserved.
%       Use under OpenFEM trademark.html license and LGPL.txt library license
%       integrules('cvs') for revision information

out = [];

[CAM,Cam]=comstr(varargin{1},1); carg=2;

if comstr(Cam,'cvs') 

 out='$Revision: 1.84 $  $Date: 2006/05/05 09:11:19 $';


% Jacobian computations for Surfaces in 3D -----------------------------
elseif comstr(Cam,'buildndn') 

typ=varargin{carg};carg=carg+1;
if typ==2 | typ==3 % 2D/3D NDN building
 rule=varargin{carg};carg=carg+1;
 nodeE=varargin{carg};carg=carg+1;
 of_mk('buildndn',typ,rule,nodeE);
else

opt=varargin{carg};carg=carg+1;
node=varargin{carg};carg=carg+1;
if ~isfield(opt,'bas'); opt.bas=zeros(9,size(opt.N,1));end

try;
 of_mk('buildndn',typ,opt,node);
 if any(~finite(opt.NDN(:))); error('not finite');end
 if (norm(opt.bas)==0 & typ~=13 ); error(1);end

catch
opt.bas
keyboard
Nw=size(opt.N,1); % number fo integration points
Nnode=size(opt.N,2);

%n0=node;
%node=n0+[rand(4,1) zeros(4,2)]/5;
%opt2=opt;opt2.jdet=opt2.jdet*0;of_mk('buildndn2d',opt2,node);opt.NDN=opt.NDN*0;
%opt.NDN(:,1:size(opt.N,1))=opt.N';

for jw=1:Nw

  % local basis on the surface
  % [dx_e;dy_e]=J[dr;ds]
  x=opt.Nr(jw,:)*node;xr=norm(x); x=x/xr; xs=0;
  y=opt.Ns(jw,:)*node;  a=-x*y'; y=y+a*x;
  ys=norm(y);yr=-a; y=y/ys;
  z=-[y(2)*x(3)-y(3)*x(2);y(3)*x(1)-y(1)*x(3);y(1)*x(2)-y(2)*x(1)];

  opt.jdet(jw)=xr*ys;
  opt.NDN(:,Nw+jw)=opt.Nr(jw,:)'/xr; % N,x(jw)
  opt.NDN(:,2*Nw+jw)=opt.Ns(jw,:)'/ys-opt.Nr(jw,:)'*(yr/xr/ys); % N,x(jw)
  opt.bas(:,jw)=[x';y';z];
end
%norm(opt.NDN-opt2.NDN)
end % catch
out=opt;

if nargout==2
 Nw=size(opt.N,1); % number fo integration points
 out1=zeros(Nw*3,opt.Nnode*3);
 % 3 dof per node
 %                  j           k
 for jw=1:Nw; 
   x=opt.bas(1:3,jw);y=opt.bas(4:6,jw);z=opt.bas(7:9,jw);
   for j1=1:3; for j3=1:3
   %                  u_ik                      Ny_i
   out1(j1+3*(jw-1),j3:3:end)=x(j1)*z(j3)*opt.NDN(:,2*Nw+jw)' ...
    -y(j1)*z(j3)*opt.NDN(:,Nw+jw)' ...
    +z(j1)/2*(y(j3)*opt.NDN(:,Nw+jw)'-x(j3)*opt.NDN(:,2*Nw+jw)');
 end;end;end
end
end % Typ 23

% ---------------------------------------------------------------------
elseif comstr(Cam,'mitc4')

  OfMkPol=of_mk('mitcinit');
  fprintf('\n%%Written by integrules(''mitc4'')\n\n');
  writetoscreen(OfMkPol)

% Integration rules for the MITC4 element ----------------------------
elseif comstr(Cam,'mit4') 

 % 4 nodes and 4 tying points
 xi = [-1 -1 0;1 -1 0;1 1 0;-1 1 0; ...
       0 1 0;-1 0 0;0 -1 0;1 0 0];

 % points of interest : integration and tying points
   w=1/sqrt(3);
   w=[-w -w 0 1;w -w 0 1;w w 0 1;-w w 0 1;
         0 1 0 0;-1 0 0 0;0 -1 0 0;1 0 0 0];

   r=w(:,1); s=w(:,2); z=zeros(size(r)); o=ones(size(r));

   N=[(1-r).*(1-s)/4 (1+r).*(1-s)/4  (1+r).*(1+s)/4  (1-r).*(1+s)/4  ...
      (1+s)/2 (1-r)/2 (1-s)/2 (1+r)/2];
 
   Nr=[(s-1)/4 (1-s)/4 (1+s)/4 (-1-s)/4 z -o/2 z o/2];
   Ns=[(r-1)/4 (-1-r)/4 (1+r)/4 (1-r)/4 o/2 z -o/2 z];

  out=struct('N',N,'Nr',Nr,'Ns',Ns,'Nw',size(w,1),'NDN',zeros(4,size(N,1)*3), ...
    'jdet',zeros(size(N,1),1),'w',w,'NDNLabels',{{'',',x',',y'}}, ...
   'DofLabels',{{'u','v','w','\theta_u','\theta_v','\theta_w'}}, ...
   'Nnode',4,'xi',xi,'type','mit4','PerField',4);

% - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
elseif comstr(Cam,'strainndn')

   opt=varargin{carg};carg=carg+1;
   j0=varargin{carg};carg=carg+1;
   r1=opt.StrainDefinition{j0}; 

   DofPerNode=length(opt.DofLabels); Nnode=opt.Nnode;
   Ndof=DofPerNode*Nnode; Nstrain=max(r1(:,1)); Nw=size(opt.N,1);
   Nshape=size(opt.N,2);

   i1=[1 cumprod([Nstrain Nw Ndof])]; i2=[1 cumprod([Nnode Nw 3])];
   c=spalloc(i1(end),i2(end),1);

   for jw=1:Nw;  for jStrain=1:size(r1,1); 
    for jNode=1:Nnode
     %             j(epsilon)  jw     jDOF
     c(i1(1:3)*([r1(jStrain,1);jw;r1(jStrain,3)+(jNode-1)*DofPerNode]-1)+1, ...
       i2(1:3)*([jNode;jw;abs(r1(jStrain,2))]-1)+1)=sign(r1(jStrain,2));
    end
   end;end
   
   in1=varargin{carg};carg=carg+1; % integration points to eliminate
   % Now eliminate strain assembly at the tying points
   ind=[]; for jw=in1(:)'; for jStrain=1:Nstrain; for jNode=1:Nnode; 
    ind=[ind i1(1:3)*([jStrain;jw;1+(jNode-1)*DofPerNode]-1)+1];
    ind=[ind i1(1:3)*([jStrain;jw;2+(jNode-1)*DofPerNode]-1)+1];
    ind=[ind i1(1:3)*([jStrain;jw;3+(jNode-1)*DofPerNode]-1)+1];
    ind=[ind i1(1:3)*([jStrain;jw;4+(jNode-1)*DofPerNode]-1)+1];
    ind=[ind i1(1:3)*([jStrain;jw;5+(jNode-1)*DofPerNode]-1)+1];
    ind=[ind i1(1:3)*([jStrain;jw;6+(jNode-1)*DofPerNode]-1)+1];
   end;end;end

   out=c;out1=Nstrain;out2=ind;

% - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
elseif comstr(Cam,'mitshellstrain') 


  opt=varargin{carg};carg=carg+1;

  % 1. need to write the e at all integration points as c*opt.NDN(:)
   DofPerNode=length(opt.DofLabels); Nnode=opt.Nnode;
   Ndof=DofPerNode*Nnode; Nw=size(opt.N,1);
   Nshape=size(opt.N,2);

   [c,Nstrain,ind]=integrules('StrainNDN',opt,1,5:8);

  % 2. then create the linear combinations to interpolate the shear strain
  %    \gamma_r (r,s) = N_A(r,s) \gamma_r(A) + N_C  \gamma_r(C)
  %    \gamma_s (r,s) = N_D(r,s) \gamma_r(D) + N_B  \gamma_r(B)

  % A,B,C,D -> jw=5,6,7,8
  i1=[1 cumprod([Nstrain Nw Ndof])]; i2=[1 cumprod([Nnode Nw 3])];
  for j1=1:2
   if j1==1; jStrain=7; DOF=[3 5]; TyingW=[5 7];% gamma_r
   else      jStrain=8; DOF=[3 4]; TyingW=[6 8];% gamma_s
%   if j1==1; jStrain=7; DOF=[3 5]; TyingW=[6 8];% gamma_r
%   else      jStrain=8; DOF=[3 4]; TyingW=[5 7];% gamma_s
   end
  for jNode=1:Nnode; for jw=1:4;
   % Do not use the local value of gamma_r but combination of tying points
   c(i1(1:3)*([jStrain;jw;DOF(1)+(jNode-1)*DofPerNode]-1)+1,:)= ...
    opt.N(jw,TyingW(1))* ...
     c(i1(1:3)*([jStrain;TyingW(1);DOF(1)+(jNode-1)*DofPerNode]-1)+1,:)+ ...
    opt.N(jw,TyingW(2))* ...
     c(i1(1:3)*([jStrain;TyingW(2);DOF(1)+(jNode-1)*DofPerNode]-1)+1,:);
   c(i1(1:3)*([jStrain;jw;DOF(2)+(jNode-1)*DofPerNode]-1)+1,:)= ...
    opt.N(jw,TyingW(1))* ...
     c(i1(1:3)*([jStrain;TyingW(1);DOF(2)+(jNode-1)*DofPerNode]-1)+1,:)+ ...
    opt.N(jw,TyingW(2))* ...
     c(i1(1:3)*([jStrain;TyingW(2);DOF(2)+(jNode-1)*DofPerNode]-1)+1,:);
   end
  end 
  end % case gamma_r or gamma_s
  c(ind,:)=[];

  opt.NDN=[opt.N(:,1:Nnode)' opt.Nr(:,1:Nnode)' opt.Ns(:,1:Nnode)'];
  c=c*opt.NDN(:);
  c=reshape(c,Nw/2*Nstrain,Ndof);

  opt.MatrixIntegrationRule{1}=struct('c',c,'Nstrain',Nstrain, ...
    'Ndof',Ndof,'typ','rs');
  % eliminate tying point shape functions that are no longer needed
  opt.N=opt.N(:,1:Nnode); opt.Nr=opt.Nr(:,1:Nnode);opt.Ns=opt.Ns(:,1:Nnode);
  opt.Nnode=Nnode;opt.NDN=zeros(Nnode,3*Nw);

  % mass rule
  [c,Nstrain,ind]=integrules('StrainNDN',opt,2,5:8);c(ind,:)=[];
  opt.MatrixIntegrationRule{2}=struct('c',c,'Nstrain',Nstrain,'Ndof',Ndof);

  out=opt;

% Integration rules for 1D [0,1] linear segment -----------------------------
elseif comstr(Cam,'bar1') 

 xi = [0 0 0;1 0 0];
 if nargin<2; w='def'; else; w=varargin{carg};carg=carg+1;end
 [w,iN]=QuadraturePoints('1d',w,xi,varargin{carg:end});

 r=w(:,1);N=[1-r r]; Nr=ones(size(r))*[-1 1];
 out=struct('w',w,'xi',xi(iN,:), ...
  'N',N(:,iN),'Nr',Nr(:,iN),'Nrr',zeros(size(r)), ...
  'NDN',zeros(2,3*size(w,1)),'jdet',zeros(size(w,1),1), ...
  'NDNLabels',{{'',',x',',x^2'}},'type','bar1');

% Integration rules for 1D [0,1] quadratic segment -----------------------
elseif comstr(Cam,'beam3') 

 xi = [0 0 0;1 0 0;.5 0 0];
 if nargin<2; w='def'; else; w=varargin{carg};carg=carg+1;end
 [w,iN]=QuadraturePoints('1d',w,xi,varargin{carg:end});

 r=w(:,1);
 N=[-r.*(1-r)/2 r.*(1+r)/2  1-r.*r]/2+.5;
 Nr=[r-0.5 r+0.5 -2*r]/2;
 Nrr=ones(size(r))*[.5 .5 -1];

 out=struct('w',w,'N',N(:,iN),'Nr',Nr(:,iN),'Nrr',Nrr(:,iN),'xi',xi(iN,:), ...
  'type','beam3','NDN',zeros(size(N,2),3*size(N,1)),'jdet',zeros(size(N,1),1));

% Integration rules for 1D [0,1] cubic segment -----------------------------
elseif comstr(Cam,'beam1') 

 xi = [0 0 0;1 0 0];
 if nargin<2; w='def'; else; w=varargin{carg};carg=carg+1;end
 [w,iN]=QuadraturePoints('1d',w,xi,varargin{carg:end});

 N=[1-3*w(:,1).^2+2*w(:,1).^3 w(:,1)-2*w(:,1).^2+w(:,1).^3 ...
      3*w(:,1).^2-2*w(:,1).^3   -w(:,1).^2+w(:,1).^3];

 Nr=[-6*w(:,1)+6*w(:,1).^2 1-4*w(:,1)+3*w(:,1).^2 ...
       6*w(:,1)-6*w(:,1).^2   -2*w(:,1)+3*w(:,1).^2];

 Nrr=[-6+12*w(:,1) -4+6*w(:,1) 6-12*w(:,1)   -2+6*w(:,1)];
 out=struct('w',w,'N',N,'Nr',Nr,'Nrr',Nrr,'xi',xi,'type','beam1', ...
   'NDN',zeros(size(N,2),3*size(N,1)),'jdet',zeros(size(N,1),1));
 %k=zeros(4);for j1=1:3 k=k+out.Nrr(j1,:)'*out.Nrr(j1,:)*w(j1,4);end

% Integration rules for 2D [-1,1 x -1,1] square ----------------------------
elseif comstr(Cam,'quad4') |comstr(Cam,'q4p')

 xi = [-1 -1 0;1 -1 0;1 1 0;-1 1 0];
 if nargin<2; w='def'; else;  w=varargin{carg};carg=carg+1; end
 [w,iN]=QuadraturePoints('q2d',w,xi,varargin{carg:end});

 N = [(1+w(:,1)*xi(:,1)').*(1+w(:,2)*xi(:,2)')/4];% ...
         %  1-w(:,1).^2 1-w(:,2).^2]
 Nr=[xi(1:4,ones(1,size(w,1))*1)'.*(1+w(:,2)*xi(:,2)')/4 ];%...
        % -2*w(:,1) w(:,1)*0];
 Ns=[xi(1:4,ones(1,size(w,1))*2)'.*(1+w(:,1)*xi(:,1)')/4 ];%...
        % w(:,1)*0 -2*w(:,2)];

 out=struct('N',N(:,iN),'Nr',Nr(:,iN),'Ns',Ns(:,iN), ...
      'Nw',size(w,1),'NDN',zeros(size(N,2),3*size(N,1)), ...
      'NDNLabels',{{'',',x',',y'}}, ...
      'jdet',zeros(size(N,1),1),'w',w,'Nnode',4,'xi',xi(iN,:),'type','quad4');

  if carg<=nargin&isempty(strfind(Cam,'new')) % OBSOLETE
      out=integrules('2d',out,varargin{carg});carg=carg+1;
  end % OBSOLETE

% Integration rules for 2D [-1,1 x -1,1] square ----------------------------
elseif comstr(Cam,'q4q')

 xi = [-1 -1 0;1 -1 0;1 1 0;-1 1 0];
 if nargin<2; w='def'; else;  w=varargin{carg};carg=carg+1; end
 [w,iN]=QuadraturePoints('q2d',w,xi,varargin{carg:end});

 N = [(1+w(:,1)*xi(:,1)').*(1+w(:,2)*xi(:,2)')/4 ...
           1-w(:,1).^2 1-w(:,2).^2];
 Nr=[xi(1:4,ones(1,size(w,1))*1)'.*(1+w(:,2)*xi(:,2)')/4 ...
         -2*w(:,1) w(:,1)*0];
 Ns=[xi(1:4,ones(1,size(w,1))*2)'.*(1+w(:,1)*xi(:,1)')/4 ...
         w(:,1)*0 -2*w(:,2)];
 out=struct('N',N,'Nr',Nr,'Ns',Ns, ...
      'Nw',size(w,1),'NDN',zeros(size(N,2),3*size(N,1)), ...
      'NDNLabels',{{'',',x',',y'}}, ...
      'jdet',zeros(size(N,1),1),'w',w,'Nnode',4,'xi',xi,'type','q4q');

% ---------------------------------------------------------------------
elseif comstr(Cam,'quad9')|comstr(Cam,'q9p')

  xi = [-1 -1 0;1 -1 0;1 1 0;-1 1 0;0 -1 0;1 0 0;0 1 0;-1 0 0;0 0 0];

 if nargin<2; w='def'; else; w=varargin{carg};carg=carg+1;end
 [w,iN]=QuadraturePoints('q2d',w,xi);

% r=w(:,1); s=w(:,2); xxx
% See Hugues 132-135                          na = 1/4 (1+xa x) (1 + ya y)
  N = [(1+w(:,1)*xi(1:4,1)').*(1+w(:,2)*xi(1:4,2)')/4 ...
      (1-w(:,1).^2).*(1-w(:,2))/2 ...   % n5 = 1/2 (1-x2  ) (1-y)
      (1-w(:,2).^2).*(1+w(:,1))/2 ...   % n6 = 1/2 (1+x)    (1-y2)
      (1-w(:,1).^2).*(1+w(:,2))/2 ...   % n7 = 1/2 (1-x2)   (1+y)
      (1-w(:,2).^2).*(1-w(:,1))/2 ...   % n8 = 1/2 (1-x)    (1-y2)
      (1-w(:,1).^2).*(1-w(:,2).^2)];    % n9 =     (1-x2)   (1-y2)

  Nr=[xi(1:4,ones(1,size(w,1))*1)'.*(1+w(:,2)*xi(1:4,2)')/4 ...
       ( -w(:,1)).*(1-w(:,2))       (1-w(:,2).^2)/2 ...
       ( -w(:,1)).*(1+w(:,2))      -(1-w(:,2).^2)/2 ...
       ( -w(:,1)).*(1-w(:,2).^2)];

  Ns=[xi(1:4,ones(1,size(w,1))*2)'.*(1+w(:,1)*xi(1:4,1)')/4 ...
       -(1-w(:,1).^2)/2     ( -w(:,2)).*(1+w(:,1)) ...
        (1-w(:,1).^2)/2     ( -w(:,2)).*(1-w(:,1)) ...
        (-w(:,2)).*(1-w(:,1).^2)];

  % if n5 n8 present
  % n1 = n1 -1/2 (n5+n8)
  % n2 = n2 -1/2 (n5+n6)
  % n3 = n3 -1/2 (n6+n7)
  % n4 = n4 -1/2 (n7+n8)

  N (:,1:4)=N (:,1:4)-N (:,5:8)*[1 0 0 1;1 1 0 0;0 1 1 0;0 0 1 1]'/2;
  Nr(:,1:4)=Nr(:,1:4)-Nr(:,5:8)*[1 0 0 1;1 1 0 0;0 1 1 0;0 0 1 1]'/2;
  Ns(:,1:4)=Ns(:,1:4)-Ns(:,5:8)*[1 0 0 1;1 1 0 0;0 1 1 0;0 0 1 1]'/2;
  i1=[9 9 9 9];
   N(:,1:4)= N(:,1:4)- N(:,i1)/4; N (:,5:8)=N (:,5:8)-N (:,i1)/2; 
  Nr(:,1:4)=Nr(:,1:4)-Nr(:,i1)/4; Nr(:,5:8)=Ns(:,5:8)-Ns(:,i1)/2; 
  Ns(:,1:4)=Ns(:,1:4)-Ns(:,i1)/4; Ns(:,5:8)=Ns(:,5:8)-Ns(:,i1)/2; 

   % if n9 is present n1:4=n1:4-n9/4 n5:8=n5:8-n9/2
   % note for Q2 : still need to remove contribution of n9 for Q2
  out=struct('N',N,'Nr',Nr,'Ns',Ns,'Nw',size(w,1), ...
      'Nnode',8,'NDN',zeros(size(N,2),size(N,1)*3), ...
      'NDNLabels',{{'',',x',',y'}}, ...
      'jdet',zeros(size(w,1),1),'w',w, ...
      'DofLabels',{{'u','v','w','ru','rw'}},'type','quadb','xi',xi);

% ---------------------------------------------------------------------
elseif comstr(Cam,'quadb')|comstr(Cam,'q8p')

  xi = [-1 -1 0;1 -1 0;1 1 0;-1 1 0;0 -1 0;1 0 0;0 1 0;-1 0 0];

 if nargin<2; w='def'; else; w=varargin{carg};carg=carg+1;end
 [w,iN]=QuadraturePoints('q2d',w,xi);

% r=w(:,1); s=w(:,2); xxx
% See Hugues 132-135                          na = 1/4 (1+xa x) (1 + ya y)
  N = [(1+w(:,1)*xi(1:4,1)').*(1+w(:,2)*xi(1:4,2)')/4 ...
      (1-w(:,1).^2).*(1-w(:,2))/2 ...   % n5 = 1/2 (1-x2  ) (1-y)
      (1-w(:,2).^2).*(1+w(:,1))/2 ...   % n6 = 1/2 (1+x)    (1-y2)
      (1-w(:,1).^2).*(1+w(:,2))/2 ...   % n7 = 1/2 (1-x2)   (1+y)
      (1-w(:,2).^2).*(1-w(:,1))/2 ];   % n8 = 1/2 (1-x)    (1-y2)

  Nr=[xi(1:4,ones(1,size(w,1))*1)'.*(1+w(:,2)*xi(1:4,2)')/4 ...
       ( -w(:,1)).*(1-w(:,2))       (1-w(:,2).^2)/2 ...
       ( -w(:,1)).*(1+w(:,2))      -(1-w(:,2).^2)/2];

  Ns=[xi(1:4,ones(1,size(w,1))*2)'.*(1+w(:,1)*xi(1:4,1)')/4 ...
       -(1-w(:,1).^2)/2     ( -w(:,2)).*(1+w(:,1)) ...
        (1-w(:,1).^2)/2     ( -w(:,2)).*(1-w(:,1))];

  % if n5 n8 present
  % n1 = n1 -1/2 (n5+n8)
  % n2 = n2 -1/2 (n5+n6)
  % n3 = n3 -1/2 (n6+n7)
  % n4 = n4 -1/2 (n7+n8)

  N (:,1:4)=N (:,1:4)-N (:,5:8)*[1 0 0 1;1 1 0 0;0 1 1 0;0 0 1 1]'/2;
  Nr(:,1:4)=Nr(:,1:4)-Nr(:,5:8)*[1 0 0 1;1 1 0 0;0 1 1 0;0 0 1 1]'/2;
  Ns(:,1:4)=Ns(:,1:4)-Ns(:,5:8)*[1 0 0 1;1 1 0 0;0 1 1 0;0 0 1 1]'/2;

  out=struct('N',N,'Nr',Nr,'Ns',Ns,'Nw',size(w,1), ...
      'Nnode',8,'NDN',zeros(size(N,2),size(N,1)*3), ...
      'NDNLabels',{{'',',x',',y'}}, ...
      'jdet',zeros(size(w,1),1),'w',w, ...
      'DofLabels',{{'u','v','w','ru','rw'}},'type','quadb','xi',xi);

% Integration rules for 2D [0,0 1,0 0,1] triangle ---------------------------
% Shape functions are 1-r-s, r and s
elseif comstr(Cam,'tria3')|comstr(Cam,'t3p')

 xi = [0 0 0;1 0 0;0 1 0];
 if nargin<2; w='def'; else; w=varargin{carg};carg=carg+1;end
 [w,iN]=QuadraturePoints('t2d',w,xi);

  N = [1-w(:,1)-w(:,2) w(:,1) w(:,2)];

  Nr=ones(size(w,1),1)*[-1 1 0];
  Ns=ones(size(w,1),1)*[-1 0 1];
  nw=size(N,1);

  out=struct('N',N,'Nr',Nr,'Ns',Ns,'w',w,'Nw',size(w,1), ...
      'jdet',zeros(nw,1),'NDN',zeros(3,nw*3),'NDNLabels',{{'',',x',',y'}}, ...
      'Nnode',3,'xi',xi,'type','tria3');
  if carg<=nargin % OBSOLETE
      out=integrules('2d',out,varargin{carg});carg=carg+1;
  end % OBSOLETE

% 6 node triangle - - - - - - - - - - - - - - - - - - - - - - - - - -
elseif comstr(Cam,'tria6') |comstr(Cam,'t6p')

 xi=[0 0 0;1 0 0;0 1 0;.5 0 0;.5 .5 0;0 .5 0];
 if nargin<2; w='def'; else; w=varargin{carg};carg=carg+1;end
 [w,iN]=QuadraturePoints('t2d',w,xi);

 r=w(:,1)*2-1; s=w(:,2)*2-1;  z=zeros(size(r));

 % shape between [-1 1]
 N=[   .5*(r+s).*(r+s+1)  .5*(1+r).*r .5*(1+s).*s   ...
          -(1+r).*(r+s)      (1+r).*(1+s) -(1+s).*(r+s)   ];  

 Nr=[   .5*(2*r+2*s+1) .5*(1+2*r) z...
         -(2*r+s+1)     (1+s) -(1+s)  ]*2;

 Ns=[.5*(2*r+2*s+1) z .5*(1+2*s)  ...
          -(1+r)   (1+r) -(2*s+r+1) ]*2;

 out=struct('N',N,'Nr',Nr,'Ns',Ns, ...
      'Nw',size(w,1),'Nnode',6,'NDN',zeros(size(N,2),size(N,1)*3), ...
      'NDNLabels',{{'',',x',',y'}}, ...
      'jdet',zeros(size(w,1),1),'w',w,'DofLabels',{{'u','v','w'}}, ...
      'xi',xi,'type','tria6');

% ---------------------------------------------------------------------
elseif comstr(Cam,'tetra4')

 xi = [0 0 0;1 0 0;0 1 0;0 0 1];
 if nargin<2; w='def'; else; w=varargin{carg};carg=carg+1;end
 [w,iN]=QuadraturePoints('t3d',w,xi);

  % Shape functions [1-r-s-t r s t] 
  r=w(:,1);  s=w(:,2);  t=w(:,3);  v=1-r-s-t;z=v*0;o=ones(size(z));

  N = [v r s t];
  Nr= [-o o z z];
  Ns= [-o z o z];
  Nt= [-o z z o];

 out=struct('N',N,'Nr',Nr,'Ns',Ns,'Nt',Nt, ...
      'Nw',size(w,1),'Nnode',4,'NDN',zeros(size(N,2),size(N,1)*4), ...
      'NDNLabels',{{'',',x',',y',',z'}}, ...
      'jdet',zeros(size(w,1),1),'w',w,'DofLabels',{{'u','v','w'}},...
      'xi',xi,'type','tetra4');

% ---------------------------------------------------------------------
elseif comstr(Cam,'tetra10')

  xi = [0 0 0; 1 0 0; 0 1 0; 0 0 1;
       .5 0 0;.5 .5 0; 0 .5 0; 0 0 .5; .5 0 .5;  0 .5 .5   ];
 if nargin<2; w='def'; else; w=varargin{carg};carg=carg+1;end
 [w,iN]=QuadraturePoints('t3d',w,xi);

  % See also DOC code aster R30101b
  % v=1-r-s-t
  % [ v(2v-1) r(2r-1) s(2s-1) t(2t-1)  
  %   4rv  4rs  4sv  4tv  4rs  4st ]
  r=w(:,1);  s=w(:,2);  t=w(:,3);  v=1-r-s-t; z=zeros(size(s));

  N = [ v.*(2*v-1)  r.*(2*r-1)  s.*(2*s-1)  t.*(2*t-1)   ...
           4*r.*v      4*s.*r      4*s.*v     4*t.*v    4*r.*t    4*s.*t ];
  Nr = [ -4*v+1  4*r-1  z z   4*(v-r)  4*s -4*s     -4*t   4*t   z];
  Ns = [-4*v+1  z  4*s-1 z    -4*r     4*r  4*(v-s) -4*t    z   4*t];
  Nt = [-4*v+1  z z   4*t-1   -4*r     z   -4*s    4*(v-t) 4*r  4*s];
   


  out=struct('N',N,'Nr',Nr,'Ns',Ns,'Nt',Nt, ...
      'Nw',size(w,1),'Nnode',10,'NDN',zeros(size(N,2),size(N,1)*4), ...
      'NDNLabels',{{'',',x',',y',',z'}}, ...
      'jdet',zeros(size(w,1),1),'w',w,'DofLabels',{{'u','v','w'}}, ...
      'xi',xi,'type','tetra10');

% ---------------------------------------------------------------------
elseif comstr(Cam,'penta6')

  xi=[0 0 0; 1 0 0; 0 1 0; 
      0 0 1; 1 0 1; 0 1 1 ];
 if nargin<2; w='def'; else; w=varargin{carg};carg=carg+1;end
 [w,iN]=QuadraturePoints('p3d',w,xi);

  r=w(:,1); s=w(:,2); t=w(:,3); z=zeros(size(t));
  N = [ (1-r-s).*(1-t)   r.*(1-t)    s.*(1-t)  ...
           (1-r-s).*t       r.*t        s.*t   ]  ;
  Nr = [ -(1-t)   (1-t)   z  -t         t      z   ]  ;
  Ns = [ -(1-t)   z    (1-t) -t       z    t   ]  ;
  Nt = [ -(1-r-s)   -r    -s  ...
           (1-r-s)     r     s   ]  ;

 out=struct('N',N,'Nr',Nr,'Ns',Ns,'Nt',Nt, ...
      'Nw',size(w,1),'Nnode',6,'NDN',zeros(size(N,2),size(N,1)*4), ...
      'NDNLabels',{{'',',x',',y',',z'}}, ...
      'jdet',zeros(size(w,1),1),'w',w,'DofLabels',{{'u','v','w'}}, ...
      'xi',xi,'type','penta6');

% ---------------------------------------------------------------------
elseif comstr(Cam,'penta15')

  xi = [0 0 0;1 0 0;0 1 0;
        0 0 1;1 0 1;0 1 1;
        .5 0 0;.5 .5 0; 0 .5 0;
        0 0 .5;1 0 .5; 0 1 .5;
        .5 0 1;.5 .5 1; 0 .5 1];

 if nargin<2; w='def'; else; w=varargin{carg};carg=carg+1;end
 [w,iN]=QuadraturePoints('p3d',w,xi);

  % shape
  r=w(:,1); s=w(:,2); t=w(:,3); u=1-r-s; z=zeros(size(u));

  N = [ u.*(t-1).*(2*t-2*u+1) ...  % Bottom triangle
           r.*(t-1).*(2*t-2*r+1) ...
           s.*(t-1).*(2*t-2*s+1) ...
           u.*t.*(2*t+2*u-3) ... % Top triangle
           r.*t.*(2*t+2*r-3) ...
           s.*t.*(2*t+2*s-3) ... 
           4*u.*r.*(1-t) ... % bottom triangle mid nodes
           4*r.*s.*(1-t) ...
           4*s.*u.*(1-t) ... 
           4*u.*t.*(1-t) ... % mid-sides
           4*r.*t.*(1-t) ...
           4*s.*t.*(1-t) ...
           4*u.*r.*t ... % top triangle mid nodes
           4*r.*s.*t ...
           4*s.*u.*t ];  
  %
  Nr = [ (t-1).*(-2*t+4*u-1) ...  % Bottom triangle
           (t-1).*(2*t-4*r+1) ...
           z ...
           t.*(-2*t-4*u+3) ... % Top triangle
           t.*(2*t+4*r-3) ...
           z ... 
           4*(1-t).*(u-r) ... % bottom triangle mid nodes
            4*s.*(1-t) ...
           -4*s.*(1-t) ... 
           -4*t.*(1-t) ... % mid-sides
           4*t.*(1-t) ...
           z ...
            4*(-r+u).*t ... % top triangle mid nodes
            4*s.*t ...
           -4*s.*t ];  
  %
  Ns = [ (t-1).*(-2*t+4*u-1) ...  % Bottom triangle
           z ...
           (t-1).*(2*t-4*s+1) ...
           t.*(-2*t-4*u+3) ... % Top triangle
           z ...
           t.*(2*t+4*s-3) ... 
           4*-r.*(1-t) ... % bottom triangle mid nodes
           4* r.*(1-t) ...
           4*(u-s).*(1-t) ... 
           -4*t.*(1-t) ... % mid-sides
           z ...
            4*t.*(1-t) ...
           -4*r.*t ... % top triangle mid nodes
            4*r.*t ...
            4*(u-s).*t ];

  Nt = [u.*(4*t-2*u-1) ...  % Bottom triangle
           r.*(4*t-2*r-1) ...
           s.*(4*t-2*s-1) ...
           u.*(4*t+2*u-3) ... % Top triangle
           r.*(4*t+2*r-3) ...
           s.*(4*t+2*s-3) ... 
           -4*u.*r ... % bottom triangle mid nodes
           -4*r.*s ...
           -4*s.*u ... 
           4*u.*(1-2*t) ... % mid-sides
           4*r.*(1-2*t) ...
           4*s.*(1-2*t) ...
           4*u.*r ... % top triangle mid nodes
           4*r.*s ...
           4*s.*u ];  

  out=struct('N',N,'Nr',Nr,'Ns',Ns,'Nt',Nt, ...
      'Nw',size(w,1),'Nnode',15,'NDN',zeros(size(N,2),size(N,1)*4), ...
      'NDNLabels',{{'',',x',',y',',z'}}, ...
      'jdet',zeros(size(w,1),1),'w',w,'DofLabels',{{'u','v','w'}}, ...
      'xi',xi,'type','penta15');

% ---------------------------------------------------------------------
elseif comstr(Cam,'flui8')|comstr(Cam,'hexa8')

 xi = [-1 -1 -1;1 -1 -1;1 1 -1;-1 1 -1;-1 -1 1;1 -1 1;1 1 1;-1 1 1];
 % xi = [ 1 1 1; -1 1 1; -1 -1 1; 1 -1 1; 1 1 -1; -1 1 -1; -1 -1 -1; 1 -1 -1];
 typ=0;
 if nargin<2; w='def'; 
 else; 
   w=varargin{carg};carg=carg+1;
   if length(w)==1&w(end)>1e4; typ=fix(w/1e4); w=rem(w,1e4);end
 end 
 [w,iN]=QuadraturePoints('h3d',w,xi);

 if typ==1 % alternate shape functions
  r=w(:,1);s=w(:,2);t=w(:,3);z=zeros(size(t));
  N = [(1+r*xi(:,1)').*(1+s*xi(:,2)').*(1+t*xi(:,3)')/8 ... % classical
      (1-r.^2) (1-s.^2) (1-t.^2) % quadratic shapes for shear protection
      ];
  Nr=[xi(1:8,ones(1,size(w,1))*1)'.*(1+s*xi(:,2)').* ...
        (1+t*xi(:,3)')/8 -2*r z z];
  Ns=[xi(1:8,ones(1,size(w,1))*2)'.*(1+r*xi(:,1)').* ...
        (1+t*xi(:,3)')/8 z -2*s z];
  Nt=[xi(1:8,ones(1,size(w,1))*3)'.*(1+r*xi(:,1)').* ...
        (1+s*xi(:,2)')/8 z z -2*t];
 else
  N = (1+w(:,1)*xi(:,1)').*(1+w(:,2)*xi(:,2)').*(1+w(:,3)*xi(:,3)')/8;
  Nr=xi(1:8,ones(1,size(w,1))*1)'.*(1+w(:,2)*xi(:,2)').* ...
        (1+w(:,3)*xi(:,3)')/8;
  Ns=xi(1:8,ones(1,size(w,1))*2)'.*(1+w(:,1)*xi(:,1)').* ...
        (1+w(:,3)*xi(:,3)')/8;
  Nt=xi(1:8,ones(1,size(w,1))*3)'.*(1+w(:,1)*xi(:,1)').* ...
        (1+w(:,2)*xi(:,2)')/8;
 end
 out=struct('N',N,'Nr',Nr,'Ns',Ns,'Nt',Nt,'Nw',size(w,1), ...
      'NDN',zeros(size(N,2),4*size(N,1)),'NDNLabels',{{'',',x',',y',',z'}}, ...
      'jdet',zeros(size(N,1),1),'w',w,'Nnode',8, ...
      'DofLabels',{{'u','v','w'}},'xi',xi,'type','hexa8');

 if carg<=nargin % Possibly compute the derivatives OBSOLETE
      out=integrules('3d',out,varargin{carg});carg=carg+1;
 end

% ---------------------------------------------------------------------
elseif comstr(Cam,'hexa20') %

  xi = [-1 -1 -1;1 -1 -1;1 1 -1;-1 1 -1;-1 -1 1;1 -1 1;1 1 1;-1 1 1;
        0 -1 -1;1 0 -1;0 1 -1;-1 0 -1;-1 -1 0;1 -1 0;1 1 0;-1 1 0;
        0 -1  1;1 0  1;0 1  1;-1 0  1];

 if nargin<2; w='def'; else; w=varargin{carg};carg=carg+1;end
 [w,iN]=QuadraturePoints('h3d',w,xi);
 if ischar(w); w=comstr(w,-27); end
  

if 1==2
  % on [-1 1] ref. in Zienkiewicz vol.1 p133 and sdt4.1
  % corner nodes : (1+r)(1+s)(1+t)(r+s+t-2)/8
  % transformation : r=(r+1)/2; s=(s+1)/2; t=(t+1)/2; => on [0,1]
  r=w(:,1); s=w(:,2); t=w(:,3);
  for j1=1:8 % corner nodes
   ri=xi(j1,1);si=xi(j1,2); ti=xi(j1,3);
   N(:,j1)=(1+r*ri).*(1+s*si).*(1+t*ti).*(r*ri+s*si+t*ti-2)/8;
   %Nr(:,j1)=(1+s*si).*(1+t*ti)*(ri+2*r+s*si+t*ti-2)/8;
  end
  r2=r.^2; s2=s.^2; t2=t.^2;
  for j1=11:20 % midside nodes
   ri=xi(j1,1);si=xi(j1,2); ti=xi(j1,3);
   ri2=abs(ri)==0;si2=abs(si)==0;ti2=abs(ti)==0;
   N(:,j1)=(1+r*ri-ri2*r2).*(1+s*si-si2*s2).*(1+t*ti-ti2*t2)/4;
  end
end

  N =[(1+w(:,1)*xi(1:8,1)').*(1+w(:,2)*xi(1:8,2)').* ...
         (1+w(:,3)*xi(1:8,3)')/8 ... %na = 1/8 (1+xa x) (1 + ya y) (1 + za z)
   ... % n9  = 1/4 (1-x2 )(1-y)(1-z)
      (1-w(:,1).^2).*(1-w(:,2)).*(1-w(:,3))/4 ... 
   ... % n10 = 1/4 (1-y2 )(1+x)(1-z)
      (1-w(:,2).^2).*(1+w(:,1)).*(1-w(:,3))/4 ... 
   ... % n11 = 1/4 (1-x2 )(1+y)(1-z)
      (1-w(:,1).^2).*(1+w(:,2)).*(1-w(:,3))/4 ... 
   ... % n12 = 1/4 (1-y2 )(1-x)(1-z)
      (1-w(:,2).^2).*(1-w(:,1)).*(1-w(:,3))/4 ... 
   ... % n13 = 1/4 (1-z2 )(1-y)(1-x)
      (1-w(:,3).^2).*(1-w(:,2)).*(1-w(:,1))/4 ... 
   ... % n14 = 1/4 (1-z2 )(1+x)(1-y)
      (1-w(:,3).^2).*(1+w(:,1)).*(1-w(:,2))/4 ... 
   ... % n15 = 1/4 (1-z2 )(1+x)(1+u)
      (1-w(:,3).^2).*(1+w(:,1)).*(1+w(:,2))/4 ... 
   ... % n16 = 1/4 (1-z2 )(1-x)(1+y)
      (1-w(:,3).^2).*(1+w(:,2)).*(1-w(:,1))/4 ... 
   ... % n17 = 1/4 (1-x2 )(1-y)(1+z)
      (1-w(:,1).^2).*(1-w(:,2)).*(1+w(:,3))/4 ... 
   ... % n18 = 1/4 (1-y2 )(1+x)(1+z)
      (1-w(:,2).^2).*(1+w(:,1)).*(1+w(:,3))/4 ... 
   ... % n19 = 1/4 (1-x2 )(1+y)(1+z)
      (1-w(:,1).^2).*(1+w(:,2)).*(1+w(:,3))/4 ... 
   ... % n20 = 1/4 (1-y2 )(1-x)(1+z)
      (1-w(:,2).^2).*(1-w(:,1)).*(1+w(:,3))/4];

  Nr=[xi(1:8,ones(1,size(w,1))*1)'.*(1+w(:,2)*xi(1:8,2)').* ...
        (1+w(:,3)*xi(1:8,3)')/8 ...
      -w(:,1).*(1-w(:,2)).*(1-w(:,3))/2 ... 
      (1-w(:,2).^2).*(1-w(:,3))/4 ... 
      -w(:,1).*(1+w(:,2)).*(1-w(:,3))/2 ... 
      -(1-w(:,2).^2).*(1-w(:,3))/4 ... 
   ...
      -(1-w(:,3).^2).*(1-w(:,2))/4 ... 
       (1-w(:,3).^2).*(1-w(:,2))/4 ... 
       (1-w(:,3).^2).*(1+w(:,2))/4 ... 
      -(1-w(:,3).^2).*(1+w(:,2))/4 ... 
   ...
      -w(:,1).*(1-w(:,2)).*(1+w(:,3))/2 ... 
       (1-w(:,2).^2).*(1+w(:,3))/4 ... 
      -w(:,1).*(1+w(:,2)).*(1+w(:,3))/2 ... 
      -(1-w(:,2).^2).*(1+w(:,3))/4];

  Ns=[xi(1:8,ones(1,size(w,1))*2)'.*(1+w(:,1)*xi(1:8,1)').* ...
        (1+w(:,3)*xi(1:8,3)')/8 ...
      -(1-w(:,1).^2).*(1-w(:,3))/4 ... 
      -w(:,2).*(1+w(:,1)).*(1-w(:,3))/2 ... 
       (1-w(:,1).^2).*(1-w(:,3))/4 ... 
      -w(:,2).*(1-w(:,1)).*(1-w(:,3))/2 ... 
   ...
      -(1-w(:,3).^2).*(1-w(:,1))/4 ... 
      -(1-w(:,3).^2).*(1+w(:,1))/4 ... 
       (1-w(:,3).^2).*(1+w(:,1))/4 ... 
       (1-w(:,3).^2).*(1-w(:,1))/4 ... 
   ...
      -(1-w(:,1).^2).*(1+w(:,3))/4 ... 
      -w(:,2).*(1+w(:,1)).*(1+w(:,3))/2 ... 
       (1-w(:,1).^2).*(1+w(:,3))/4 ... 
      -w(:,2).*(1-w(:,1)).*(1+w(:,3))/2];
  Nt=[xi(1:8,ones(1,size(w,1))*3)'.*(1+w(:,1)*xi(1:8,1)').* ...
        (1+w(:,2)*xi(1:8,2)')/8 ...
      -(1-w(:,1).^2).*(1-w(:,2))/4 ... 
      -(1-w(:,2).^2).*(1+w(:,1))/4 ... 
      -(1-w(:,1).^2).*(1+w(:,2))/4 ... 
      -(1-w(:,2).^2).*(1-w(:,1))/4 ... 
   ...
      -w(:,3).*(1-w(:,2)).*(1-w(:,1))/2 ... 
      -w(:,3).*(1+w(:,1)).*(1-w(:,2))/2 ... 
      -w(:,3).*(1+w(:,1)).*(1+w(:,2))/2 ... 
      -w(:,3).*(1+w(:,2)).*(1-w(:,1))/2 ... 
   ...
      (1-w(:,1).^2).*(1-w(:,2))/4 ... 
      (1-w(:,2).^2).*(1+w(:,1))/4 ... 
      (1-w(:,1).^2).*(1+w(:,2))/4 ... 
      (1-w(:,2).^2).*(1-w(:,1))/4];
   i1 = [     9     9    10    11    13    14    15    16
             12    10    11    12    17    17    18    19
             13    14    15    16    20    18    19    20];
  for j1=1:8
    N (:,j1)=N (:,j1)-N (:,i1(:,j1))*[.5;.5;.5];  
    Nr(:,j1)=Nr(:,j1)-Nr(:,i1(:,j1))*[.5;.5;.5];  
    Ns(:,j1)=Ns(:,j1)-Ns(:,i1(:,j1))*[.5;.5;.5];  
    Nt(:,j1)=Nt(:,j1)-Nt(:,i1(:,j1))*[.5;.5;.5];  
  end

 out=struct('N',N,'Nr',Nr,'Ns',Ns,'Nt',Nt,'Nw',size(w,1), ...
      'NDN',zeros(size(N,2),size(N,1)*4), ...
      'NDNLabels',{{'',',x',',y',',z'}}, ...
      'jdet',zeros(size(w,1),1),'w',w,'DofLabels',{{'u','v','w'}}, ...
      'Nnode',20,'xi',xi,'type','hexa20');

 if carg<=nargin % Possibly compute the derivatives OBSOLETE
      out=integrules('3d',out,varargin{carg});carg=carg+1;
 end

% ---------------------------------------------------------------------
elseif comstr(Cam,'hexa21') %

 % xi gefdyn
 %  xi = [1 1 1;-1 1 1;-1 -1 1;1 -1 1;1 1 -1;-1 1 -1;-1 -1 -1;1 -1 -1;
 %       0 1 1;-1 0 1;0 -1 1;1 0 1;0 1 -1;-1 0 -1;0 -1 -1;1 0 -1;
 %       1 1 0; -1 1 0; -1 -1 0;1 -1 0; 0 0 0];

  N =[(1+FEw(:,1)*xi(1:8,1)').*(1+FEw(:,2)*xi(1:8,2)').* ...
         (1+FEw(:,3)*xi(1:8,3)')/8 ... %na = 1/8 (1+xa x) (1 + ya y) (1 + za z)
   ... % n9  = 1/4 (1-x2 )(1+y)(1+z)
      (1-FEw(:,1).^2).*(1+FEw(:,2)).*(1+FEw(:,3))/4 ... 
   ... % n10 = 1/4 (1-y2 )(1-x)(1+z)
      (1-FEw(:,2).^2).*(1-FEw(:,1)).*(1+FEw(:,3))/4 ... 
   ... % n11 = 1/4 (1-x2 )(1-y)(1+z)
      (1-FEw(:,1).^2).*(1-FEw(:,2)).*(1+FEw(:,3))/4 ... 
   ... % n12 = 1/4 (1-y2 )(1+x)(1+z)
      (1-FEw(:,2).^2).*(1+FEw(:,1)).*(1+FEw(:,3))/4 ... 
   ... % n13 = 1/4 (1-x2 )(1+y)(1-z)
      (1-FEw(:,1).^2).*(1+FEw(:,2)).*(1-FEw(:,3))/4 ... 
   ... % n14 = 1/4 (1-y2 )(1-x)(1-z)
      (1-FEw(:,2).^2).*(1-FEw(:,1)).*(1-FEw(:,3))/4 ... 
   ... % n15 = 1/4 (1-x2 )(1-y)(1-z)
      (1-FEw(:,1).^2).*(1-FEw(:,2)).*(1-FEw(:,3))/4 ...  % n16 = 1/4 (1-y2 )(1+x)(1-z)
      (1-FEw(:,2).^2).*(1+FEw(:,1)).*(1-FEw(:,3))/4 ...  % n17 = 1/4 (1-z2 )(1+x)(1+y)
      (1-FEw(:,3).^2).*(1+FEw(:,1)).*(1+FEw(:,2))/4 ... % n18 = 1/4 (1-z2 )(1-x)(1+y)
      (1-FEw(:,3).^2).*(1-FEw(:,1)).*(1+FEw(:,2))/4 ... % n19 = 1/4 (1-z2 )(1-x)(1-y)
      (1-FEw(:,3).^2).*(1-FEw(:,1)).*(1-FEw(:,2))/4 ... % n20 = 1/4 (1-z2 )(1+x)(1-y)
      (1-FEw(:,3).^2).*(1+FEw(:,1)).*(1-FEw(:,2))/4 ... % n21 = 1/4 (1-z2 )(1-x2)(1-y2)
      (1-FEw(:,3).^2).*(1-FEw(:,1).^2).*(1-FEw(:,2).^2)];

Nr=[xi(1:8,ones(1,size(FEw,1))*1)'.*(1+FEw(:,2)*xi(1:8,2)').* ...
        (1+FEw(:,3)*xi(1:8,3)')/8 ...
   ... % n9  = 1/4 (1-x2 )(1+y)(1+z)
      -FEw(:,1).*(1+FEw(:,2)).*(1+FEw(:,3))/2 ... 
   ... % n10 = 1/4 (1-y2 )(1-x)(1+z)
      -(1-FEw(:,2).^2).*(1+FEw(:,3))/4 ... 
   ... % n11 = 1/4 (1-x2 )(1-y)(1+z)
      -FEw(:,1).*(1-FEw(:,2)).*(1+FEw(:,3))/2 ... 
   ... % n12 = 1/4 (1-y2 )(1+x)(1+z)
      (1-FEw(:,2).^2).*(1+FEw(:,3))/4 ... 
   ... % n13 = 1/4 (1-x2 )(1+y)(1-z)
      -FEw(:,1).*(1+FEw(:,2)).*(1-FEw(:,3))/2 ... 
   ... % n14 = 1/4 (1-y2 )(1-x)(1-z)
      -(1-FEw(:,2).^2).*(1-FEw(:,3))/4 ... 
   ... % n15 = 1/4 (1-x2 )(1-y)(1-z)
      -FEw(:,1).*(1-FEw(:,2)).*(1-FEw(:,3))/2 ... 
   ... % n16 = 1/4 (1-y2 )(1+x)(1-z)
      (1-FEw(:,2).^2).*(1-FEw(:,3))/4 ... 
   ... % n17 = 1/4 (1-z2 )(1+x)(1+y)
      (1-FEw(:,3).^2).*(1+FEw(:,2))/4 ... 
   ... % n18 = 1/4 (1-z2 )(1-x)(1+y)
      -(1-FEw(:,3).^2).*(1+FEw(:,2))/4 ... 
   ... % n19 = 1/4 (1-z2 )(1-x)(1-y)
      -(1-FEw(:,3).^2).*(1-FEw(:,2))/4 ... 
   ... % n20 = 1/4 (1-z2 )(1+x)(1-y)
      (1-FEw(:,3).^2).*(1-FEw(:,2))/4 ...
... % n21 = 1/4 (1-z2 )(1-x2)(1-y2)
      -2*(1-FEw(:,3).^2).*FEw(:,1).*(1-FEw(:,2).^2)];
    


  Ns=[xi(1:8,ones(1,size(FEw,1))*2)'.*(1+FEw(:,1)*xi(1:8,1)').* ...
        (1+FEw(:,3)*xi(1:8,3)')/8 ...
   ... % n9  = 1/4 (1-x2 )(1+y)(1+z)
      (1-FEw(:,1).^2).*(1+FEw(:,3))/4 ... 
   ... % n10 = 1/4 (1-y2 )(1-x)(1+z)
      -FEw(:,2).*(1-FEw(:,1)).*(1+FEw(:,3))/2 ... 
   ... % n11 = 1/4 (1-x2 )(1-y)(1+z)
      -(1-FEw(:,1).^2).*(1+FEw(:,3))/4 ... 
   ... % n12 = 1/4 (1-y2 )(1+x)(1+z)
      -FEw(:,2).*(1+FEw(:,1)).*(1+FEw(:,3))/2 ... 
   ... % n13 = 1/4 (1-x2 )(1+y)(1-z)
      (1-FEw(:,1).^2).*(1-FEw(:,3))/4 ... 
   ... % n14 = 1/4 (1-y2 )(1-x)(1-z)
      -FEw(:,2).*(1-FEw(:,1)).*(1-FEw(:,3))/2 ... 
   ... % n15 = 1/4 (1-x2 )(1-y)(1-z)
      -(1-FEw(:,1).^2).*(1-FEw(:,3))/4 ... 
   ... % n16 = 1/4 (1-y2 )(1+x)(1-z)
      -FEw(:,2).*(1+FEw(:,1)).*(1-FEw(:,3))/2 ... 
   ... % n17 = 1/4 (1-z2 )(1+x)(1+y)
      (1-FEw(:,3).^2).*(1+FEw(:,1))/4 ... 
   ... % n18 = 1/4 (1-z2 )(1-x)(1+y)
      (1-FEw(:,3).^2).*(1-FEw(:,1))/4 ... 
   ... % n19 = 1/4 (1-z2 )(1-x)(1-y)
      -(1-FEw(:,3).^2).*(1-FEw(:,1))/4 ... 
   ... % n20 = 1/4 (1-z2 )(1+x)(1-y)
      -(1-FEw(:,3).^2).*(1+FEw(:,1))/4 ...
... % n21 = 1/4 (1-z2 )(1-x2)(1-y2)
      -2*(1-FEw(:,3).^2).*(1-FEw(:,1).^2).*FEw(:,2)];



  Nt=[xi(1:8,ones(1,size(FEw,1))*3)'.*(1+FEw(:,1)*xi(1:8,1)').* ...
        (1+FEw(:,2)*xi(1:8,2)')/8 ...

    ... % n9  = 1/4 (1-x2 )(1+y)(1+z)
      (1-FEw(:,1).^2).*(1+FEw(:,2))/4 ... 
   ... % n10 = 1/4 (1-y2 )(1-x)(1+z)
      (1-FEw(:,2).^2).*(1-FEw(:,1))/4 ... 
   ... % n11 = 1/4 (1-x2 )(1-y)(1+z)
      (1-FEw(:,1).^2).*(1-FEw(:,2))/4 ... 
   ... % n12 = 1/4 (1-y2 )(1+x)(1+z)
      (1-FEw(:,2).^2).*(1+FEw(:,1))/4 ... 
   ... % n13 = 1/4 (1-x2 )(1+y)(1-z)
      -(1-FEw(:,1).^2).*(1+FEw(:,2))/4 ... 
   ... % n14 = 1/4 (1-y2 )(1-x)(1-z)
      -(1-FEw(:,2).^2).*(1-FEw(:,1))/4 ... 
   ... % n15 = 1/4 (1-x2 )(1-y)(1-z)
      -(1-FEw(:,1).^2).*(1-FEw(:,2))/4 ... 
   ... % n16 = 1/4 (1-y2 )(1+x)(1-z)
      -(1-FEw(:,2).^2).*(1+FEw(:,1))/4 ... 
   ... % n17 = 1/4 (1-z2 )(1+x)(1+y)
      -FEw(:,3).*(1+FEw(:,1)).*(1+FEw(:,2))/2 ... 
   ... % n18 = 1/4 (1-z2 )(1-x)(1+y)
      -FEw(:,3).*(1-FEw(:,1)).*(1+FEw(:,2))/2 ... 
   ... % n19 = 1/4 (1-z2 )(1-x)(1-y)
      -FEw(:,3).*(1-FEw(:,1)).*(1-FEw(:,2))/2 ... 
   ... % n20 = 1/4 (1-z2 )(1+x)(1-y)
      -FEw(:,3).*(1+FEw(:,1)).*(1-FEw(:,2))/2 ...
... % n21 = 1/4 (1-z2 )(1-x2)(1-y2)
      -2*FEw(:,3).*(1-FEw(:,1).^2).*(1-FEw(:,2).^2)];


  
  i1 = [     9     9    11    11    13   13  15    15    
             12    10   10    12    16   14  14    16    
             17    18   19    20    17   18  19    20    ];
  for j1=1:8
    N (:,j1)=N (:,j1)-N (:,i1(:,j1))*[.5;.5;.5];  
    Nr(:,j1)=Nr(:,j1)-Nr(:,i1(:,j1))*[.5;.5;.5];  
    Ns(:,j1)=Ns(:,j1)-Ns(:,i1(:,j1))*[.5;.5;.5];  
    Nt(:,j1)=Nt(:,j1)-Nt(:,i1(:,j1))*[.5;.5;.5];  
end

%
% si le noeuds 21 existe
%

 for j1=1:8
    N (:,j1)=N (:,j1)-N (:,21)*.125;  
    Nr (:,j1)=Nr (:,j1)-Nr (:,21)*.125;  
    Ns (:,j1)=Ns (:,j1)-Ns (:,21)*.125;  
    Nt (:,j1)=Nt (:,j1)-Nt (:,21)*.125;   
end
 for j1=9:20
    N (:,j1)=N (:,j1)-N (:,21)*.25;  
    Nr (:,j1)=Nr (:,j1)-Nr (:,21)*.25;  
    Ns (:,j1)=Ns (:,j1)-Ns (:,21)*.25;  
    Nt (:,j1)=Nt (:,j1)-Nt (:,21)*.25;   
end

 out=struct('N',N,'Nr',Nr,'Ns',Ns,'Nt',Nt, ...
      'NDN',zeros(size(N,2),size(N,1)*4),'NDNLabels',{{'',',x',',y',',z'}}, ...
      'jdet',zeros(size(FEw,1),1),'w',FEw,'DofLabels',{{'u','v','w'}}, ...
      'Nnode',20,'Nw',size(FEw,1),'type','hexa20');

% ---------------------------------------------------------------------
elseif comstr(Cam,'hexa27')

 xi=[-1,1,1,-1, -1,1,1,-1, 0,1,0,-1, -1,1,1,-1,0,1,0,-1,0.,-1.,0.,0.,1.,0.,0.;
    -1, -1,1,1,  -1,-1,1,1, -1,0,1,0,-1,-1,1,1,-1,0,1,0,0.,0.,-1.,0.,0.,1.,0.;
   -1,-1,-1,-1, 1,1,1,1,  -1,-1,-1,-1, 0,0,0,0, 1,1,1,1,-1.,0.,0.,1.,0.,0.,0]';
 if nargin<2; w='def'; else; w=varargin{carg};carg=carg+1;end
 if ischar(w); w=comstr(w,-27); end

 if ~ischar(w)&size(w,2)==4
 elseif comstr(w,'node'); w=xi;w(:,4)=1/2/size(xi,1);
 elseif comstr(w,'center'); w=[1/3 1/3 .5 1/2];
 else

   Z=[-.774596669241483 0  .774596669241483]';
   W=[5/9,8/9,5/9]';
   i1=[1 1 1;1 1 2;1 1 3;1 2 1;1 2 2;1 2 3;1 3 1;1 3 2;1 3 3;
    2 1 1;2 1 2;2 1 3;2 2 1;2 2 2;2 2 3;2 3 1;2 3 2;2 3 3;
    3 1 1;3 1 2;3 1 3;3 2 1;3 2 2;3 2 3;3 3 1;3 3 2;3 3 3];
   w=[Z(i1) prod(W(i1),2)];
  end

for jW=1:size(w,1); r=w(jW,1); s=w(jW,2);z=w(jW,3);

% Fonctions de base 
l(1,jW)=-1/8*(1-r)*r*(1-s)*s*(1-z)*z;
l(2,jW)=1/8*(1+r)*r*(1-s)*s*(1-z)*z;
l(3,jW)=-1/8*(1+r)*r*(1+s)*s*(1-z)*z;
l(4,jW)=1/8*(1-r)*r*(1+s)*s*(1-z)*z;
l(5,jW)=1/8*(1-r)*r*(1-s)*s*(1+z)*z;
l(6,jW)=-1/8*(1+r)*r*(1-s)*s*(1+z)*z;
l(7,jW)=1/8*(1+r)*r*(1+s)*s*(1+z)*z;
l(8,jW)=-1/8*(1-r)*r*(1+s)*s*(1+z)*z;
l(9,jW)=.25*(1-r^2)*(1-s)*s*(1-z)*z;
l(10,jW)=-.25*(1+r)*r*(1-s^2)*(1-z)*z;
l(11,jW)=-.25*(1-r^2)*(1+s)*s*(1-z)*z;
l(12,jW)=.25*(1-r)*r*(1-s^2)*(1-z)*z;
l(13,jW)=.25*(1-r)*r*(1-s)*s*(1-z^2);
l(14,jW)=-.25*(1+r)*r*(1-s)*s*(1-z^2);
l(15,jW)=.25*(1+r)*r*(1+s)*s*(1-z^2);
l(16,jW)=-.25*(1-r)*r*(1+s)*s*(1-z^2);
l(17,jW)=-.25*(1-r^2)*(1-s)*s*(1+z)*z;
l(18,jW)=.25*(1+r)*r*(1-s^2)*(1+z)*z;
l(19,jW)=.25*(1-r^2)*(1+s)*s*(1+z)*z;
l(20,jW)=-.25*(1-r)*r*(1-s^2)*(1+z)*z;
l(21,jW)=-.5*(1-r^2)*(1-s^2)*z*(1-1.*z);
l(22,jW)=-.5*(1-1.*r)*r*(1-s^2)*(1-z^2);
l(23,jW)=-.5*(1-r^2)*s*(1-1.*s)*(1-z^2);
l(24,jW)=.5*(1-r^2)*(1-s^2)*z*(1+1.*z);
l(25,jW)=.5*(1+1.*r)*r*(1-s^2)*(1-z^2);
l(26,jW)=.5*(1-r^2)*s*(1+1.*s)*(1-z^2);
l(27,jW)=(1-r^2)*(1-s^2)*(1-z^2);

% Derivees en r
lr(1,jW)=1/8*r*(1-s)*s*(1-z)*z-1/8*(1-r)*(1-s)*s*(1-z)*z;
lr(2,jW)=1/8*r*(1-s)*s*(1-z)*z+1/8*(1+r)*(1-s)*s*(1-z)*z;
lr(3,jW)=-1/8*r*(1+s)*s*(1-z)*z-1/8*(1+r)*(1+s)*s*(1-z)*z;
lr(4,jW)=-1/8*r*(1+s)*s*(1-z)*z+1/8*(1-r)*(1+s)*s*(1-z)*z;
lr(5,jW)=-1/8*r*(1-s)*s*(1+z)*z+1/8*(1-r)*(1-s)*s*(1+z)*z;
lr(6,jW)=-1/8*r*(1-s)*s*(1+z)*z-1/8*(1+r)*(1-s)*s*(1+z)*z;
lr(7,jW)=1/8*r*(1+s)*s*(1+z)*z+1/8*(1+r)*(1+s)*s*(1+z)*z;
lr(8,jW)=1/8*r*(1+s)*s*(1+z)*z-1/8*(1-r)*(1+s)*s*(1+z)*z;
lr(9,jW)=-.50*r*(1-s)*s*(1-z)*z;
lr(10,jW)=-.25*r*(1-s^2)*(1-z)*z-.25*(1+r)*(1-s^2)*(1-z)*z;
lr(11,jW)=.50*r*(1+s)*s*(1-z)*z;
lr(12,jW)=-.25*r*(1-s^2)*(1-z)*z+.25*(1-r)*(1-s^2)*(1-z)*z;
lr(13,jW)=-.25*r*(1-s)*s*(1-z^2)+.25*(1-r)*(1-s)*s*(1-z^2);
lr(14,jW)=-.25*r*(1-s)*s*(1-z^2)-.25*(1+r)*(1-s)*s*(1-z^2);
lr(15,jW)=.25*r*(1+s)*s*(1-z^2)+.25*(1+r)*(1+s)*s*(1-z^2);
lr(16,jW)=.25*r*(1+s)*s*(1-z^2)-.25*(1-r)*(1+s)*s*(1-z^2);
lr(17,jW)=.50*r*(1-s)*s*(1+z)*z;
lr(18,jW)=.25*r*(1-s^2)*(1+z)*z+.25*(1+r)*(1-s^2)*(1+z)*z;
lr(19,jW)=-.50*r*(1+s)*s*(1+z)*z;
lr(20,jW)=.25*r*(1-s^2)*(1+z)*z-.25*(1-r)*(1-s^2)*(1+z)*z;
lr(21,jW)=1.0*r*(1-s^2)*z*(1-1.*z);
lr(22,jW)=.5*r*(1-s^2)*(1-z^2)-.5*(1-1.*r)*(1-s^2)*(1-z^2);
lr(23,jW)=1.0*r*s*(1-1.*s)*(1-z^2);
lr(24,jW)=-1.0*r*(1-s^2)*z*(1+1.*z);
lr(25,jW)=.5*r*(1-s^2)*(1-z^2)+.5*(1+1.*r)*(1-s^2)*(1-z^2);
lr(26,jW)=-1.0*r*s*(1+1.*s)*(1-z^2);
lr(27,jW)=-2*r*(1-s^2)*(1-z^2);

% Derivees en s
ls(1,jW)=1/8*(1-r)*r*s*(1-z)*z-1/8*(1-r)*r*(1-s)*(1-z)*z;
ls(2,jW)=-1/8*(1+r)*r*s*(1-z)*z+1/8*(1+r)*r*(1-s)*(1-z)*z;
ls(3,jW)=-1/8*(1+r)*r*s*(1-z)*z-1/8*(1+r)*r*(1+s)*(1-z)*z;
ls(4,jW)=1/8*(1-r)*r*s*(1-z)*z+1/8*(1-r)*r*(1+s)*(1-z)*z;
ls(5,jW)=-1/8*(1-r)*r*s*(1+z)*z+1/8*(1-r)*r*(1-s)*(1+z)*z;
ls(6,jW)=1/8*(1+r)*r*s*(1+z)*z-1/8*(1+r)*r*(1-s)*(1+z)*z;
ls(7,jW)=1/8*(1+r)*r*s*(1+z)*z+1/8*(1+r)*r*(1+s)*(1+z)*z;
ls(8,jW)=-1/8*(1-r)*r*s*(1+z)*z-1/8*(1-r)*r*(1+s)*(1+z)*z;
ls(9,jW)=-.25*(1-r^2)*s*(1-z)*z+.25*(1-r^2)*(1-s)*(1-z)*z;
ls(10,jW)=.50*(1+r)*r*s*(1-z)*z;
ls(11,jW)=-.25*(1-r^2)*s*(1-z)*z-.25*(1-r^2)*(1+s)*(1-z)*z;
ls(12,jW)=-.50*(1-r)*r*s*(1-z)*z;
ls(13,jW)=-.25*(1-r)*r*s*(1-z^2)+.25*(1-r)*r*(1-s)*(1-z^2);
ls(14,jW)=.25*(1+r)*r*s*(1-z^2)-.25*(1+r)*r*(1-s)*(1-z^2);
ls(15,jW)=.25*(1+r)*r*s*(1-z^2)+.25*(1+r)*r*(1+s)*(1-z^2);
ls(16,jW)=-.25*(1-r)*r*s*(1-z^2)-.25*(1-r)*r*(1+s)*(1-z^2);
ls(17,jW)=.25*(1-r^2)*s*(1+z)*z-.25*(1-r^2)*(1-s)*(1+z)*z;
ls(18,jW)=-.50*(1+r)*r*s*(1+z)*z;
ls(19,jW)=.25*(1-r^2)*s*(1+z)*z+.25*(1-r^2)*(1+s)*(1+z)*z;
ls(20,jW)=.50*(1-r)*r*s*(1+z)*z;
ls(21,jW)=1.0*(1-r^2)*s*z*(1-1.*z);
ls(22,jW)=1.0*(1-1.*r)*r*s*(1-z^2);
ls(23,jW)=-.5*(1-r^2)*(1-1.*s)*(1-z^2)+.5*(1-r^2)*s*(1-z^2);
ls(24,jW)=-1.0*(1-r^2)*s*z*(1+1.*z);
ls(25,jW)=-1.0*(1+1.*r)*r*s*(1-z^2);
ls(26,jW)=.5*(1-r^2)*(1+1.*s)*(1-z^2)+.5*(1-r^2)*s*(1-z^2);
ls(27,jW)=-2*(1-r^2)*s*(1-z^2);

% Derivees en z
lz(1,jW)=1/8*(1-r)*r*(1-s)*s*z-1/8*(1-r)*r*(1-s)*s*(1-z);
lz(2,jW)=-1/8*(1+r)*r*(1-s)*s*z+1/8*(1+r)*r*(1-s)*s*(1-z);
lz(3,jW)=1/8*(1+r)*r*(1+s)*s*z-1/8*(1+r)*r*(1+s)*s*(1-z);
lz(4,jW)=-1/8*(1-r)*r*(1+s)*s*z+1/8*(1-r)*r*(1+s)*s*(1-z);
lz(5,jW)=1/8*(1-r)*r*(1-s)*s*z+1/8*(1-r)*r*(1-s)*s*(1+z);
lz(6,jW)=-1/8*(1+r)*r*(1-s)*s*z-1/8*(1+r)*r*(1-s)*s*(1+z);
lz(7,jW)=1/8*(1+r)*r*(1+s)*s*z+1/8*(1+r)*r*(1+s)*s*(1+z);
lz(8,jW)=-1/8*(1-r)*r*(1+s)*s*z-1/8*(1-r)*r*(1+s)*s*(1+z);
lz(9,jW)=-.25*(1-r^2)*(1-s)*s*z+.25*(1-r^2)*(1-s)*s*(1-z);
lz(10,jW)=.25*(1+r)*r*(1-s^2)*z-.25*(1+r)*r*(1-s^2)*(1-z);
lz(11,jW)=.25*(1-r^2)*(1+s)*s*z-.25*(1-r^2)*(1+s)*s*(1-z);
lz(12,jW)=-.25*(1-r)*r*(1-s^2)*z+.25*(1-r)*r*(1-s^2)*(1-z);
lz(13,jW)=-.50*(1-r)*r*(1-s)*s*z;
lz(14,jW)=.50*(1+r)*r*(1-s)*s*z;
lz(15,jW)=-.50*(1+r)*r*(1+s)*s*z;
lz(16,jW)=.50*(1-r)*r*(1+s)*s*z;
lz(17,jW)=-.25*(1-r^2)*(1-s)*s*z-.25*(1-r^2)*(1-s)*s*(1+z);
lz(18,jW)=.25*(1+r)*r*(1-s^2)*z+.25*(1+r)*r*(1-s^2)*(1+z);
lz(19,jW)=.25*(1-r^2)*(1+s)*s*z+.25*(1-r^2)*(1+s)*s*(1+z);
lz(20,jW)=-.25*(1-r)*r*(1-s^2)*z-.25*(1-r)*r*(1-s^2)*(1+z);
lz(21,jW)=-.5*(1-r^2)*(1-s^2)*(1-1.*z)+.5*(1-r^2)*(1-s^2)*z;
lz(22,jW)=1.0*(1-1.*r)*r*(1-s^2)*z;
lz(23,jW)=1.0*(1-r^2)*s*(1-1.*s)*z;
lz(24,jW)=.5*(1-r^2)*(1-s^2)*(1+1.*z)+.5*(1-r^2)*(1-s^2)*z;
lz(25,jW)=-1.0*(1+1.*r)*r*(1-s^2)*z;
lz(26,jW)=-1.0*(1-r^2)*s*(1+1.*s)*z;
lz(27,jW)=-2*(1-r^2)*(1-s^2)*z;

end %jW

out=struct('N',l','Nr',lr','Ns',ls','Nt',lz','xi',xi, ...
  'NDN',zeros(27,4*size(w,1)), ...
      'NDNLabels',{{'',',x',',y',',z'}}, ...
      'jdet',zeros(size(w,1),1),'w',w,'DofLabels',{{'u','v','w'}}, ...
      'Nnode',27,'Nw',size(w,1),'xi',xi,'type','hexa27');



%------------------------------------------------------------------------------
% Display and generation of matrix integration rules based on StrainDefinition
% and ConstitTopology
elseif comstr(Cam,'matrixrule')  

 opt=varargin{carg};carg=carg+1;
 sdx=opt.NDNLabels;
 su=opt.DofLabels;
 if isfield(opt,'PerField'); PerField=opt.PerField;
 else; PerField=size(opt.N,2);
 end

for j0=1:length(opt.StrainDefinition)
 if isempty(opt.StrainDefinition{j0}); opt.MatrixIntegrationRule{j0}=[];
 else
 ss=opt.StrainLabels{j0};
 dd=double(opt.ConstitTopology{j0});
 ep=double(opt.StrainDefinition{j0});
 out=[];
 % Dof1 Dof2 NDN1 NDN2 Constit StepConstit StepNW NwIni
 for j1=1:size(dd,1); for j1b=find(ep(:,1)==j1)';
   for j2=1:size(dd,2); for j2b=find(ep(:,1)==j2)';

   i2=sign(ep(j1b,2)*double(ep(j2b,2)))*dd(j1,j2); if i2~=0
    out(end+1,1:8)=[ ...
     (ep(j1b,3)-1)*PerField,(ep(j2b,3)-1)*PerField, ... % DDL1, DDL2
     (abs(ep(j1b,2))-1)*opt.Nw+abs(ep(j1b,4))-1, ... % NDNi
     (abs(ep(j2b,2))-1)*opt.Nw+abs(ep(j2b,4))-1, ... % NDNj
     sign(ep(j1b,2)*ep(j2b,2))*(dd(j1,j2)-1), ... % constit
     0,min(ep(j2b,5),ep(j2b,5)),ep(j2b,4)-1]; % ConsStep Nw W0
    if nargout==0
     fprintf( ...
      '   %2i %2i   %2i %2i  %3i    %i %i %i %% G(%s,%s) * %s%s * %s%s  \n',...
      out(end,:), ...
      ss{j1},ss{j2},su{abs(ep(j1b,3))},sdx{abs(ep(j1b,2))}, ...
      su{abs(ep(j2b,3))},sdx{abs(ep(j2b,2))});
    end
    end
   end;end
 end;end
 opt.MatrixIntegrationRule{j0}=int32(out);
 end % empty matrix or not
end
if nargout>0;out=opt;end

%------------------------------------------------------------------------------
% Display and generation of stress integration rules based on StrainDefinition
% and ConstitTopology
elseif comstr(Cam,'texstrain')

opt=varargin{carg};carg=carg+1;
fid=1; fprintf(fid,'\n');

for j0=1:length(opt.StrainDefinition)

 fprintf(fid,'\\begin{equation}\n');
 st=opt.StrainLabels{j0};st=sprintf('%s \\\\',st{:});
 fprintf('\\ve{\\ba{c}%s\\ea}\n',st(1:end-2))

 r1=opt.StrainDefinition{j0}; st=opt.StrainLabels{j0};
 r2=cell(length(unique(r1(:,1))),length(unique(r1(:,3))));
 for j1=1:size(r1,1)
  if r1(j1,2)>0;r2{r1(j1,1),r1(j1,3)}=sprintf('N%s',opt.NDNLabels{r1(j1,2)});
  else;r2{r1(j1,1),r1(j1,3)}=sprintf('-N%s',opt.NDNLabels{-r1(j1,2)});
  end
 end
 st='c'; fprintf(fid,'=\\ma{\\ba{%s}\n',st(ones(1,size(r2,1))));
 for j1=1:prod(size(r2)); if isempty(r2{j1}) r2{j1}='0';end;end
 st=[' %s &']';st=st(:,ones(size(r2,2),1));st=st(:)';st(end+[0:5])='\\\\\n';
 r2=r2';st=sprintf(st,r2{:});st(end+[-2:1])='\ea}';
 fprintf(fid,'%s',st);
 st=opt.DofLabels;st=sprintf(' %s \\\\',st{:});
 fprintf('\n\\ve{\\ba{c}%s\\ea}\n\\end{equation}\n\n',st(1:end-2))

end
clear out;

  
%------------------------------------------------------------------------------
% Display and generation of stress integration rules based on StrainDefinition
% and ConstitTopology
elseif comstr(Cam,'stressrule')  

opt=varargin{carg};carg=carg+1;
if carg<=nargin % Possibly evaluate stress elsewhere than integration point
 st=varargin{carg};carg=carg+1;
 r1=integrules(opt.type,st);
 st={'N','Nr','Ns','Nt','NDN','jdet','w','Nnode','Nw'};
 for j1=1:length(st)
   try; eval(sprintf('opt.%s=r1.%s;',st{j1},st{j1}));end
 end
end

if ~isfield(opt,'StressRule'); opt.StressRule={};end
nNode=size(opt.N,2);
for j0=1:length(opt.StrainDefinition)
 ss=opt.StrainLabels{j0};
 dd=double(opt.ConstitTopology{j0});
 ep=double(opt.StrainDefinition{j0});
 out=[];
 rule=[];
 % sigma = dd * ep
 for j1=1:size(dd,1);for j2=1:size(dd,2); for j2b=find(ep(:,1)==j2)';
 % StressRowValue NDNColumn ConstitCoefPos DOFOffSet nStres]
 if dd(j1,j2)~=0
  rule(end+1,:)=[j1-1 (abs(ep(j2b,2))-1)*opt.Nw dd(j1,j2)-1 (ep(j2b,3)-1) size(dd,1)];
 end % dd~=0
 end;end;end
 opt.StressRule{j0}=int32(rule);
end

if nargout>0;out=opt;else; integrules('texstress',opt); end

%------------------------------------------------------------------------------
% Display and generation of stress integration rules based on StrainDefinition
% and ConstitTopology 
elseif comstr(Cam,'texmacros')


st={'You may need to define the following macros',
 '\newcommand{\ba}[1]{\begin{array}{#1}}'
'\newcommand{\ea}{\end{array}}'
'\newcommand{\ve}[1]{\left\{#1\right\}}'
'\newcommand{\ma}[1]{\left[#1\right]}';''};
fprintf('%s\n',st{:})
clear out; 

%------------------------------------------------------------------------------
% Display and generation of stress integration rules based on StrainDefinition
% and ConstitTopology 
elseif comstr(Cam,'texstress')

opt=varargin{carg};carg=carg+1;
fid=1; integrules('texmacros');fprintf(fid,'\n');

for j0=1:length(opt.StressRule)

 fprintf(fid,'\\begin{equation}\n');
 st=opt.StrainLabels{j0};st=sprintf('\\sigma %s \\\\',st{:});
 fprintf('\\ve{\\ba{c}%s\\ea}\n',st(1:end-2));

 r1=double(opt.StressRule{j0});
 r2=cell(length(unique(r1(:,1))),length(unique(r1(:,4))));
 for j1=1:size(r1,1)
  st2=r2{r1(j1,1)+1,r1(j1,4)+1}; 
  [i1,i2]=find(opt.ConstitTopology{j0}==(r1(j1,3)+1));i1=i1(1);i2=i2(1);
  st3=sprintf('d_{%i,%i} N%s',i1,i2,opt.NDNLabels{r1(j1,2)/opt.Nw+1});
  if isempty(st2) r2{r1(j1,1)+1,r1(j1,4)+1}=st3;
  else;  r2{r1(j1,1)+1,r1(j1,4)+1}=sprintf('%s+%s',st2,st3);
  end
 end

 st='c'; fprintf(fid,'=\\ma{\\ba{%s}\n',st(ones(1,size(r2,1))));
 for j1=1:prod(size(r2)); if isempty(r2{j1}) r2{j1}='0';end;end
 st=[' %s &']';st=st(:,ones(size(r2,2),1));st=st(:)';st(end+[0:5])='\\\\\n';
 r2=r2';st=sprintf(st,r2{:});st(end+[-2:1])='\ea}';
 fprintf(fid,'%s',st);
 st=opt.DofLabels;st=sprintf(' %s \\\\',st{:});
 fprintf('\n\\ve{\\ba{c}%s\\ea}\n\\end{equation}\n\n',st(1:end-2))

end
clear out;

% 2D Tensor transformation code generation
if 1==2

 st=cell(4,4); N=2;
 for j1=0:N-1; for j2=0:N-1;for j3=0:N-1;for j4=0:N-1;
  st1=st{j1*N+j4+1,j2*N+j3+1}; if isempty(st1); st1='';else st1(end+1)='+';end
  st{j1*N+j4+1,j2*N+j3+1}=sprintf('%sJ[%i*%i+%i]*J[%i*%i+%i]',st1, ...
   j2,N,j1,j3,N,j4);
 end;end;end;end

end

% Node positions for gefdyn-------------------------------------------------
% OBSOLETE
elseif comstr(Cam,'gef'); [CAM,Cam]=comstr(CAM,4);

if nargin<2; w='def'; else; w=varargin{carg};carg=carg+1;end

if comstr(Cam,'bar1');      xi=[0 0 0;1 0 0];
elseif comstr(Cam,'beam1'); xi=[0 0 0;1 0 0];
elseif comstr(Cam,'beam3'); xi=[0 0 0;1 0 0;.5 0 0];
elseif comstr(Cam,'quad4'); xi=[1 1 0;-1 1 0;-1 -1 0;1 -1 0]; 
elseif comstr(Cam,'quadb');
 xi= [1 1 0;-1 1 0;-1 -1 0;1 -1 0;0 1 0;-1 0 0;0 -1 0; 1 0 0];
elseif comstr(Cam,'tria3');
 xi=[1 0 0;0 1 0;0 0 0]; %GEFDYN
elseif comstr(Cam,'tria6');
 xi=[1 0 0;0 1 0;0 0 0;.5 0.5 0;0 .5 0;0.5 0 0]; % GEFDYN 
else; error('unknown');
end
out=integrules(horzcat(Cam,'new'),w,xi);

% Integration rules for 2D jacobian -----------------------------
% OBSOLETE use BuildNdN
elseif comstr(Cam,'2d') 

  out=varargin{carg};carg=carg+1;
  x=varargin{carg};carg=carg+1;
      xr = out.Nr*x(:,1); xs = out.Ns*x(:,1);
      yr = out.Nr*x(:,2); ys = out.Ns*x(:,2);
      jdet = xr.*ys-xs.*yr;  jdet=jdet*sign(jdet(1));
      if any(jdet<0); disp('quad4 negative Jacobian');end
      i3=ones(size(out.Nr,2),1); % number of shape functions
      out.Nx = [ ys(:,i3).*out.Nr-yr(:,i3).*out.Ns]./jdet(:,i3);
      out.Ny = [-xs(:,i3).*out.Nr+xr(:,i3).*out.Ns]./jdet(:,i3);;
      out.jdet=jdet;

% Integration rules for 3D jacobian -----------------------------
% OBSOLETE use BuildNdN
elseif comstr(Cam,'3d') 

  out=varargin{carg};carg=carg+1;
  x=varargin{carg};carg=carg+1;
      Nr=out.Nr; Ns=out.Ns; Nt=out.Nt; 
      xr = Nr*x(:,1); xs = Ns*x(:,1); xt = Nt*x(:,1);
      yr = Nr*x(:,2); ys = Ns*x(:,2); yt = Nt*x(:,2);
      zr = Nr*x(:,3); zs = Ns*x(:,3); zt = Nt*x(:,3);

      cof11 = ys.*zt - yt.*zs;
      cof12 = yt.*zr - yr.*zt;
      cof13 = yr.*zs - ys.*zr;
      cof21 = zs.*xt - zt.*xs;
      cof22 = zt.*xr - zr.*xt;
      cof23 = zr.*xs - zs.*xr;
      cof31 = xs.*yt - xt.*ys;
      cof32 = xt.*yr - xr.*yt;
      cof33 = xr.*ys - xs.*yr;

      jdet = xr.*cof11+xs.*cof12+xt.*cof13;
      jdet=jdet*sign(jdet(1));
      if any(jdet<0); warning('negative Jacobian');end
      i3=ones(size(Nr,2),1); % number of shape functions
      out.jdet=jdet;jdet=jdet(:,i3);
      out.Nx=[Nr.*cof11(:,i3)+Ns.*cof12(:,i3)+Nt.*cof13(:,i3)]./jdet;
      out.Ny=[Nr.*cof21(:,i3)+Ns.*cof22(:,i3)+Nt.*cof23(:,i3)]./jdet; 
      out.Nz=[Nr.*cof31(:,i3)+Ns.*cof32(:,i3)+Nt.*cof33(:,i3)]./jdet; 

% ---------------------------------------------------------------------
elseif comstr(Cam,'gauss');[CAM,Cam]=comstr(CAM,6);

 if nargin==1&isempty(Cam)
  out={'1d' 'lines';'q2d' '2D quadrangles';
       't2d','2D triangles';
       't3d','3D tetrahedrons '
       'p3d','3D pentahedrons (prisms)';
       'h3d','3D hexahedrons (brick)'};
  if nargout==0; 
   out=out';fprintf('\nAvailable rule families\n\n')
   fprintf('integrules(''Gauss %-4s'') %% %s\n',out{:});clear out;
  end
 elseif nargin==1
  out=QuadraturePoints(CAM,'list');
 else
  [out,out1]=QuadraturePoints(CAM,varargin{carg:end});
 end

% ---------------------------------------------------------------------
else; sdtw('_nb','''%s'' is not a known command');
end

% ---------------------------------------------------------------------
function [w,iN]=QuadraturePoints(topo,w,xi,varargin)

if nargin<2; w='def';
elseif ischar(w); 
 w=comstr(w,-27); 
 if comstr(w,'node'); w=-2;
 elseif comstr(w,'center'); w=-1;
 end 
elseif length(w)==1; w=double(w); if isequal(w,0); w='def';end
end
if nargin<3; xi=[];iN=[];
elseif nargin>3  % Allow for reordering of reference nodes
 xi1=varargin{1};carg=2;
 [xi2,i1,i2]=intersect(xi,xi1,'rows');
 iN=[];iN(i1)=i2;
else; iN=1:size(xi,1);
end % reference node reordering

if ~ischar(w)&size(w,2)==4; 

else; switch topo

case '1d' % - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


 rules={ ...
  -2,[xi ones(size(xi,1),1)/max(1,size(xi,1))],'node'
  -1,[.5 0 0 1],'center'
  2, [.5-sqrt(3)/6 0 0 .5; .5+sqrt(3)/6 0 0 .5],'2'; % order 2 integration GEFDYN
    % order 3 integration GEFDYN
  3, [.5-sqrt(3/20) 0 0 5/18
       .5 0 0 8/18
       .5+sqrt(3/20) 0 0 5/18],'3'};
 
    % order 4 integration GEFDYN
   r1=[ -.8611363115941 0 0 .3478548451375
        -.3399810435849 0 0 .6521451548625
         .3399810435849 0 0 .6521451548625
         .8611363115941 0 0 .3478548451375];r1(:,1)=r1(:,1)+1;r1=r1/2;
 rules(end+1,1:3)={4,r1,'4'};

 if ~ischar(w)
 elseif strcmp(w,'def') % defaults
   if size(xi,1)==2; w=3;else; w=4; end
 elseif strcmp(w,'list'); w=rules; return;
 end

case 'q2d' % 2D quadrangle - - - - - - - - - - - - - - - - - - - - - -


FEw=1/sqrt(3);    xg13 = -sqrt(3/5); xg23 = 0. ; xg33 = sqrt(3/5) ;
    wg1 = 5/9; wg2 = 8/9; wg3 = 5/9;
rules ={ ...
  -2,[xi ones(size(xi,1),1)*(4/max(1,size(xi,1)))],'node'; % nodes
  -1, [0 0 0 4],'center';
  % order 2 integration GEFDYN
  102,  [-FEw -FEw 0 1;-FEw FEw 0 1;FEw -FEw 0 1;FEw FEw 0 1],'gefdyn 2x2';
  % order 2 integration standard
  2,  [-FEw -FEw 0 1;FEw -FEw 0 1;FEw FEw 0 1;-FEw FEw 0 1],'standard 2x2';
  % Q4WT standard 2x2 , center, midsides
  109, [-FEw -FEw 0 1;FEw -FEw 0 1;FEw FEw 0 1;-FEw FEw 0 1;0 0 0 4;
       -1 -1 0 1;1 -1 0 0;1 1 0 0;-1 1 0 0],'Q4WT';
  % order 3 integration GEFDYN
  103,  [ xg13 xg13 0 wg1*wg1
          xg13 xg23 0 wg1*wg2
          xg13 xg33 0 wg1*wg3
          xg23 xg13 0 wg2*wg1
          xg23 xg23 0 wg2*wg2
          xg23 xg33 0 wg2*wg3
          xg33 xg13 0 wg3*wg1
          xg33 xg23 0 wg3*wg2
          xg33 xg33 0 wg3*wg3],'gefdyn 3x3';
      % order 4 integration GEFDYN
  104, [  -.8611363115941 -.8611363115941 0 .3478548451375*.3478548451375
         -.8611363115941 -.3399810435849  0 .6521451548625*.3478548451375
         -.8611363115941 .3399810435849  0 .6521451548625*.3478548451375
         -.8611363115941 .8611363115941  0 .3478548451375*.3478548451375

         -.3399810435849 -.8611363115941  0 .3478548451375*.6521451548625
         -.3399810435849 -.3399810435849  0 .6521451548625*.6521451548625
         -.3399810435849 .3399810435849  0 .6521451548625*.6521451548625
         -.3399810435849 .8611363115941  0 .3478548451375*.6521451548625
         
         .3399810435849 -.8611363115941  0 .3478548451375*.6521451548625
         .3399810435849 -.3399810435849  0 .6521451548625*.6521451548625
         .3399810435849 .3399810435849  0 .6521451548625*.6521451548625
         .3399810435849 .8611363115941  0 .3478548451375*.6521451548625
         
         .8611363115941 -.8611363115941 0  .3478548451375*.3478548451375
         .8611363115941 -.3399810435849 0  .6521451548625*.3478548451375
         .8611363115941 .3399810435849 0  .6521451548625*.3478548451375
         .8611363115941 .8611363115941 0  .3478548451375*.3478548451375] ...
,'gefdyn 4x4';};  % --- ordre gefdyn

% 9 point rule plate rule
r9=[-sqrt(3/5) sqrt(3/5)  0 25/81 40/81 64/81];
rules(end+1,1:3)={9, r9([1 1 3 4;2 1 3 4;2 2 3 4;1 2 3 4;
  1 3 3 5;3 1 3 5;2 3 3 5;3 2 3 5;3 3 3 6]),'9 point'};
% 3x3 standard rule
  w2=[sqrt(3/5) 5/9 0 8/9]; % 3x3 integ.
  rules(end+1,1:3)={3, ...
   [-w2(1) -w2(1) 0 w2(2)^2;0 -w2(1) 0 w2(2)*w2(4);w2(1) -w2(1) 0 w2(2)^2;
    -w2(1) 0 0 w2(2)*w2(4);0 0 0 w2(4)^2;w2(1) 0 0 w2(2)*w2(4);
    -w2(1)  w2(1) 0 w2(2)^2;0  w2(1) 0 w2(2)*w2(4);w2(1)  w2(1) 0 w2(2)^2],...
  'standard 3x3'};
% 2x2 standard rule
   r1=1/sqrt(3);
  rules(end+1,1:3)={2, ...
     [ -r1  -r1 0  1; r1  -r1 0  1; r1   r1 0  1; -r1   r1 0  1 ], ...
    'standard 2x2'};
% combine 4 and 9 point rules

  w1=1/sqrt(3);w1=[-w1 -w1 0 1;w1 -w1 0 1;-w1 w1 0 1;w1 w1 0 1]; %2x2 integ.
  w2=[sqrt(3/5) 5/9 0 8/9]; % 3x3 integ.
  rules(end+1,1:3)={13, ...
   [w1;-w2(1) -w2(1) 0 w2(2)^2;0 -w2(1) 0 w2(2)*w2(4);w2(1) -w2(1) 0 w2(2)^2;
     -w2(1) 0 0 w2(2)*w2(4);0 0 0 w2(4)^2;w2(1) 0 0 w2(2)*w2(4);
   -w2(1)  w2(1) 0 w2(2)^2;0  w2(1) 0 w2(2)*w2(4);w2(1)  w2(1) 0 w2(2)^2],...
   '2x2 and 3x3'};
 if ~ischar(w)
 elseif strcmp(w,'def') % defaults
   if size(xi,1)==4; w=2;else; w=9; end
 elseif strcmp(w,'list'); w=rules; return;
 end

case 't2d' % 2D triangle - - - - - - - - - - - - - - - - - - - - - -

  r3=[1/6 2/3 0 1/6];

rules={
  -2,[xi ones(size(xi,1),1)*(.5/max(1,size(xi,1)))],'node'; % nodes
  -1,[1/3 1/3 0 1/2],'center';
   3,r3([1 2 3 4;1 1 3 4;2 1 3 4]),'standard 3 point';
 203,[.5 0 0 1/6;.5 .5 0 1/6;0 .5 0 1/6],'midside';

};
  % 6 points between [-1 1] documentation Code_Aster R30101b
  p1=0.11169079483905; p2=0.0549758718227661;
  a=0.445948490915965; b=0.091576213509771;
  w1=[2*b-1  1-4*b  0  p2;
     2*b-1  2*b-1  0  p2;
     1-4*b  2*b-1  0  p2;
     1-4*a  2*a-1  0  p1;
     2*a-1  1-4*a  0  p1; 
     2*a-1  2*a-1  0  p1 ];
   w1(:,1:2)=(w1(:,1:2)+1)/2;
 rules(end+1,1:3)={6,w1,'6 point aster'};


 if ~ischar(w)
 elseif strcmp(w,'def') % defaults
   if size(xi,1)==3; w=3;else; w=6;end
 elseif strcmp(w,'list'); w=rules; return;
 end

case 't3d' % 3D tetrahedron - - - - - - - - - - - - - - - - - - - - - -

  xg1 = 1/3 ; xg2 = 0.6 ; xg3 = 0.2  ; 
  wg1 = - 0.28125; wg2 = 0.2604166667;
  xG1 = 1/3 ; xG2 = 0.0597158717D0; xG3 = 0.4701420641D0; xG4 = xG3 ;
  xG5 = 0.7974269853D0; xG6 = 0.1012865073D0; xG7 = 0.1012865073D0;
  wG1 = 0.1125D0; wG2 = 0.0661970764D0 ; wG3 = wG2 ; wG4 = wG2 ; wG5 = 0.0629695903D0; wG6 = wG5 ; wG7 = wG5 ;

rules={
  -2,[xi ones(size(xi,1),1)*(1/6/max(1,size(xi,1)))],'node'; % nodes
 -1,[.25 .25 .25 1/6],'center';
 % ordre d'integration 2
 102,[ 1/2 0 1/2 1/6; 0 1/2 1/2 1/6; 1/2 0 1/2 1/6],'gefdyn 2nd order';
  %
  % ordre d'integration 3
  %
 103,[ xg1 xg1 xg1 wg1
         xg2 xg3 xg3 wg2
         xg3 xg2 xg3 wg2
         xg3 xg3 xg2 wg2],'gefdyn order 3';
  %
  % ordre d'integration 4
  %
 104,[xG1 xG1 xG1 wG1
        xG2 xG3 xG3 wG2
        xG3 xG2 xG3 wG2
        xG3 xG3 xG2 wG2
        xG5 xG6 xG6 wG5
        xG6 xG5 xG6 wG5
        xG6 xG6 xG5 wG5],'gefdyn order 4'
 % 5 Point rule. Hughes page 174
  5,[1/4 1/4 1/4   -4/30;
     1/2 1/6 1/6   3/40;
     1/6 1/2 1/6   3/40;
     1/6 1/6 1/2   3/40;
     1/6 1/6 1/6   3/40],'5 point hughes';
};

% Default 4 point rules Code_Aster R30101b
w1=[(5-sqrt(5))/20 (5+3*sqrt(5))/20 1/24];
rules(end+1,1:3)={4,w1([1 1 1 3;1 1 2 3;1 2 1 3;2 1 1 3]),'4 point'};
% Default 15 point rules Code_Aster R30101b
      % This is ordered as in MODULEF

  w1=[(7-sqrt(15))/34 (13+3*sqrt(15))/34 (7+sqrt(15))/34 (13-3*sqrt(15))/34 ...
     (5-sqrt(15))/20 (5+sqrt(15))/20];
  a=(2665+14*sqrt(15))/226800; b=(2665-14*sqrt(15))/226800; c=5/567;

  w1=[  1/4 1/4 1/4         8/405;
       w1([ 1 1 1 ])        a ; %
       w1([ 2 1 1 ])        a ;
       w1([ 1 2 1 ])        a ;
       w1([ 1 1 2 ])        a ;
       w1([ 3 3 3 ])        b ; %
       w1([ 4 3 3 ])        b ;
       w1([ 3 4 3 ])        b ;
       w1([ 3 3 4 ])        b ;
       w1([ 5 6 6 ])        c ; %
       w1([ 5 5 6 ])        c ;
       w1([ 6 5 5 ])        c ;
       w1([ 6 6 5 ])        c ;
       w1([ 6 5 6 ])        c ;
       w1([ 5 6 5 ])        c ];
rules(end+1,1:3)={15,w1,'15 point'};

 if ~ischar(w)
 elseif strcmp(w,'def') % defaults
   if size(xi,1)==4; w=4;else; w=15;end
 elseif strcmp(w,'list'); w=rules; return;
 end

case 'p3d' % 3D pentahedron - - - - - - - - - - - - - - - - - - - - - -

rules={
-2,[xi ones(size(xi,1),1)*(.5/max(1,size(xi,1)))],'node';
-1,[1/3 1/3 .5 1/2],'center';
};
   w1=1/sqrt(3);w1=[-w1 -w1 -w1 1;w1 -w1 -w1 1;-w1 w1 -w1 1;w1 w1 -w1 1;
               -w1 -w1 w1 1;w1 -w1 w1 1;-w1 w1 w1 1;w1 w1 w1 1;];
   w1(:,1:3)=(w1(:,1:3)+1)/2;w1(:,4)=w1(:,4)/8;
rules(end+1,1:3)={15,w1,'2x2x2'};
% Modulef rule (documented in OpenFEM)
  w1=[1/6 4/6 .5-.5/sqrt(3) .5+.5/sqrt(3) 1/12];  
  w1=w1([1 1 3 5;
       2 1 3 5;
       1 2 3 5;
       1 1 4 5;
       2 1 4 5;
       1 2 4 5]);
rules(end+1,1:3)={6,w1,'modulef 6'};

  b=[(9-2*sqrt(15))/21 (9+2*sqrt(15))/21 (6+sqrt(15))/21 (6-sqrt(15))/21  ...
     .5*(1-sqrt(3/5)) .5 .5*(1+sqrt(3/5))     1/3];
  w1=[(155-sqrt(15))/2400 5/18 (155+sqrt(15))/2400 9/80 8/18 ];
 
  w1=[ b([4 4 5]) w1(1)*w1(2) ;
        b([2 4 5]) w1(1)*w1(2) ;
        b([4 2 5]) w1(1)*w1(2) ;
        b([3 1 5]) w1(3)*w1(2) ;
        b([3 3 5]) w1(3)*w1(2) ;
        b([1 3 5]) w1(3)*w1(2) ;
        b([8 8 5]) w1(4)*w1(2) ;
        b([4 4 6]) w1(1)*w1(5) ;
        b([2 4 6]) w1(1)*w1(5) ;
        b([4 2 6]) w1(1)*w1(5) ;
        b([3 1 6]) w1(3)*w1(5) ;
        b([3 3 6]) w1(3)*w1(5) ;
        b([1 3 6]) w1(3)*w1(5) ;
        b([8 8 6]) w1(4)*w1(5) ;
        b([4 4 7]) w1(1)*w1(2) ;
        b([2 4 7]) w1(1)*w1(2) ;
        b([4 2 7]) w1(1)*w1(2) ;
        b([3 1 7]) w1(3)*w1(2) ;
        b([3 3 7]) w1(3)*w1(2) ;
        b([1 3 7]) w1(3)*w1(2) ;
        b([8 8 7]) w1(4)*w1(2) ];

rules(end+1,1:3)={21,w1,'modulef 21'};

 if ~ischar(w)
 elseif strcmp(w,'def') % defaults
   if size(xi,1)==6; w=6;else; w=21;end
 elseif strcmp(w,'list'); w=rules; return;
 end

case 'h3d' % 3D hexahedron - - - - - - - - - - - - - - - - - - - - - -

w1=1/sqrt(3);
    % order 3 integration
    xg13 = -.7745966692415;     xg23 = 0.;     xg33 = .7745966692415;
    wg1= 5/9; wg2=8/9; wg3=5/9;

    xG14 = -.8611363115941; xG24 = -.3399810435849; xG34 = .3399810435849; xG44 = .8611363115941;
    wG1 = .3478548451375 ; wG2 = .6521451548625; wG3 = .6521451548625; wG4 = .3478548451375 ;

rules={
-2,[xi ones(size(xi,1),1)*(8/max(1,size(xi,1)))],'node';
-1,[0 0 0 8],'center';
% GEFDYN
%     %  des tests a ajouter !!!
    % order 2 integration
 102,[-w1 -w1 -w1 1;-w1 -w1 w1 1;-w1 w1 -w1 1;-w1 w1 w1 1;
               w1 -w1 -w1 1;w1 -w1 w1 1;w1 w1 -w1 1;w1 w1 w1 1;],'gefdyn 2';
 103, [xg13 xg13 xg13 wg1*wg1*wg1
         xg13 xg13 xg23 wg1*wg1*wg2
         xg13 xg13 xg33 wg1*wg1*wg3
         xg13 xg23 xg13 wg1*wg2*wg1
         xg13 xg23 xg23 wg1*wg2*wg2
         xg13 xg23 xg33 wg1*wg2*wg3
         xg13 xg33 xg13 wg1*wg3*wg1
         xg13 xg33 xg23 wg1*wg3*wg2
         xg13 xg33 xg33 wg1*wg3*wg3
         
         xg23 xg13 xg13 wg2*wg1*wg1
         xg23 xg13 xg23 wg2*wg1*wg2
         xg23 xg13 xg33 wg2*wg1*wg3
         xg23 xg23 xg13 wg2*wg2*wg1
         xg23 xg23 xg23 wg2*wg2*wg2
         xg23 xg23 xg33 wg2*wg2*wg3
         xg23 xg33 xg13 wg2*wg3*wg1
         xg23 xg33 xg23 wg2*wg3*wg2
         xg23 xg33 xg33 wg2*wg3*wg3
         
         xg33 xg13 xg13 wg3*wg1*wg1
         xg33 xg13 xg23 wg3*wg1*wg2
         xg33 xg13 xg33 wg3*wg1*wg3
         xg33 xg23 xg13 wg3*wg2*wg1
         xg33 xg23 xg23 wg3*wg2*wg2
         xg33 xg23 xg33 wg3*wg2*wg3
         xg33 xg33 xg13 wg3*wg3*wg1
         xg33 xg33 xg23 wg3*wg3*wg2
         xg33 xg33 xg33 wg3*wg3*wg3],'gefdyn 3';
    % order 4 integration
  104, [   xG14 xG14 xG14 wG1*wG1*wG1;
           xG14 xG14 xG24 wG1*wG1*wG2
           xG14 xG14 xG34 wG1*wG1*wG3
           xG14 xG14 xG44 wG1*wG1*wG4
           xG14 xG24 xG14 wG1*wG2*wG1
           xG14 xG24 xG24 wG1*wG2*wG2
           xG14 xG24 xG34 wG1*wG2*wG3
           xG14 xG24 xG44 wG1*wG2*wG4
           xG14 xG34 xG14 wG1*wG3*wG1
           xG14 xG34 xG24 wG1*wG3*wG2
           xG14 xG34 xG34 wG1*wG3*wG3
           xG14 xG34 xG44 wG1*wG3*wG4
           xG14 xG44 xG14 wG1*wG4*wG1
           xG14 xG44 xG24 wG1*wG4*wG2
           xG14 xG44 xG34 wG1*wG4*wG3
           xG14 xG44 xG44 wG1*wG4*wG4
           xG24 xG14 xG14 wG2*wG1*wG1
           xG24 xG14 xG24 wG2*wG1*wG2
           xG24 xG14 xG34 wG2*wG1*wG3
           xG24 xG14 xG44 wG2*wG1*wG4
           xG24 xG24 xG14 wG2*wG2*wG1
           xG24 xG24 xG24 wG2*wG2*wG2
           xG24 xG24 xG34 wG2*wG2*wG3
           xG24 xG24 xG44 wG2*wG2*wG4
           xG24 xG34 xG14 wG2*wG3*wG1
           xG24 xG34 xG24 wG2*wG3*wG2
           xG24 xG34 xG34 wG2*wG3*wG3
           xG24 xG34 xG44 wG2*wG3*wG4
           xG24 xG44 xG14 wG2*wG4*wG1
           xG24 xG44 xG24 wG2*wG4*wG2
           xG24 xG44 xG34 wG2*wG4*wG3
           xG24 xG44 xG44 wG2*wG4*wG4
           xG34 xG14 xG14 wG3*wG1*wG1
           xG34 xG14 xG24 wG3*wG1*wG2
           xG34 xG14 xG34 wG3*wG1*wG3
           xG34 xG14 xG44 wG3*wG1*wG4
           xG34 xG24 xG14 wG3*wG2*wG1
           xG34 xG24 xG24 wG3*wG2*wG2
           xG34 xG24 xG34 wG3*wG2*wG3
           xG34 xG24 xG44 wG3*wG2*wG4
           xG34 xG34 xG14 wG3*wG3*wG1
           xG34 xG34 xG24 wG3*wG3*wG2
           xG34 xG34 xG34 wG3*wG3*wG3
           xG34 xG34 xG44 wG3*wG3*wG4
           xG34 xG44 xG14 wG3*wG4*wG1
           xG34 xG44 xG24 wG3*wG4*wG2
           xG34 xG44 xG34 wG3*wG4*wG3
           xG34 xG44 xG44 wG3*wG4*wG4
           xG44 xG14 xG14 wG4*wG1*wG1
           xG44 xG14 xG24 wG4*wG1*wG2
           xG44 xG14 xG34 wG4*wG1*wG3
           xG44 xG14 xG44 wG4*wG1*wG4
           xG44 xG24 xG14 wG4*wG2*wG1
           xG44 xG24 xG24 wG4*wG2*wG2
           xG44 xG24 xG34 wG4*wG2*wG3
           xG44 xG24 xG44 wG4*wG2*wG4
           xG44 xG34 xG14 wG4*wG3*wG1
           xG44 xG34 xG24 wG4*wG3*wG2
           xG44 xG34 xG34 wG4*wG3*wG3
           xG44 xG34 xG44 wG4*wG3*wG4
           xG44 xG44 xG14 wG4*wG4*wG1
           xG44 xG44 xG24 wG4*wG4*wG2
           xG44 xG44 xG34 wG4*wG4*wG3
           xG44 xG44 xG44 wG4*wG4*wG4],'gefdyn 4';
};

   w1=[5/18  8/18  5/18];
   b=[(1-sqrt(3/5))/2 .5 (1+sqrt(3/5))/2 (1-sqrt(3/5))/2];
   w1=[ b([1 1 1]) w1(1)*w1(1)*w1(1) ;
         b([1 1 2]) w1(1)*w1(1)*w1(2) ;
         b([1 1 3]) w1(1)*w1(1)*w1(3) ;
         b([1 2 1]) w1(1)*w1(2)*w1(1) ;
         b([1 2 2]) w1(1)*w1(2)*w1(2) ;
         b([1 2 3]) w1(1)*w1(2)*w1(3) ;
         b([1 3 1]) w1(1)*w1(3)*w1(1) ;
         b([1 3 2]) w1(1)*w1(3)*w1(2) ;
         b([1 3 3]) w1(1)*w1(3)*w1(3) ; 
         b([2 1 1]) w1(2)*w1(1)*w1(1) ;
         b([2 1 2]) w1(2)*w1(1)*w1(2) ;
         b([2 1 3]) w1(2)*w1(1)*w1(3) ;
         b([2 2 1]) w1(2)*w1(2)*w1(1) ;
         b([2 2 2]) w1(2)*w1(2)*w1(2) ;
         b([2 2 3]) w1(2)*w1(2)*w1(3) ;
         b([2 3 1]) w1(2)*w1(3)*w1(1) ;
         b([2 3 2]) w1(2)*w1(3)*w1(2) ;
         b([2 3 3]) w1(2)*w1(3)*w1(3) ;
         b([3 1 1]) w1(3)*w1(1)*w1(1) ;
         b([3 1 2]) w1(3)*w1(1)*w1(2) ;
         b([3 1 3]) w1(3)*w1(1)*w1(3) ;
         b([3 2 1]) w1(3)*w1(2)*w1(1) ;
         b([3 2 2]) w1(3)*w1(2)*w1(2) ;
         b([3 2 3]) w1(3)*w1(2)*w1(3) ;
         b([3 3 1]) w1(3)*w1(3)*w1(1) ;
         b([3 3 2]) w1(3)*w1(3)*w1(2) ;
         b([3 3 3]) w1(3)*w1(3)*w1(3) ;
       ];
   w1(:,1:3)=2*w1(:,1:3)-1;w1(:,4)=w1(:,4)*8;
 rules(end+1,1:3)={3,w1,'3x3x3'};
   w1=1/sqrt(3);w1=[-w1 -w1 -w1 1;w1 -w1 -w1 1;-w1 w1 -w1 1;w1 w1 -w1 1;
               -w1 -w1 w1 1;w1 -w1 w1 1;-w1 w1 w1 1;w1 w1 w1 1;];
 rules(end+1,1:3)={2,w1,'2x2x2'};
 if ~ischar(w)
 elseif strcmp(w,'def') % defaults
   if size(xi,1)==8; w=2; else; w=3;end
 elseif strcmp(w,'list'); w=rules; return;
 end

otherwise error('Not supported');
end


 if size(w,2)==4  % standard rule selection
 elseif length(w)==1; 
   for j1=1:size(rules,1)
    if isequal(w,rules{j1,1}); w=rules{j1,2};break;end
   end
   if length(w)==1;w=rules{end,2};end
 else
   error('not a valid case');
 end % standard rule selection


end; % w provided or not



% ---------------------------------------------------------------------
% Writes the whole rule to screen OBSOLETE
function writerule(out)

   fprintf('\n%%Written by integrules(''flui6'')\n\n');
   w=out.w;writetoscreen(w)
   N=out.N;writetoscreen(N)
   Nr=out.Nr;writetoscreen(Nr)
   Ns=out.Ns;writetoscreen(Ns)
   Nt=out.Nt;writetoscreen(Nt)

% ---------------------------------------------------------------------
function writetoscreen(in)


fprintf('%s = [ ',inputname(1));
for j1=1:size(in,1)
 i0=0;
 for j2=1:size(in,2);
  st1=sprintf('%20.16e ',in(j1,j2)); i0=i0+1;
  st1=strrep(st1,'e-00','e-'); st1=strrep(st1,'e+00','e+');
  st1=strrep(st1,'00000000000000e','e');
  fprintf(st1);
  if i0==3&j2<size(in,2); fprintf(', ...\n'); 
      i0=0; end
 end
 fprintf(';\n');
end
 fprintf('];%%%s\n',inputname(1));



