// Script Scilab d'import de données texte et d'export en WAV
// But : "écouter" une température
// Suite des opérations effectuées :
// 1 - Chargement des données texte
// source : http://www.meteo-paris.com/ile-de-france/station-meteo-paris/pro/)
// 2 - Bouclage des données sur 11 périodes (de manière à avoir un nombre entier d'échantillons sur 11 périodes pour un La3 440Hz @ Fs=48 kHz)
// 3 - Rééchantillonnage, les 11 périodes doivent représenter 1200 échantillons
// 4 - Génération d'une enveloppe paramétrée
// 5 - Bouclage des 11 périodes précédentes sur la durée de l'enveloppe + conditionnement du signal
// Remarque : le conditionnement se fait par un filtre passe-bande, éliminant à la fois la composante continue et les sauts haute fréquence entre motifs
// 6 - Application de l'enveloppe sur le pattern
// 7 - Exportation en WAV 16bits / 48kHz

  clear all ;
  clc ;
  
// 0 - Définition des données utiles
  Fs = 48000 ;                                                                  // Définition de la fréquence d'échantillonnage
  At =    10 ;                                                                  // Enveloppe : durée de l'attaque, en ms
  An =   100 ;                                                                  // Enveloppe : niveau final de l'attaque, en %
  Dt =     5 ;                                                                  // Enveloppe : durée de l'atténuation, en ms
  Dn =    80 ;                                                                  // Enveloppe : niveau final de l'atténuation, en %
  S  =   100 ;                                                                  // Enveloppe : durée du maintien, en ms
  R  =  1000 ;                                                                  // Enveloppe : durée du relachement, en ms
  duree  = (At + Dt + S + R) / 1000 ;                                           // Durée du signal, en s.
  taille = duree*Fs                 ;                                           // Nombre d'échantillons dans le signal

// 1 - Chargement des données texte
  cd 'd:\perso\Futura\xx - Ecouter une température\Scilab\Donnees'              // Définition du répertoire local où se trouvent les données
  Donnees  = fscanfMat('14.01.2012_2j.txt')      ;                              // Récupération des seules données numériques dans le fichier météo
  longueur = size(Donnees,1)                     ;                              // Nombre de points de mesure dans les relevés météo
  cd 'd:\perso\Futura\xx - Ecouter une température\Scilab'                      // On se remet un cran en amont
  motif    = Donnees(longueur-287+1:longueur,3)' ;                              // Extraction des 24 dernières heures

// 2 - Bouclage des données sur 11 périodes
  sequence = [motif motif motif motif motif motif motif motif motif motif motif] ;

// 3 - Rééchantillonnage
  longueur = size(sequence,2) ;                                                 // Nombre d'échantillons de "sequence", à rééchantillonner sur 1200 échantillons
  x        = [1:longueur]     ;                                                 // Définition des abscisses pour la construction d'une fonction spline
  f        = sequence         ;                                                 // Définition des ordonnées pour la construction d'une fonction spline
  d        = splin(x,f)       ;                                                 // Construction d'une fonction spline égale à toute la séquence originale
  sequence = interp([1:longueur/1200:longueur],x,f,d) ;                         // Echantillonnage de la fonction spline sur 1200 échantillons
  
// 4 - Génération d'une enveloppe paramétrée
  enveloppe = zeros(1,duree*Fs) ;                                               // Création d'un vecteur vide pour l'enveloppe
  // Attaque
  debut                = 1                    ;
  longueur             = Fs * At / 1000       ;
  fin                  = debut + longueur - 1 ;
  pas                  = An/100/longueur      ;
  enveloppe(debut:fin) = pas:pas:longueur*pas ;
  // Atténuation
  debut                = fin + 1                                ;
  longueur             = Fs * Dt / 1000                         ;
  fin                  = debut + longueur - 1                   ;
  pas                  = (Dn-An)/100/longueur                   ;
  enveloppe(debut:fin) = (An/100)+pas:pas:(An/100)+longueur*pas ;
  // Maintien
  debut                = fin + 1                                ;
  longueur             = Fs * S / 1000                          ;
  fin                  = debut + longueur - 1                   ;
  enveloppe(debut:fin) = enveloppe(debut-1)                     ;
  // Relachement
  debut                = fin + 1                                ;
  longueur             = Fs * R / 1000                          ;
  fin                  = debut + longueur - 1                   ; 
  pas                  = enveloppe(debut-1) / longueur          ;
  enveloppe(debut:fin) = enveloppe(debut-1)-pas:-pas:0          ;
  
// 5 - Bouclage des 11 périodes de la séquence sur la durée de l'enveloppe
  signal             = zeros(1,taille)                  ;
  indice             = 0                                ;
  while indice < taille
      indice         = indice + 1                       ;
      signal(indice) = sequence(pmodulo(indice,1200)+1) ;
  end
  
  hz     = iir(4,'bp','butt',[220/Fs 10000/Fs],[1 1]) ;                         // Générations des poles et des zeros du filtre récursif
  sl     = tf2ss(hz)                                  ;                         // Génération des coefficients linéaires du filtre à partir de hz
  signal = flts(signal,sl)                            ;                         // Filtrage
  
// 6 - Application de l'enveloppe sur le pattern et tracé
  signal = signal .* enveloppe       ;                                          // Multiplication de 2 vecteurs point par point
  signal = signal / max(abs(signal)) ;                                          // Normisation entre -1 et +1, nécessaire pour la fonction ""savewave"
  plot(signal) ;                                                                // Tracé
  
// 7 - Exportation en WAV 16bits / 48kHz  
  savewave('La3.wav',signal,Fs) ;