% Rsolutions temporelle et frquentielle du comportement d'un systme masse-ressort
% Supports
% http://www.mit.edu/~tokmakof/TDQMS/Notes/8._Linear_Response_2-09.pdf
% http://fr.wikipedia.org/wiki/Syst%C3%A8me_oscillant_%C3%A0_un_degr%C3%A9_de_libert%C3%A9
% But : constater l'quivalence des deux mthodes, par TF et TF-1
% Conventions :
% f(t) : fonction temporelle
% F(w) : fonction frquentielle associe
% Equation diffrentielle :
% m.dx/dt + b.dx/dt + k.x = 0
% Rgime forc : 
% dm.dx/dt + b.dx/dt + k.x = f(t)


clear all ;
clc ;

% Dfinition des constantes
  Fech    = 10000         ;  % Frquence d'chantillonnage
  pas     = 1/Fech        ;  % Pas de temps
  Nyquist = round(Fech/2) ;  % Quid d'une Fech impaire ?
  Nfft    = Fech          ;  % => DeltaF = 1 Hz
  
% Construction des vecteurs de base
  duree = 3                         ; % [s]
  t     = [0:pas:duree-pas]         ;
  f     = [0:Fech/(duree*Nfft):Nyquist]    ;
  w     = 2*pi*f                    ;
  f_tot = [0:Fech/(duree*Nfft):Fech-Fech/(duree*Nfft)]    ;
  w_tot = 2*pi*f_tot ;
  
% Dfinition des paramtres du problme
  k    = 3e3        ;  % [N/m]
  m    = 0.3        ;  % [kg]
  w0   = sqrt(k/m)  ;  % [rad/s]
  f0   = w0 /(2*pi) ;  % [Hz]
  b    = 10         ;  % [kg/s]
  beta = b/(2*w0*m) ;
  phi = -pi/2       ;  % condition initiale des oscillations libres : position de repos
  
% Dfintion de l'excitation : 1s nulle, 1s  force constante, 1s nulle
  xc(       1 :   Fech) = 0 ;
  xc(  Fech+1 : 2*Fech) = k ; % (k/4) * sin(2*pi*45*t(Fech+1:2*Fech)) ;
  xc(2*Fech+1 : duree*Fech) = 0 ;


%--------------------------------------------------------------------------------------%
%                                                                                      %
%                        ETAPE 1 : Solution analytique (A FAIRE)                       %
%                                                                                      %
%--------------------------------------------------------------------------------------%

% Solution analytique (oscillations libres !!!)
  % A = 1 ;
  % x = A * exp(-beta*w0*t) .* cos(sqrt(1-beta^2)*w0*t+phi) ;


%--------------------------------------------------------------------------------------%
%                                                                                      %
%                       ETAPE 2 : Rsolution temporelle pas  pas                      %
%                                                                                      %
%--------------------------------------------------------------------------------------%

% Schma utilis (simpliste, mais a fonctionne...) :
% v(n) = dx/dt(n)   = (x(n) - x(n-1)) / pas
% a(n) = dx/dt(n) = (v(n) - v(n-1)) / pas
% x(n) = 1/alpha * [xc(n) + x(n-1).beta + x(n-2).gamma]
% alpha =  m/pas + b/pas + k
% beta  = 2m/pas + b/pas
% gamma = -m/pas
  
% Rsolution en temporel
  alpha =   (m/pas^2) + (b/pas) + k ;
  beta  = (2*m/pas^2) + (b/pas)     ;
  gamma =   -m/pas^2                ;

  x_t = zeros(size(xc,2),1)' ;
  
  for i = 3:duree*Fech
      x_t(i) = (1/alpha) * (xc(i) + beta * x_t(i-1) + gamma * x_t(i-2)) ;
  end


%--------------------------------------------------------------------------------------%
%                                                                                      %
%                          ETAPE 3 : Rsolution en frquentiel                         %
%                                                                                      %
%--------------------------------------------------------------------------------------%

% Rsolution en frquentiel
  % m.dx/dt + b.dx/dt + k.x = 0
  % X/F(w) = 1/(-w^2.m + jb.w + k)
  FT      = 1./(-w.^2*m + j*b*w + k) ;
  FT_norm = k * FT ;
  % figure(2) ;
  % semilogx(w,20*log10(abs(FT))) ;
  % hold on ;
  % semilogx(w,20*log10(abs(temp(1:1+Nfft/2)/(Nfft/2))),'r') ;
  % axis([1 Nyquist -60 40]) ;
  % hold on ;
  
% Construction d'une fonction de transfert inversible
  Taille = size(FT,2) ;
  FT_inv = [FT conj(FT(Taille-1:-1:2))] ;
  
% Multiplication en frquentiel du spectre de l'excitation et de la fonction de transfert
  x_f     = FT_inv .* fft(xc)  ;

% Inversion
  x_f_inv = real(ifft(x_f)) ;


%--------------------------------------------------------------------------------------%
%                                                                                      %
%                                 ETAPE 4 : Convolution                                %
%                                                                                      %
%--------------------------------------------------------------------------------------%

% Rponse impulsionnelle du systme
  RI = real(ifft(FT_inv)) ;
  
% Convolution
  x_conv = filter(RI,1,xc) ;

%--------------------------------------------------------------------------------------%
%                                                                                      %
%                                  ETAPE 5 : Energies                                  %
%                                                                                      %
%--------------------------------------------------------------------------------------%

% Calcul du vecteur vitesses, shma un peu plus labor
  v = zeros(size(xc,2),1)' ;
  for i = 2:duree*Fech-1
      v(i) = (x_conv(i+1) - x_conv(i-1)) / (2*pas) ;
  end
  
% Energies cintique et potentielle
  NRJ_cin = 0.5 * m * v.^2 ;
  NRJ_pot = 0.5 * k * x_conv.^2 ;


%--------------------------------------------------------------------------------------%
%                                                                                      %
%                         ETAPE 6 : Relations de Kramers-Kronig                        %
%                                                                                      %
%--------------------------------------------------------------------------------------%

% Extraction des partie relle et imaginaire de FT_inv
  FT_inv_Re = real(FT_inv) ;
  FT_inv_Im = imag(FT_inv) ;
  

  
% Calcul des parties relle et imaginaire  partir des relations de KK
  Taille = size(FT_inv,2) ;
  FT_inv_Re_KK = zeros(1,Taille) ;
  FT_inv_Im_KK = zeros(1,Taille) ;
  
  % for i = 2 : Taille-1
  %     FT_inv_Re_KK(i) =  (2/pi) * ( sum(((w_tot(1:i-1).*FT_inv_Im(1:i-1))./(w_tot(1:i-1).^2-w_tot(i)^2))*(w_tot(2)-w_tot(1)) + sum(((w_tot(i+1:Taille).*FT_inv_Im(i+1:Taille))./(w_tot(i+1:Taille).^2-w_tot(i)^2))*(w(2)-w(1))   ) )) ;
  %     FT_inv_Im_KK(i) = -(2*w_tot(i)/pi) * ( sum((FT_inv_Re(1:i-1)./(w_tot(1:i-1).^2-w_tot(i)^2))*(w_tot(2)-w_tot(1)) + sum((FT_inv_Re(i+1:Taille)./(w_tot(i+1:Taille).^2-w_tot(i)^2))*(w(2)-w(1))   ) )) ;
  % end  
  FT_inv_Re_KK = imag(Hilbert(FT_inv_Im)) ;
  FT_inv_Im_KK = -imag(Hilbert(FT_inv_Re)) ;


%--------------------------------------------------------------------------------------%
%                                                                                      %
%                        ETAPE ? : Retard de groupe et de phase                        %
%                                                                                      %
%--------------------------------------------------------------------------------------%


%--------------------------------------------------------------------------------------%
%                                                                                      %
%                            ETAPE ? : Fourier et temporalit                          %
%                                                                                      %
%--------------------------------------------------------------------------------------%

% Mise en vidence de la reprsentation en Fourier de l'mergence d'un sinus + Heaviside


%--------------------------------------------------------------------------------------%
%                                                                                      %
%                      ETAPE ? : Construction d'une FT non causale                     %
%                                                                                      %
%--------------------------------------------------------------------------------------%


%--------------------------------------------------------------------------------------%
%                                                                                      %
%                                        GRAPHES                                       %
%                                                                                      %
%--------------------------------------------------------------------------------------%

% Rsolutions temporelle et frquentielle : excitation, FT, rsolution temporelle et
% rsolution frquentielle sur la meme page
  figure(1) ;
  subplot(2,2,1) ;
  plot(t,xc,'r','LineWidth',2) ;
  axis([0 duree 0 1.5*k]) ;
  xlabel('Temps [s]') ;
  ylabel('Force [N]') ;
  legend('Excitation') ;
  title('Force d''excitation du systme','FontSize',12,'FontWeight','Bold') ;
  subplot(2,2,2) ;
  semilogx(w(1:1+Nfft/2)/(2*pi),20*log10(abs(FT_norm(1:1+Nfft/2))),'k','LineWidth',2) ;
  hold on ;
  semilogx([f0 f0],[-80 20],'b:') ;
  hold off ;
  axis([1 Nyquist -80 20]) ;
  legend('Fonction de transfert normalise','Frquence propre') ;
  xlabel('Frquence [Hz]') ;
  ylabel('Gain normalis [dB]') ;
  title('Fonction de transfert','FontSize',12,'FontWeight','Bold') ;
  subplot(2,2,3) ;
  plot(t,x_t) ;
  axis([0 duree -1 2]) ;
  xlabel('Temps [s]') ;
  ylabel('Dplacement [m]') ;
  legend('x(t)') ;
  title('Rponse du systme : x(t)','FontSize',12,'FontWeight','Bold') ;
  subplot(2,2,4) ;
  plot(t,x_f_inv) ;
  axis([0 duree -1 2]) ;
  xlabel('Temps [s]') ;
  ylabel('Dplacement [m]') ;
  legend('FT-1(x(w))') ;
  title('Rponse du systme : FT-1(x(w))','FontSize',12,'FontWeight','Bold') ;
  
% Convolution : excitation, rponse impulsionnelle, rsolution temporelle et convolution
% sur la meme page
  figure(1) ;
  subplot(2,2,1) ;
  plot(t,xc,'r','LineWidth',2) ;
  axis([0 duree 0 1.5*k]) ;
  xlabel('Temps [s]') ;
  ylabel('Force [N]') ;
  legend('Excitation') ;
  title('Force d''excitation du systme','FontSize',12,'FontWeight','Bold') ;
  subplot(2,2,2) ;
  plot(t,RI,'r','LineWidth',1) ;
  hold on ;
  hold off ;
  axis([0 3 -5e-6 5e-6]) ;
  legend('RI') ;
  xlabel('Frquence [Hz]') ;
  ylabel('Gain normalis [dB]') ;
  title('Rponse impulsionnelle','FontSize',12,'FontWeight','Bold') ;
  subplot(2,2,3) ;
  plot(t,x_t) ;
  axis([0 duree -1 2]) ;
  xlabel('Temps [s]') ;
  ylabel('Dplacement [m]') ;
  legend('x(t)') ;
  title('Rponse du systme : x(t)','FontSize',12,'FontWeight','Bold') ;
  subplot(2,2,4) ;
  plot(t,x_f_inv) ;
  axis([0 duree -1 2]) ;
  xlabel('Temps [s]') ;
  ylabel('Dplacement [m]') ;
  legend('Convolution') ;
  title('Rponse du systme : x(t) par convolution','FontSize',12,'FontWeight','Bold') ;
  
% Energies
  figure(3) ;
  subplot(2,2,1) ;
  plot(t,xc,'r','LineWidth',2) ;
  axis([0 duree 0 1.5*k]) ;
  xlabel('Temps [s]') ;
  ylabel('Force [N]') ;
  legend('Excitation') ;
  title('Force d''excitation du systme','FontSize',12,'FontWeight','Bold') ;
  subplot(2,2,2) ;
  plot(t,NRJ_cin) ;
  axis([0 duree 0 1000]) ;
  xlabel('Temps [s]') ;
  ylabel('Energie [J]') ;
  legend('NRJ cintique') ;
  title('Energie cintique du systme','FontSize',12,'FontWeight','Bold') ;
  subplot(2,2,3) ;
  plot(t,x_conv,'b','LineWidth',1) ;
  hold on ;
  hold off ;
  axis([0 3 -1 2]) ;
  legend('RI') ;
  xlabel('Frquence [Hz]') ;
  ylabel('Gain normalis [dB]') ;
  title('Rponse du systme : x(t) par convolution','FontSize',12,'FontWeight','Bold') ;
  subplot(2,2,4) ;
  plot(t,NRJ_pot) ;
  axis([0 duree 0 4000]) ;
  xlabel('Temps [s]') ;
  ylabel('Energie [J]') ;
  legend('NRJ potentielle') ;
  title('Energie potentielle lastique du systme','FontSize',12,'FontWeight','Bold') ;
  
% Relations de KK
  figure(4) ;
  subplot(2,2,1) ;
  semilogx(f_tot,FT_inv_Re) ;
  xlabel('Frquence [Hz]') ;
  ylabel('Partie relle') ;
  legend('Re(FT)') ;
  title('Partie relle extraite de la FT','FontSize',12,'FontWeight','Bold') ;
  subplot(2,2,2) ;
  semilogx(f_tot,FT_inv_Im) ;
  xlabel('Frquence [Hz]') ;
  ylabel('Partie imaginaire') ;
  legend('Im(FT)') ;
  title('Partie imaginaire extraite de la FT','FontSize',12,'FontWeight','Bold') ;
  subplot(2,2,3) ;
  semilogx(f_tot,FT_inv_Re_KK) ;
  xlabel('Frquence [Hz]') ;
  ylabel('Partie relle') ;
  legend('H(Im(FT))') ;
  title('Partie relle calcule  partir de la partie imaginaire','FontSize',12,'FontWeight','Bold') ;
  subplot(2,2,4) ;
  semilogx(f_tot,FT_inv_Im_KK) ;
  xlabel('Frquence [Hz]') ;
  ylabel('Partie imaginaire') ;
  legend('-H(Re(FT))') ;
  title('Partie imaginaire calcule  partir de la partie relle','FontSize',12,'FontWeight','Bold') ;
  