# -*- coding: utf-8 -*-

import numpy as np
import scipy.spatial
import scipy.sparse as spar

def rect_tambour(L,H,U):
    """
    Paramètres
    ----------
    L : Largeur du tambour
    
    H : Hauteur du tambour
    
    U : Valeur propre maximale voulue

    Retourne
    -------
    Modes propres de valeur propre supérieure à -U.
    """
    vals = []
    val = 0
    k = 1
    l = 1
    while val >= -U:
        while val >= -U:
            #print(val,l,k)
            vals.append(val)
            val = -np.pi**2*((k/L)**2+(l/H)**2)
            l += 1
        l = 2
        k += 1
        val = -np.pi**2*((k/L)**2+(l/H)**2)
    return -np.sort(-np.array(vals)[1:])

def Hermann_Weyl(U,A,P):
    return A*U/(4*np.pi)-P*np.sqrt(U)/(4*np.pi)

def count_vp(tab,U):
    #compte les n valeurs propres upérieures au égales à -U dans le tableau tab
    return tab[tab>=-U]

def in_curve(f,fargs,shape,a):
    """
    Permet de trouver les point situés à l'intérieur de la coube délimitée par l'équation f(x,y,*fargs)=0
    
    Paramètres.
    ----------
    f : fonction de type f(x,y) définissant l'intérieur de la courbe par f(x,y,fargs)<=0
    Exemple, cercle de rayon R : f(x,y,R) = x**2+y**2-R
    
    fargs : Autre arguments de f. Exemple, cercle de rayon R : fargs=[R]
    
    shape : dimensions du tableau représentant l'espace où la coube est calculée.
    
    a : distance représentée par chaque case.
    
    eps : précision pour la détermination du bord

    Retourne
    -------
    Tableau de points.
    
    """
    points = [] # les points à l'intérieur de la courbe
    for j in range(shape[0]):
        for i in range(shape[1]):
            if f(i*a,j*a,*fargs)<0:
                points.append([i*a,j*a])

    return np.array(points)

def in_curve2(points,f,fargs,a):
    in_points = []
    for p in points:
        if f(p[0],p[1],*fargs) < 0:
            in_points.append(p)
    return np.array(in_points)
              

def triang(points,a,f,fargs,bord):
    """
    Transforme un réseau carré en un réseau triangulaire.
    
    Paramètres
    ----------
    points : tableau de points.
    a : côté des carré.
    f : intérieur de la courbe.
    fargs : Autre arguments de f.
    bord : bord de la courbe.

    Retourne
    -------
    tableau de points.

    """
    tri_points = points.copy()
    tri_points[:,1] *= np.sqrt(3)
    tri_points2 = np.vstack((points,bord))
    tri_points2[:,1] *= np.sqrt(3)
    tri_points2[:,0] += a/2
    tri_points2[:,1] += np.sqrt(3)/2*a
    fin = np.vstack((tri_points,tri_points2))
    i = 0
    eps = 0.01
    while i < len(fin):
        if f(fin[i,0]+eps,fin[i,1]+eps,*fargs) > 0:
            fin = np.delete(fin,i,0)
            i -= 1
        i += 1
    return np.vstack((fin,bord)),len(fin),len(bord)

def surf(points,tri):
    return np.cross(points[tri.simplices[:,1]]-points[tri.simplices[:,0]],points[tri.simplices[:,2]]-points[tri.simplices[:,0]])/2


def tri_ang(points,ind,p0):
    # trie les points dans l'ordre trigo
    vec=np.arctan2((points-p0)[:,1],(points-p0)[:,0]) # les angles de chaque point
    values = []
    dtype = [('val',float),('n',int)]
    # on associe à chaque valeur un entier afin de retrouver à quel point ils appartiennent après le tri
    for i in range(len(vec)):
        values.append((vec[i],i))
    values = np.sort(np.array(values,dtype),order='val') # on trie les angles
    new_points = []
    new_ind = []
    # on utilise les entier afin de trier la liste de points et les indices :
    for tup in values:
        new_points.append(points[tup[1]])
        new_ind.append(ind[tup[1]])
    return np.array(new_points),new_ind





def M(points,tri,Nint,dl):
    
    indptr,ind = tri.vertex_neighbor_vertices
    N = len(points)
    M = np.zeros((N,N))
    
    for i in range(Nint):
        tot = 0
        nhb_ind = ind[indptr[i]:indptr[i+1]] # indices des points voisin du point d'indice k
        nhb = points[nhb_ind] # leurs coordonnées
        nhb,nhb_ind = tri_ang(nhb,nhb_ind,points[i])
        # tiers de la somme des aires : 
        A = 0 # pour les aires
        for j in range(len(nhb_ind)):
            vec = points[nhb_ind[j]]-points[i] # un veteur reliant le point à un voisin
            vec_av = points[nhb_ind[j-1]]-points[i] # un autre vecteur mais avec le vosin d'avant
            # on utilise le produit vectoriel pour calculer les aires des triangles : ||A^B||/2
            A += np.linalg.norm(np.cross(vec,vec_av))/2
        #print(A)
        A /= 3
        
        for j in range(len(nhb_ind)):
            if nhb_ind[j] < Nint:
                vec = points[nhb_ind[j]]-points[i] # un veteur reliant le point à son voisin d'indice 0
                vec_av = points[nhb_ind[j-1]]-points[i] # un autre vecteur mais avec le vosin d'avant
                if j+1 >= len(nhb_ind):
                    k = 0
                else:
                    k = j+1
                vec_ap = points[nhb_ind[k]]-points[i] # un autre vecteur mais avec le voisin d'après
                
                # on utilise le produit vectoriel pour calculer les sinus : ||A^B||/(||A|| ||B||)
                sin_alpha = np.linalg.norm(np.cross(vec_av,vec_av-vec))/(np.linalg.norm(vec_av)*np.linalg.norm(vec_av-vec))
                sin_beta = np.linalg.norm(np.cross(vec_ap-vec,vec_ap))/(np.linalg.norm(vec_ap)*np.linalg.norm(vec_ap-vec))
                
                # on utilise le produit scalaire pour calculer les cosinus : ||A.B||/(||A|| ||B||)
                cos_alpha = np.dot(vec_av,vec_av-vec)/(np.linalg.norm(vec_av)*np.linalg.norm(vec_av-vec))
                cos_beta = np.dot(vec_ap-vec,vec_ap)/(np.linalg.norm(vec_ap)*np.linalg.norm(vec_ap-vec))
                
                cotan_alpha = np.abs(cos_alpha/sin_alpha)
                cotan_beta = np.abs(cos_beta/sin_beta)
                #print(cotan_alpha,cotan_beta)
                #print(nhb,vec_av,vec_ap,vec,points[i])
                M[i,nhb_ind[j]] = -1/(2*A)*(cotan_alpha+cotan_beta)
        
                tot += cotan_alpha+cotan_beta
    
            M[i,i] = -tot/(2*A) # les valeurs pour la diagonale
    
    return M
    

        
        
    

def cercle(x,y,R,x0,y0):
    return (x-x0)**2+(y-y0)**2-R

def cercle_bord(R,x0,y0,dl):
    t = np.arange(0,2*np.pi,dl)[:,np.newaxis]
    return np.hstack((R*np.cos(t)+x0,R*np.sin(t)+y0))


def rect(x,y,L,H,x0=0,y0=0):
    if 0<x-x0<L and 0<y-y0<H:
        return -1
    else:
        return 1

def rect_bord(L,H,a,x0=0,y0=0):
    tab1 = np.arange(x0,L+x0,a)[:,np.newaxis]
    h = np.hstack((tab1,H*np.ones((len(tab1),1))+y0))
    b = np.hstack((tab1,np.zeros((len(tab1),1))+y0))
    tab2 = np.arange(y0+a,H+y0,a)[:,np.newaxis]
    g = np.hstack((np.zeros((len(tab2),1))+x0,tab2))
    d = np.hstack((L*np.ones((len(tab2),1))+x0,tab2))
    hp = np.array([[L+x0,H+y0]])
    bp = np.array([[L+x0,0]])
    return np.vstack((h,b,g,d,hp,bp))
    

def cardioïde(x,y,a):
    return (x**2+y**2-a*x)**2-a**2*(x**2+y**2)

def ellipse(x,y,a,b):
    return x**2/(a**2)+y**2/(b**2)-1